/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

/**
 * A form reference is used as property value to assign a component to a property.
 * Normally used for <code>JLabel.setLabelFor()</code>.
 * <p>
 * Example:
 * <pre>
 *   FormComponent nameLabel = new FormComponent("javax.swing.JLabel");
 *   nameLabel.setName("nameLabel");
 *   nameLabel.setProperty("text", "Name:");
 *   nameLabel.setPropertyInt("displayedMnemonic", 'N');
 *   nameLabel.setProperty("labelFor", <b>new FormReference("nameField")</b>);
 *
 *   FormComponent nameField = new FormComponent("javax.swing.JTextField");
 *   nameField.setName("nameField");
 * </pre>
 *
 * @author Karl Tauber
 */
public class FormReference
{
	private final String name;

	/**
	 * Constructs a form reference for the specified form component name.
	 */
	public FormReference( String name ) {
		this.name = name;
	}

	/**
	 * Returns the name of the referenced form component.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Compares the names of form references.
	 */
	@Override
	public boolean equals( Object obj ) {
		if( obj instanceof FormReference )
			return name.equals( ((FormReference)obj).getName() );
		return false;
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return name;
	}
}
