/*
 * Copyright (C) 2008-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.ArrayList;

/**
 * A form binding represents a group of bindings in the form model.
 * Usually a <code>org.jdesktop.beansbinding.BindingGroup</code>.
 *
 * @author Karl Tauber
 * @since 5.0
 */
public class FormBindingGroup
	extends FormObject
	implements FormSelectable
{
	public static final String PROP_NAME = "name";
	public static final String PROP_BOUND = "bound";

	private static final FormBinding[] EMPTY_BINDINGS = {};

	private final Class<?> bindingGroupClass;
	private final ArrayList<FormBinding> bindings = new ArrayList<FormBinding>();

	private FormRoot root;

	public FormBindingGroup( Class<?> bindingGroupClass ) {
		this.bindingGroupClass = bindingGroupClass;
	}

	FormBindingGroup( FormBindingGroup obj ) {
		super( obj, 0 );
		bindingGroupClass = obj.bindingGroupClass;

		// clone bindings
		int bindingCount = obj.getBindingCount();
		for( int i = 0; i < bindingCount; i++ )
			addBinding( new FormBinding( obj.getBinding( i ) ) );

		// do not copy root
	}

	/**
	 * Clones this form binding group.
	 */
	@Override
	public Object clone() {
		return new FormBindingGroup( this );
	}

	/**
	 * Returns the binding group class.
	 */
	public Class<?> getBindingGroupClass() {
		return bindingGroupClass;
	}

	/**
	 * Returns the name of this binding group.
	 */
	public String getName() {
		String name = getPropertyString( PROP_NAME );
		if( name == null )
			name = "bindingGroup"; // fallback
		return name;
	}

	/**
	 * Returns the number of form bindings in this form binding group.
	 */
	public int getBindingCount() {
		return bindings.size();
	}

	/**
	 * Returns the form binding at <code>index</code>.
	 */
	public FormBinding getBinding( int index ) {
		return bindings.get( index );
	}

	/**
	 * Returns all form bindings in this form binding group.
	 */
	public FormBinding[] getBindings() {
		return !bindings.isEmpty()
			? bindings.toArray( new FormBinding[bindings.size()] )
			: EMPTY_BINDINGS;
	}

	/**
	 * Returns the index of <code>binding</code> in this binding group;
	 * or -1 if the binding is not a child of this binding group.
	 */
	public int getBindingIndex( FormBinding binding ) {
		return bindings.indexOf( binding );
	}

	/**
	 * Adds a form binding to the end of this form binding group.
	 */
	public void addBinding( FormBinding binding ) {
		addBinding( binding, -1 );
	}

	/**
	 * Adds a form binding to this form binding group at the specified position.
	 *
	 * @param binding The form binding to be added.
	 * @param index The position in the binding list at which
	 * 		to insert the binding; or -1 to insert at the end
	 */
	public void addBinding( FormBinding binding, int index ) {
		if( index < 0 || index == bindings.size() ) {
			index = bindings.size();
			bindings.add( binding );
		} else
			bindings.add( index, binding );

		binding.setBindingGroup( this );

		FormModelEventProvider eventProvider = getEventProvider();
		if( eventProvider != null )
			eventProvider.fireBindingAdded( this, binding, index );
	}

	/**
	 * Removes the specified form binding from this form binding group.
	 */
	public void removeBinding( FormBinding binding ) {
		int index = bindings.indexOf( binding );
		if( index >= 0 )
			removeBinding( index );
	}

	/**
	 * Removes the form binding at the specified index from this form binding group.
	 */
	public void removeBinding( int index ) {
		FormBinding binding = bindings.remove( index );
		binding.setBindingGroup( null );

		FormModelEventProvider eventProvider = getEventProvider();
		if( eventProvider != null )
			eventProvider.fireBindingRemoved( this, binding, index );
	}

	/**
	 * Returns the form root of this form binding group.
	 */
	public FormRoot getRoot() {
		return root;
	}

	/**
	 * Sets the form root of this form binding group.
	 */
	void setRoot( FormRoot root ) {
		if( this.root != null && root != null )
			throw new IllegalStateException( "Already attached (current=\""
						+ this.root + "\", new=\"" + root + "\")." );

		this.root = root;
	}

	/**
	 * Accepts the given visitor.
	 * The visitor's {@link FormBindingVisitor#visit} is called
	 * with all form bindings of this form binding group.
	 *
	 * @param visitor The visitor.
	 * @return The result of {@link FormBindingVisitor#visit}.
	 */
	public boolean accept( FormBindingVisitor visitor ) {
		int count = bindings.size();
		for( int i = 0; i < count; i++ ) {
			if( !bindings.get(i).accept( visitor ) )
				return false;
		}
		return true;
	}

	FormModelEventProvider getEventProvider() {
		return (root != null && root.model != null && root.model.eventProvider != null)
			? root.model.eventProvider : null;
	}

	// necessary for FormObject
	@Override
	void firePropertyChanged( String name, int index, Object oldValue, Object newValue ) {
		FormModelEventProvider eventProvider = getEventProvider();
		if( eventProvider != null )
			eventProvider.fireBindingGroupPropertyChanged( this, name, index, oldValue, newValue );
	}

	@Override
	void updateReferences( String oldName, String newName ) {
		super.updateReferences( oldName, newName );

		int count = bindings.size();
		for( int i = 0; i < count; i++ )
			bindings.get(i).updateReferences( oldName, newName );
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return unqualifiedClassName( getClass() )
			+ "[bindingGroupClass=" + bindingGroupClass + "] "
			+ super.toString();
	}
}
