/******************************************************************************
 * Project JKobo
 * XKobo arcade style game
 *
 * Copyright (C) 1997 John "Jake" Wetzel All Rights Reserved
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 *****************************************************************************/

/***************************** FILE INFORMATION ******************************
 * NAME OF FILE 	  : Space.java
 * CONCEPT & ARCHITECTURE : Akira Higuchi
 * MAIN PROGRAMMER	  : Jake Wetzel
 * DATE CREATED 	  : 10-25-98
 * DESCRIPTION		  :
 * CHANGE LOG		  :
 *	  WHO		  WHEN			  WHY
 *	  ---		  ----			  ---
 *
 *****************************************************************************/

/******************************* IMPORT FILES ********************************/

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.ImageProducer;
import java.awt.image.MemoryImageSource;
import java.lang.String;
import java.util.Random;

import Enemies;
import Manage;
import Map;
import PlayWin;
import Radar;
import ScaleParms;
import Scene;
import Tiff;


class Space implements MapDefines, GlobalDefines, EnemyDefines, ErrorDefines
{

  static final int SPRITE_MAX = 1024;
  static final int TRANSPARENT_RED   = 255;
  static final int TRANSPARENT_GREEN = 0;
  static final int TRANSPARENT_BLUE  = 0;
  static final int sint[] =
      { 0, 12, 23, 30, 32, 30, 23, 12, 0, -12, -23, -30, -32, -30, -23, -12};
  static final int cost[] =
      { 32, 30, 23, 12, 0, -12, -23, -30, -32, -30, -23, -12, 0, 12, 23, 30};

/******************************** GLOBAL DATA ********************************/

  int scene_num;
  int level;
  int generate_count;
  Map map;
  int scene_max;
  Color pixel_f0;
  Color pixel_f1;
  Color pixel_f2;
  Color pixel_s[];
  Image space_bitmap;	 /* bitmap containing all space stations on this level */
  Graphics space_gc;

/**************************** FUNCTION DECLARATIONS  *************************/

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : Space
 * DESCRIPTION	  :
 *****************************************************************************/
  Space()
  {
    int sl;
    ScaleParms sp;
    Canvas canvas;


    sl = JKobo.Context.getScaleLog2();
    sp = JKobo.Context.getScaleParms();
    map = new Map(sp.getRadarWidth() >>> sl, sp.getRadarHeight() >>> sl);
    pixel_s = new Color[8];
    canvas = JKobo.Context.getPlayWin().getCanvas();
    space_bitmap = canvas.createImage(sp.getSpaceWidth() + sp.getWindowWidth(),
				      sp.getSpaceHeight() + sp.getWindowHeight());
    space_gc = space_bitmap.getGraphics();

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : title
 * DESCRIPTION	  :
 *****************************************************************************/
  void title()
  {

    map.init();
    JKobo.Context.getRadar().prepare();

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : copyAChip
 * DESCRIPTION	  :
 *****************************************************************************/
  void copyAChip(int n, int posx, int posy)
  {
    int x, y, wsize;
    int dx, dy, sx, sy, h, w;

    x = n;
    y = 7;
    if((n & HARD) != 0)
      {
	y = 6;
	if((n & U_MASK) != 0)
	  x = 8;
	else if((n & R_MASK) != 0)
	  x = 9;
	else if((n & D_MASK) != 0)
	  x = 10;
	else if((n & L_MASK) != 0)
	  x = 11;
      }
    else if((n & CORE) != 0)
      {
	y = 6;
	x = 15;
      }
    w = JKobo.Context.getScaleParms().getSpriteWidth();
    h = JKobo.Context.getScaleParms().getSpriteHeight();
    dx = posx * w;
    dy = posy * h;
    sx = x * w;
    sy = y * h;
    copySprite(sx, sy, h, w, dx, dy);

    /* make sure overlap areas are accounted for */
    wsize = JKobo.Context.getScaleParms().getWindowWidth();
    x = JKobo.Context.getScaleParms().getSpaceWidth();
    y = JKobo.Context.getScaleParms().getSpaceHeight();
    if(dx < wsize)
      {
	copySprite(sx, sy, h, w, dx + x, dy);
	if(dy < wsize)
	  {
	    copySprite(sx, sy, h, w, dx, dy + y);
	    copySprite(sx, sy, h, w, dx + x, dy + y);
	  }
      }
    else if(dy < wsize)
      {
	copySprite(sx, sy, h, w, dx, dy + y);
      }

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : init
 * DESCRIPTION	  :
 *****************************************************************************/
  void init()
  {
    int i;

    scene_max = 0;
    while(JKobo.Context.getScenes()[scene_max].ratio != -1)
      scene_max++;

    if(pixel_f0 == null)
      {
	pixel_f0 = new Color(200, 255, 255);
	pixel_f1 = new Color(140, 200, 200);
	pixel_f2 = new Color(100, 200, 100);
	for(i=0; i<8; i++)
	  {
	    pixel_s[i] = new Color(156, 59+(i*16), 195-(i*20));
	  }
      }

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : initScene
 * DESCRIPTION	  :
 *****************************************************************************/
  void initScene(int sc)
  {
    int i;
    Scene s;

    if(sc < 0)
      {
	scene_num = -1;
	title();
	return;
      }
    scene_num = sc % scene_max;
    level     = sc / scene_max;
    s = JKobo.Context.getScenes()[scene_num];

    map.init();
    for(i=0; i<s.base_max; i++)
      map.makeMaze(s.base[i].x, s.base[i].y, s.base[i].h, s.base[i].v);
    map.convert(s.ratio);

    generate_count = 0;

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : prepare
 * DESCRIPTION	  :
 *****************************************************************************/
  int prepare()
  {
    int i, j, m, x, y;
    int count_core, c, sl;
    int w, h, wsize, scrx, scry;
    int interval_1, interval_2;
    int spr_shift;
    Scene s;
    Graphics g;
    Random rand;
    Enemies enemies;
    ScaleParms sp;


    if(scene_num < 0)
      return(0);

    count_core = c = 0;
    s = JKobo.Context.getScenes()[scene_num];
    sp = JKobo.Context.getScaleParms();
    enemies = JKobo.Context.getEnemies();
    interval_1 = (s.ek1_interval) >>> level;
    interval_2 = (s.ek2_interval) >>> level;
    if(interval_1 < 4)
      interval_1 = 4;
    if(interval_2 < 4)
      interval_2 = 4;
    enemies.setEnemyKindToGenerate(s.ek1, interval_1, s.ek2, interval_2);

    g = space_gc;
    g.setColor(Color.black);
    g.fillRect(0, 0, space_bitmap.getWidth(null), space_bitmap.getHeight(null));
    rand = JKobo.Context.getRandom();
    sl = JKobo.Context.getScaleLog2();
    w = sp.getSpriteWidth();
    h = sp.getSpriteHeight();
    scrx = sp.getRadarWidth() >>> sl;
    scry = sp.getRadarHeight() >>> sl;

    for(i=0; i<scrx; i++)
      {
	for(j=0; j<scry; j++)
	  {
	    m = map.getPos(i, j);
	    if(m != 0)
	      copyAChip(m, i, j);
	    if((m == U_MASK) || (m == R_MASK) || (m == D_MASK) || (m == L_MASK))
	      {
		enemies.make(CANNON, (i * w) + (w >> 1), (j * h) + (h >> 1),
			     0, 0, 0);
		c++;
	      }
	    else if(m == CORE)
	      {
		enemies.make(ECORE, (i * w) + (w >> 1), (j * h) + (h >> 1),
			     0, 0, 0);
		count_core++;
		c++;
	      }
	  }
      }

    /* fill in stars on background */
    scrx = sp.getSpaceWidth();
    scry = sp.getSpaceHeight();
    wsize = sp.getWindowWidth();
    spr_shift = sp.getSpriteWidthShift();
    for(i=0; i<8; i++)
      {
	g.setColor(pixel_s[i]);
	for(j=0; j<2000; j++)
	  {
	    x = (rand.nextInt() & 0x7fffffff) % scrx;
	    y = (rand.nextInt() & 0x7fffffff) % scry;
	    if(map.getPos(x >> spr_shift, y >> spr_shift) == SPACE)
	      g.fillOval(x << sl, y << sl, 1 << sl, 1 << sl);
	  }
      }

    /* copy overlap regions */
    g.copyArea(0, 0, wsize, scry, scrx, 0);
    g.copyArea(0, 0, scrx, wsize, 0, scry);
    g.copyArea(0, 0, wsize, wsize, scrx, scry);

    JKobo.Context.getMyShip().setPosition(s.startx * w, s.starty * h);
    JKobo.Context.getRadar().prepare();

    return(count_core);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : generateFixedEnemies
 * DESCRIPTION	  :
 *****************************************************************************/
  void generateFixedEnemies()
  {
    int j, sp;
    int x, y, h, v, t;
    int wsize, scrx, scry, shift;
    Scene s;
    Random rand;
    MyShip ship;
    Enemies enemies;


    s = JKobo.Context.getScenes()[scene_num];
    rand = JKobo.Context.getRandom();
    ship = JKobo.Context.getMyShip();
    enemies = JKobo.Context.getEnemies();
    wsize = JKobo.Context.getScaleParms().getWindowWidth();
    scrx = JKobo.Context.getScaleParms().getSpaceWidth();
    scry = JKobo.Context.getScaleParms().getSpaceHeight();
    shift = JKobo.Context.getScaleParms().getEnemyShift();
    if(generate_count < s.enemy_max)
      {
	for(j=0; j<s.enemy[generate_count].num; j++)
	  {
	    sp = s.enemy[generate_count].speed;
	    x = (rand.nextInt() & 0x7fffffff) % (scrx - (wsize << 1));
	    y = (rand.nextInt() & 0x7fffffff) % (scry - (wsize << 1));
	    x -= ((scrx >>> 1) - wsize);
	    y -= ((scry >>> 1) - wsize);
	    if(x < 0)
	      x -= wsize;
	    else
	      x += wsize;
	    if(y < 0)
	      y -= wsize;
	    else
	      y += wsize;
	    x += ship.getX();
	    y += ship.getY();

	    t = (rand.nextInt() & 0x7fffffff) % sint.length;
	    h = (sp * sint[t]) << (shift - SHIFT);
	    v = (sp * cost[t]) << (shift - SHIFT);
	    enemies.make(s.enemy[generate_count].kind, x, y, h, v, 0);
	}
	generate_count++;
      }

    if(generate_count >= s.enemy_max)
      generate_count = 0;

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : copySprite
 * DESCRIPTION	  :
 *****************************************************************************/
  boolean copySprite(int sx, int sy, int w, int h, int dx, int dy)
  {

    return(space_gc.drawImage(JKobo.Context.getEnemies().getEnemyBitmap(),
			      dx, dy, dx + w, dy + h, sx, sy, sx + w, sy + h,
			      null));

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getChipNumber
 * DESCRIPTION	  :
 *****************************************************************************/
  final int getChipNumber(int x, int y)
  {
    return(map.getPos(x, y));
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : setChipNumber
 * DESCRIPTION	  :
 *****************************************************************************/
  void setChipNumber(int x, int y, int n)
  {

    map.setPos(x, y, n);
    copyAChip(n, x, y);

    if(n == 0)
      JKobo.Context.getRadar().erase(x, y);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getPixels
 * DESCRIPTION	  :
 *****************************************************************************/
  Color[] getPixels()
  {
    int i;
    Color c[];


    c = new Color[3 + pixel_s.length];
    c[0] = pixel_f0;
    c[1] = pixel_f1;
    c[2] = pixel_f2;
    for(i=0; i<pixel_s.length; i++)
      c[i+3] = pixel_s[i];

    return(c);

  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getSpaceBitmap
 * DESCRIPTION	  :
 *****************************************************************************/
  final Image getSpaceBitmap()
  {
    return(space_bitmap);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : getSpaceGraphics
 * DESCRIPTION	  :
 *****************************************************************************/
  final Graphics getSpaceGraphics()
  {
    return(space_gc);
  }

/***************************** MODULE INFORMATION ****************************
 * NAME OF MODULE : close
 * DESCRIPTION	  :
 *****************************************************************************/
  void close()
  {
    space_gc.dispose();
  }

}  /* END OF CLASS : Space */


/* END OF FILE : Space.java */

