/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.SoundMap.Sound.AFTERBURNER;
import static javoids.SoundMap.Sound.GUN1;
import static javoids.SoundMap.Sound.GUN2;
import static javoids.SoundMap.Sound.GUN3;
import static javoids.SoundMap.Sound.JAVOIDDIE;
import static javoids.SoundMap.Sound.JAVOIDHIT;
import static javoids.SoundMap.Sound.MACHINEGUN1;
import static javoids.SoundMap.Sound.MACHINEGUN2;
import static javoids.SoundMap.Sound.MACHINEGUN3;
import static javoids.SoundMap.Sound.MINEDIE;
import static javoids.SoundMap.Sound.MINEHIT;
import static javoids.SoundMap.Sound.NO_SOUND;
import static javoids.SoundMap.Sound.PACMAN_DIE;
import static javoids.SoundMap.Sound.PACMAN_EATDOT;
import static javoids.SoundMap.Sound.PACMAN_EATFRUIT;
import static javoids.SoundMap.Sound.PACMAN_EATGHOST;
import static javoids.SoundMap.Sound.PACMAN_EXTRALIFE;
import static javoids.SoundMap.Sound.PACMAN_GAMESTART;
import static javoids.SoundMap.Sound.PACMAN_LEVELUP;
import static javoids.SoundMap.Sound.POWERUP;
import static javoids.SoundMap.Sound.POWERUPDIE;
import static javoids.SoundMap.Sound.ROCKET1;
import static javoids.SoundMap.Sound.SHIPDIE;
import static javoids.SoundMap.Sound.SHIPHIT;
import static javoids.SoundMap.Sound.SHIP_EXTRALIFE;
import static javoids.SoundMap.Sound.THRUST;

import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.util.HashMap;

import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Clip;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.swing.JOptionPane;

/* SoundMap------------------- */
/**
 * A class to provide mappings of sound names to sound clips and other functions to support their loading and use.
 * @author mallette
 */
public class SoundMap extends Thread implements Serializable
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;

  /**
   * A list of sound names for the game.
   * @author mallette
   */
  public enum Sound
  {
    /** a default to represent when no sound is selected */
    NO_SOUND,
    /** the sound of an engine thrust */
    THRUST,
    /** the sound of an engine afterburner */
    AFTERBURNER,
    /** the sound of a gun */
    GUN1,
    /** the sound of a gun */
    GUN2,
    /** the sound of a gun */
    GUN3,
    /** the sound of a gun */
    MACHINEGUN1,
    /** the sound of a gun */
    MACHINEGUN2,
    /** the sound of a gun */
    MACHINEGUN3,
    /** the sound of a missile */
    ROCKET1,
    /** the sound of a powerup */
    POWERUP,
    /** the sound of a powerup being destroyed */
    POWERUPDIE,
    /** the sound of a javoid being hit */
    JAVOIDHIT,
    /** the sound of a javoid being destroyed */
    JAVOIDDIE,
    /** the sound of a mine being hit */
    MINEHIT,
    /** the sound of a mine being destroyed */
    MINEDIE,
    /** the sound of a ship being hit */
    SHIPHIT,
    /** the sound of a ship being destroyed */
    SHIPDIE,
    /** the sound of a ship being given an extra life */
    SHIP_EXTRALIFE,
    /** the sound of a pacman being destroyed */
    PACMAN_DIE,
    /** the sound of a pacman eating a dot */
    PACMAN_EATDOT,
    /** the sound of a pacman eating a fruit */
    PACMAN_EATFRUIT,
    /** the sound of a pacman eating a ghost */
    PACMAN_EATGHOST,
    /** the sound of a pacman being awarded an extra life */
    PACMAN_EXTRALIFE,
    /** the sound of a level up when in pacman mode */
    PACMAN_LEVELUP,
    /** the sound of the pacman mode being started */
    PACMAN_GAMESTART
  }

  /** a map of sound names to their file names */
  private static HashMap<Sound,String> fileNames = new HashMap<Sound,String>();
  /** a map of sound names to the sound clips (the loaded sound files) */
  private static HashMap<Sound,Clip>   sounds = new HashMap<Sound,Clip>();

  static
  {
    SoundMap.fileNames.put(NO_SOUND,null);
    SoundMap.fileNames.put(THRUST,Messages.getString("SoundMap.Thrust")); //$NON-NLS-1$
    SoundMap.fileNames.put(AFTERBURNER,Messages.getString("SoundMap.AfterBurner")); //$NON-NLS-1$
    SoundMap.fileNames.put(GUN1,Messages.getString("SoundMap.Gun1")); //$NON-NLS-1$
    SoundMap.fileNames.put(GUN2,Messages.getString("SoundMap.Gun2")); //$NON-NLS-1$
    SoundMap.fileNames.put(GUN3,Messages.getString("SoundMap.Gun3")); //$NON-NLS-1$
    SoundMap.fileNames.put(MACHINEGUN1,Messages.getString("SoundMap.MachineGun1")); //$NON-NLS-1$
    SoundMap.fileNames.put(MACHINEGUN2,Messages.getString("SoundMap.MachineGun2")); //$NON-NLS-1$
    SoundMap.fileNames.put(MACHINEGUN3,Messages.getString("SoundMap.MachineGun3")); //$NON-NLS-1$
    SoundMap.fileNames.put(ROCKET1,Messages.getString("SoundMap.Rocket1")); //$NON-NLS-1$
    SoundMap.fileNames.put(POWERUP,Messages.getString("SoundMap.PowerUp")); //$NON-NLS-1$
    SoundMap.fileNames.put(POWERUPDIE,Messages.getString("SoundMap.PowerUpDie")); //$NON-NLS-1$
    SoundMap.fileNames.put(JAVOIDHIT,Messages.getString("SoundMap.JavoidHit")); //$NON-NLS-1$
    SoundMap.fileNames.put(JAVOIDDIE,Messages.getString("SoundMap.JavoidDie")); //$NON-NLS-1$
    SoundMap.fileNames.put(MINEHIT,Messages.getString("SoundMap.MineHit")); //$NON-NLS-1$
    SoundMap.fileNames.put(MINEDIE,Messages.getString("SoundMap.MineDie")); //$NON-NLS-1$
    SoundMap.fileNames.put(SHIPHIT,Messages.getString("SoundMap.ShipHit")); //$NON-NLS-1$
    SoundMap.fileNames.put(SHIPDIE,Messages.getString("SoundMap.ShipDie")); //$NON-NLS-1$
    SoundMap.fileNames.put(SHIP_EXTRALIFE,Messages.getString("SoundMap.ShipExtraLife")); //$NON-NLS-1$
    SoundMap.fileNames.put(PACMAN_DIE,Messages.getString("SoundMap.PacmanDie")); //$NON-NLS-1$
    SoundMap.fileNames.put(PACMAN_EATDOT,Messages.getString("SoundMap.PacmanEatDot")); //$NON-NLS-1$
    SoundMap.fileNames.put(PACMAN_EATFRUIT,Messages.getString("SoundMap.PacmanEatFruit")); //$NON-NLS-1$
    SoundMap.fileNames.put(PACMAN_EATGHOST,Messages.getString("SoundMap.PacmanEatGhost")); //$NON-NLS-1$
    SoundMap.fileNames.put(PACMAN_EXTRALIFE,Messages.getString("SoundMap.PacmanExtraLife")); //$NON-NLS-1$
    SoundMap.fileNames.put(PACMAN_LEVELUP,Messages.getString("SoundMap.PacmanEnd")); //$NON-NLS-1$
    SoundMap.fileNames.put(PACMAN_GAMESTART,Messages.getString("SoundMap.PacmanStart")); //$NON-NLS-1$
  }

  /**
   * Run a separate thread to load the sounds.
   */
  /*
   * (non-Javadoc)
   * @see java.lang.Thread#run()
   */
  @Override
  public void run()
  {
    URL url = null;
    System.out.printf(Messages.getString("SoundMap.GetSoundFiles")); //$NON-NLS-1$
    for (Sound sound : SoundMap.fileNames.keySet())
    {
      if (!NO_SOUND.equals( sound ))
      {
        String location = Messages.getString("SoundMap.SoundsDirectory") + this.getSoundName(sound); //$NON-NLS-1$
        try
        {
          url = this.getClass().getResource(location);
          if (Debug.debugSound)
            System.out.printf(Messages.getString("SoundMap.LoadingSounds"),url.toString()); //$NON-NLS-1$
          AudioInputStream audioInputStream = AudioSystem.getAudioInputStream(this.getClass().getResource(location));
          final Clip clip = (Clip)AudioSystem.getLine(new DataLine.Info(Clip.class,audioInputStream.getFormat()));
          clip.addLineListener(new LineListener()
          {
            public void update(LineEvent lineEvent)
            {
              if (LineEvent.Type.STOP.equals( lineEvent.getType() ))
              {
                clip.stop();
                clip.setMicrosecondPosition(0);
              }
            }
          });
          clip.open(audioInputStream);
          SoundMap.sounds.put(sound,clip);
        }
        catch (NullPointerException exception)
        {
          JOptionPane.showMessageDialog(null,Messages.getString("SoundMap.MissingFile") + location + Messages.getString("SoundMap.ErrorSoundWillNotPlay"),Messages.getString("SoundMap.ErrorMissingSoundFile"),JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        }
        catch (UnsupportedAudioFileException unsupportedAudioFileException)
        {
          JOptionPane.showMessageDialog(null,Messages.getString("SoundMap.FormatBad") + location + Messages.getString("SoundMap.ErrorSoundWillNotPlay"),Messages.getString("SoundMap.ErrorFormatBad"),JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        }
        catch (IOException iOException)
        {
          JOptionPane.showMessageDialog(null,Messages.getString("SoundMap.FileIO") + location + Messages.getString("SoundMap.ErrorSoundWillNotPlay"),Messages.getString("SoundMap.ErrorFileRead"),JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        }
        catch (LineUnavailableException lineUnavailableException)
        {
          JOptionPane.showMessageDialog(null,Messages.getString("SoundMap.ErrorLineUnavailable") + location + Messages.getString("SoundMap.ErrorSoundWillNotPlay"),Messages.getString("SoundMap.ErrorLineRead"),JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        }
      }
    }
    System.out.printf(Messages.getString("SoundMap.GotFiles")); //$NON-NLS-1$
  }

  /**
   * @return the number of sounds loaded
   */
  public int size()
  {
    return SoundMap.sounds.size();
  }

  /**
   * @param sound the sound name
   * @return the sound clip associated with the name
   */
  public Clip getSound(Sound sound)
  {
    return SoundMap.sounds.get(sound);
  }

  /**
   * @return the mapping of sound names to sound clips
   */
  public HashMap<Sound,Clip> getSounds()
  {
    return SoundMap.sounds;
  }

  /**
   * @param sound the sound name
   * @return the text representation of the sound name
   */
  public String getSoundName(Sound sound)
  {
    return SoundMap.fileNames.get(sound);
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("SoundMap.ToString"),SoundMap.fileNames,SoundMap.sounds); //$NON-NLS-1$
  }
}
/* SoundMap------------------- */
