/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.BasicSprite.Collision.NO_COLLISION;
import static javoids.Shapes.Shape.CIRCLE;
import static javoids.Shapes.Shape.POINT;

import java.awt.Color;

/* Singularity---------------- */
/**
 * A singularity (not quite, but close enough). This is used for white and black holes. White holes repel and sometimes create Javoids. Black holes attract and destroy anything that enters them.
 * @author mallette
 */
public final class Singularity extends GravityPoint
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long   serialVersionUID      = 1L;
  /** The maximum number of these sprites allowed in the game */
  public final static int     MAX_NUMBER            = 2;
  /** the default chance that a singularity will move */
  private final static double DEFAULT_MOVING_CHANCE = 0.5;
  /** the sprite's maximum size */
  protected final static int  MAX_SIZE              = 100;
  /** the sprite's minimum size */
  protected final static int  MIN_SIZE              = 1;
  /** the sprite's default size */
  protected final static int  DEFAULT_SIZE          = 20; // This should match the size of the image!
  /** the initial size of the singularity */
  private int                 initialSize;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   */
  public Singularity(BasicSprite parent,Health health,Move move,int _size)
  {
    super(parent,health,move,_size);
    this.initialSize = this.getSize();
    if (Math.random() > Singularity.DEFAULT_MOVING_CHANCE)
      this.getMove().setMaxVelocity(move.getMaxVelocity());
    this.accelerate(this.getMove().getMaxVelocity());
    this.setDisplayAreas(true);
  }

  /**
   * @return the default size of the sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.GravityPoint#getDefaultSize()
   */
  @Override
  public int getDefaultSize()
  {
    return Singularity.DEFAULT_SIZE;
  }

  /**
   * @return the maximum size of the sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.GravityPoint#getMaximumSize()
   */
  @Override
  public int getMaximumSize()
  {
    return Singularity.MAX_SIZE;
  }

  /**
   * @return the minimum size fo the sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.GravityPoint#getMinimumSize()
   */
  @Override
  public int getMinimumSize()
  {
    return Singularity.MIN_SIZE;
  }

  /**
   * @param _size the size to set for this sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.GravityPoint#setSize(int)
   */
  @Override
  public void setSize(int _size)
  {
    super.setSize(_size >= Singularity.MIN_SIZE ? _size <= Singularity.MAX_SIZE ? _size : Singularity.MAX_SIZE : Singularity.MIN_SIZE);
  }

  /**
   * @return the mass of the sprite (different than normal)
   */
  /*
   * (non-Javadoc)
   * @see javoids.GravityPoint#getMass()
   */
  @Override
  public double getMass()
  {
    return (double)this.getSize() / this.initialSize;
  }

  /**
   * @param gravity the gravity type to use for the singularity
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#setGravity(javoids.BasicSprite.Gravity)
   */
  @Override
  public void setGravity(Gravity gravity)
  {
    super.setGravity(gravity);
    this.setShape();
  }

  /**
   * Set the shape and colors of the singularity (based on the gravity type).
   */
  private void setShape()
  {
    if (this.isGravityWell())
    {
      this.getAreas().clear();
      this.getAreas().addAll(Shapes.getAreas(POINT,this.getSize(),Singularity.MAX_SIZE));
      this.getAreas().addAll(Shapes.getAreas(CIRCLE,this.getSize() + 2,Singularity.MAX_SIZE));
      this.getColors().clear();
      this.getColors().add(Color.black);
      this.getColors().add(Color.white);
    }
    else
    {
      this.getAreas().clear();
      this.getAreas().addAll(Shapes.getAreas(POINT,this.getSize(),Singularity.MAX_SIZE));
      this.getColors().clear();
      this.getColors().add(Color.white);
    }
  }

  /**
   * Cause a singularity to change its size.
   * @param amount the amount to make the singularity grow.
   */
  public void grow(int amount)
  {
    if (this.isGravityWell() && amount > 0 || this.isGravitySource() && amount < 0)
    {
      if (this.getSize() + amount < Singularity.MAX_SIZE)
      {
        super.setSize(this.getSize() + amount);
        this.setShape();
      }
    }
  }

  /**
   * Move the singularity if it moves at all.
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#move()
   */
  @Override
  public void move()
  {
    int _size = this.getSize();
    if (this.isAlive())
    {
      super.move();
      if (this.isGravitySource())
      {
        double value = (double)(this.getMaxDuration() - this.getDuration()) / this.getMaxDuration() * 100.0 / 10;
        
        if (value - (int)value <= 0.001)
        {
          this.grow(-(this.initialSize * 10 / 100));
          if (_size == this.getSize())
            this.kill();
        }
      }
    }
  }

  /**
   * @param sprite The sprite that is colliding into this sprite.
   * @param collisionType The type of interaction required for this collision (shield on shield etc).
   * @return the number of points obtained from the collision
   */
  public int collide(Singularity sprite,BasicSprite.Collision collisionType)
  {
    if (!this.equals( sprite ) && !NO_COLLISION.equals( collisionType ))
    {
      if (this.isGravityWell())
      {
        sprite.setDamage(sprite.getMaxDamage());
        this.grow(+10);
      }
      else if (sprite.isGravityWell())
      {
        this.setDamage(this.getMaxDamage());
        sprite.grow(+10);
      }
      else
      {
        this.setDamage(this.getMaxDamage());
        sprite.setDamage(sprite.getMaxDamage());
      }
    }
    return 0;
  }

  /**
   * @param sprite The sprite that is colliding into this sprite.
   * @param collisionType The type of interaction required for this collision (shield on shield etc).
   * @return the number of points obtained from the collision
   */
  @Override
  public int collide(BasicSprite sprite,BasicSprite.Collision collisionType)
  {
    if (!NO_COLLISION.equals( collisionType ))
    {
      super.collide(sprite,collisionType);
      this.grow(+1);
    }
    return 0;
  }

  /**
   * @param sprite The sprite that is colliding into this sprite.
   * @param collisionType The type of interaction required for this collision (shield on shield etc).
   * @return the number of points obtained from the collision
   */
  public int collide(Explosion sprite,BasicSprite.Collision collisionType)
  {
    if (!NO_COLLISION.equals( collisionType ))
    {
      super.collide(sprite,collisionType);
      this.grow(0);
    }
    return 0;
  }

  /**
   * @param sprite The sprite that is colliding into this sprite.
   * @param collisionType The type of interaction required for this collision (shield on shield etc).
   * @return the number of points obtained from the collision
   */
  public int collide(Shield sprite,BasicSprite.Collision collisionType)
  {
    if (!NO_COLLISION.equals( collisionType ))
    {
      super.collide(sprite,collisionType);
      this.grow(0);
    }
    return 0;
  }

  /**
   * @param sprite The sprite that is colliding into this sprite.
   * @param collisionType The type of interaction required for this collision (shield on shield etc).
   * @return the number of points obtained from the collision
   */
  public int collide(ShieldedSprite sprite,@SuppressWarnings("unused") BasicSprite.Collision collisionType)
  {
    this.collide(sprite.getShield(),this.collisionDetected(sprite.getShield()));
    if (sprite.getShield() != null && !sprite.getShield().isAlive())
      this.collide((BasicSprite)sprite,this.collisionDetected(sprite));
    return 0;
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Singularity.ToString"),super.toString(),Integer.valueOf(this.initialSize)); //$NON-NLS-1$
  }
}
/* Singularity---------------- */
