/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.BasicSprite.Collision.COLLISION;
import static javoids.BasicSprite.Collision.NO_COLLISION;
import static javoids.BasicSprite.Collision.SHIELD_SHIELD;
import static javoids.BasicSprite.Collision.SHIELD_SPRITE;
import static javoids.BasicSprite.Collision.SPRITE_SHIELD;
import static javoids.BasicSprite.Collision.SPRITE_SPRITE;
import static javoids.Health.DurationType.NORMAL;
import static javoids.Shapes.Shape.POINT;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Area;
import java.util.Vector;

/* ShieldedSprite------------- */
/**
 * A sprite with a shield (also a sprite).
 * @author mallette
 */
public abstract class ShieldedSprite extends Sprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;
  /** the ship's sheild */
  private Shield            shield;

  /**
   * Constructor
   * @param parent the parent sprite
   * @param health the health information
   * @param move the movement information
   * @param _shield the shield information
   * @param _size the size of the sprite
   * @param image the sprite's original image
   * @param areas the areas that represent the sprite
   * @param colors the colors that match the areas of the sprite
   */
  protected ShieldedSprite(BasicSprite parent,Health health,Move move,Shield _shield,int _size,ImageMap.Name image,Vector<Area> areas,Vector<Color> colors)
  {
    super(parent,health,move,_size,image,areas,colors);
    
    Shield temporaryShield = _shield;

    if (temporaryShield == null)
    {
      temporaryShield = new Shield(this,new Health(1,Shield.MAXIMUM_DAMAGE_CAPACITY,Shield.DEFAULT_DURATION,Shield.DEFAULT_AGING_RATE,NORMAL),new Move(move),_size,POINT);
      temporaryShield.kill();
      temporaryShield.expire();
    }
    this.setShield(temporaryShield);
  }

  /**
   * @return the shield of this sprite
   */
  public Shield getShield()
  {
    return this.shield;
  }

  /**
   * @param _shield the shield to give this sprite
   */
  public void setShield(Shield _shield)
  {
    this.shield = _shield;
  }

  /**
   * @param move the movement information to set
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#setMove(javoids.Move)
   */
  @Override
  public void setMove(Move move)
  {
    super.setMove(move);
    this.shield.setMove(new Move(move));
  }

  /**
   * @param direction the direction to set
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#setDirection(double)
   */
  @Override
  public void setDirection(double direction)
  {
    super.setDirection(direction);
    this.shield.setDirection(direction);
  }

  /**
   * @param amount the amount to rotate every time the direction is changed
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#setRotation(double)
   */
  @Override
  public void setRotation(double amount)
  {
    super.setRotation(amount);
    this.shield.setRotation(amount);
  }

  /**
   * Detect for collisions between two sprites.
   * @param shieldedSprite the other sprite
   * @return the status of the collision (NO_COLLISION | SHIELD_SHIELD | SHIELD_SPRITE | SPRITE_SHIELD | SPRITE_SPRITE)
   */
  public BasicSprite.Collision collisionDetected(ShieldedSprite shieldedSprite)
  {
    BasicSprite.Collision status = NO_COLLISION;
    if (this.isAlive() && shieldedSprite.isAlive())
    {
      if (COLLISION.equals( this.getShield().collisionDetected(shieldedSprite.getShield()) ))
        status = SHIELD_SHIELD;
      if (NO_COLLISION.equals( status ) && COLLISION.equals( this.getShield().collisionDetected(shieldedSprite) ))
        status = SHIELD_SPRITE;
      if (NO_COLLISION.equals( status ) && COLLISION.equals( super.collisionDetected(shieldedSprite.getShield()) ))
        status = SPRITE_SHIELD;
      if (NO_COLLISION.equals( status )&& COLLISION.equals( super.collisionDetected(shieldedSprite) ))
        status = SPRITE_SPRITE;
    }
    return status;
  }

  /**
   * @param sprite check for a collision with this sprite
   * @param collisionType the collision type
   * @return the number of points to award for the collision
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#collide(javoids.BasicSprite, javoids.BasicSprite.Collision)
   */
  @Override
  public int collide(BasicSprite sprite,BasicSprite.Collision collisionType)
  {
    int points = 0;
    if (!NO_COLLISION.equals( collisionType ))
    {
      double relativeVelocity = this.getRelativeVelocity(sprite);
      int damage1 = (int)(relativeVelocity * sprite.getMass() + 1);
      int damage2 = (int)(relativeVelocity * this.getMass() + 1);

      points = damage1;
      switch (collisionType)
      {
        case NO_COLLISION :
          System.out.printf(Messages.getString("ShieldedSprite.ShouldNotOccur")); //$NON-NLS-1$
          break;
        case SHIELD_SHIELD : // by this point both shields are guaranteed to be alive!
          this.getShield().modifyDamage(damage1);
          ((ShieldedSprite)sprite).getShield().modifyDamage(damage2);
          break;
        case SHIELD_SPRITE : // by this point this shield is guaranteed to be alive!
          this.getShield().modifyDamage(damage1);
          sprite.modifyDamage(damage2);
          break;
        case SPRITE_SHIELD : // by this point the other shield is guaranteed to be alive!
          this.modifyDamage(damage1);
          ((ShieldedSprite)sprite).getShield().modifyDamage(damage2);
          break;
        default : // covers COLLISION (non-shielded sprites) and sprites with no shields
          this.modifyDamage(damage1);
          sprite.modifyDamage(damage2);
          break;
      }
    }
    return points;
  }

  /**
   * Draw the shield and sprite image.
   * @param g2d the graphics context to use for drawing.
   * @param foregroundImage the image to draw on for double buffering
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#draw(java.awt.Graphics2D, java.awt.Graphics2D)
   */
  @Override
  public void draw(Graphics2D g2d,Graphics2D foregroundImage)
  {
    super.draw(g2d,foregroundImage);
    if (this.isAlive())
      this.shield.draw(g2d,foregroundImage);
  }

  /**
   * @return the mass of the sprite
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#getMass()
   */
  @Override
  public double getMass()
  {
    return this.shield.isAlive() ? this.shield.getMass() : super.getMass();
  }

  /**
   * @param acceleration the amount to accelerate by at one time
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#accelerate(double)
   */
  @Override
  public void accelerate(double acceleration)
  {
    super.accelerate(acceleration);
    this.shield.accelerate(acceleration);
    // fix the location of the shield if shield is rotating.
    this.getShield().setX(this.getX());
    this.getShield().setY(this.getY());
  }

  /**
   * @param turn_direction the direction to turn
   * @param amount the amount to turn by
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#turn(double, double)
   */
  @Override
  public void turn(double turn_direction,double amount)
  {
    super.turn(turn_direction,amount);
    if (this.getShield().getRotation() == 0)
      this.getShield().turn(turn_direction,amount);
    // fix the location of the shield if shield is rotating.
    this.getShield().setX(this.getX());
    this.getShield().setY(this.getY());
  }

  /**
   * Move the sprite according to its velocity. Move the shield to the new location.
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#move()
   */
  @Override
  public void move()
  {
    super.move();
    this.shield.move();
    // fix the location of the shield if shield is rotating.
    this.getShield().setX(this.getX());
    this.getShield().setY(this.getY());
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("ShieldedSprite.ToString"),super.toString(),this.shield); //$NON-NLS-1$
  }
}
/* ShieldedSprite------------- */
