/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.io.Serializable;

/* Move----------------------- */
/**
 * A class to track the movement and speed of an game sprite.
 * @author mallette
 */
public class Move implements Serializable
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long        serialVersionUID   = 1L;
  /** maximum velocity for any object in the game */
  protected final static int       MAX_VELOCITY       = 20;
  /** don't allow the moving object to turn */
  protected final static int       NOTURN             = 0;
  /** go left */
  protected final static int       LEFT               = 1;
  /** go right */
  protected final static int       RIGHT              = 2;
  /** go up */
  protected final static int       UP                 = 3;
  /** go down */
  protected final static int       DOWN               = 4;
  /** the default acceleration rate */
  protected static double          DEFAULT_ACCELERATE = 1;
  /** the screen dimensions and location */
  private Rectangle                screen;
  /** the location of the object */
  private transient Point2D.Double location;
  /** the velocity of the object */
  private transient Point2D.Double velocity;
  /** the maximum velocity for this object */
  private double                   maxVelocity;
  /** the direction the object is moving */
  private double                   direction;              // up is 0, clock-wise is positive, counter-clock-wise is negative

  /**
   * Constructor (copy)
   * @param move the movement information to copy
   */
  public Move(Move move)
  {
    this(move.screen,move.location.x,move.location.y,move.direction,move.maxVelocity);
    this.velocity.x = move.velocity.x;
    this.velocity.y = move.velocity.y;
  }

  /**
   * Constructor
   * @param _screen the game screen location and dimensions
   * @param _x the x coordinate of the player
   * @param _y the y coordinate of the player
   * @param _direction the direction of the player (0..360 where 0 and 360 are straight up)
   * @param _maxVelocity the maximum velocity the sprite can move
   */
  public Move(Rectangle _screen,double _x,double _y,double _direction,double _maxVelocity)
  {
    this.location = new Point2D.Double(0,0); // create a dummy start location
    this.velocity = new Point2D.Double(0,0); // always create the object stopped since some Sprites don't move or don't have to
    this.setScreen(_screen);
    this.setMaxVelocity(_maxVelocity);
    this.setDirection(_direction);
    this.setX(_x);
    this.setY(_y);
  }

  /**
   * @return the game screen information
   */
  public Rectangle getScreen()
  {
    return this.screen;
  }

  /**
   * @return the x coordinate
   */
  public double getX()
  {
    return this.location.x;
  }

  /**
   * @return the y coordinate
   */
  public double getY()
  {
    return this.location.y;
  }

  /**
   * @return the portion of the velocity along the x axis
   */
  public Point2D.Double getVelocity()
  {
    return this.velocity;
  }

  /**
   * @return the maximum velocity
   */
  public double getMaxVelocity()
  {
    return this.maxVelocity;
  }

  /**
   * @return the direction the sprite is facing
   */
  public double getDirection()
  {
    return this.direction;
  }

  /**
   * @param _screen the location and dimensions of the game screen
   */
  public void setScreen(Rectangle _screen)
  {
    this.screen = _screen;
  }

  /**
   * @param _x the x coordinate
   */
  public void setX(double _x)
  {
    this.location.x = _x;
    if (this.location.x < this.screen.x)
    {
      this.location.x = this.screen.x - this.location.x; // get distance from screen's side
      this.location.x %= this.screen.width; // get to within one screen width
      this.location.x = this.screen.x + this.screen.width - this.location.x; // make the value wrap from the right side (a distance it was away from the left side)
    }
    else if (_x >= this.screen.x + this.screen.width)
    {
      this.location.x -= this.screen.x; // make the value 0 based
      this.location.x %= this.screen.width; // make the value within the range of 0..width
      this.location.x += this.screen.x; // make the value range between x..(x+width)
    }
  }

  /**
   * @param _y the y coordinate
   */
  public void setY(double _y)
  {
    this.location.y = _y;
    if (this.location.y < this.screen.y)
    {
      this.location.y = this.screen.y - this.location.y; // get distance from screen's top
      this.location.y %= this.screen.height; // get to within one screen height
      this.location.y = this.screen.y + this.screen.height - this.location.y; // make the value wrap from the bottom side (a distance it was away from the top side)
    }
    else if (_y >= this.screen.y + this.screen.height)
    {
      this.location.y -= this.screen.y; // make the value 0 based
      this.location.y %= this.screen.height; // make the value within the range of 0..height
      this.location.y += this.screen.y; // make the value range between y..(y+height)
    }
  }

  /**
   * @param _direction the direction the sprite is facing
   */
  public void setDirection(double _direction)
  {
    this.direction = _direction % 360;
  }

  /**
   * @param _maxVelocity the maximum velocity to use
   */
  public void setMaxVelocity(double _maxVelocity)
  {
    this.maxVelocity = _maxVelocity < 0 ? 0 : _maxVelocity;
  }

  /**
   * @param move replace the current movement information with the new information
   */
  public void setMove(Move move)
  {
    this.setScreen(move.screen);
    this.setMaxVelocity(move.maxVelocity);
    this.setDirection(move.direction);
    this.setX(move.location.x);
    this.setY(move.location.y);
    this.velocity = move.velocity;
  }

  /**
   * @param acceleration the amount to increase the sprite's velocity by
   */
  public void accelerate(double acceleration)
  {
    double angle = (this.direction - 90) * Common.toRadians;
    double speed;

    if (acceleration > 0)
    {
      this.velocity.x += acceleration * Math.cos(angle);
      this.velocity.x = this.velocity.x >= this.maxVelocity ? this.maxVelocity : this.velocity.x;
      this.velocity.x = this.velocity.x < -this.maxVelocity ? -this.maxVelocity : this.velocity.x;
      this.velocity.x = Math.abs(this.velocity.x) < 0.000001 ? 0 : this.velocity.x;
      this.velocity.y += acceleration * Math.sin(angle);
      this.velocity.y = this.velocity.y >= this.maxVelocity ? this.maxVelocity : this.velocity.y;
      this.velocity.y = this.velocity.y < -this.maxVelocity ? -this.maxVelocity : this.velocity.y;
      this.velocity.y = Math.abs(this.velocity.y) < 0.000001 ? 0 : this.velocity.y;
      speed = Math.sqrt(Math.pow(this.velocity.x,2) + Math.pow(this.velocity.y,2));
      if (speed > this.maxVelocity)
      {
        double ratio =  speed / this.maxVelocity;
        
        this.velocity.x /= ratio;
        this.velocity.y /= ratio;
      }
    }
    else // stop!
    {
      this.velocity.x = 0;
      this.velocity.y = 0;
    }
  }

  /**
   * @param turn_direction what direction the sprite should turn
   * @param n the amount to turn by
   */
  public void turn(double turn_direction,double n)
  {
    switch ((int)turn_direction)
    {
      case Move.LEFT :
        this.direction = (this.direction - n + 360) % 360;
        break;
      case Move.RIGHT :
        this.direction = (this.direction + n + 360) % 360;
        break;
      default :
        break;
    }
  }

  /**
   * move the sprite along the x and y coordinates based on the sprite's velocities
   */
  public void move()
  {
    this.setX(this.location.x + this.velocity.x);
    this.setY(this.location.y + this.velocity.y);
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("Move.ToString"),this.location,this.velocity,Double.valueOf(this.maxVelocity),Double.valueOf(this.direction)); //$NON-NLS-1$
  }
}
/* Move----------------------- */
