/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.KeyMap.Action.ACTION_AFTERBURNER;
import static javoids.KeyMap.Action.ACTION_BOMB1;
import static javoids.KeyMap.Action.ACTION_BOMB2;
import static javoids.KeyMap.Action.ACTION_CYCLE_LEFT;
import static javoids.KeyMap.Action.ACTION_CYCLE_RIGHT;
import static javoids.KeyMap.Action.ACTION_ESCAPE;
import static javoids.KeyMap.Action.ACTION_FIRE;
import static javoids.KeyMap.Action.ACTION_FLIP;
import static javoids.KeyMap.Action.ACTION_JUMP;
import static javoids.KeyMap.Action.ACTION_MENU;
import static javoids.KeyMap.Action.ACTION_NEWGAME;
import static javoids.KeyMap.Action.ACTION_PACMAN;
import static javoids.KeyMap.Action.ACTION_PAUSE;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_01;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_02;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_03;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_04;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_05;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_06;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_07;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_08;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_09;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_10;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_11;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_12;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_13;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_14;
import static javoids.KeyMap.Action.ACTION_SELECT_WEAPON_BEST;
import static javoids.KeyMap.Action.ACTION_SHIELD;
import static javoids.KeyMap.Action.ACTION_THRUST;
import static javoids.KeyMap.Action.ACTION_TURN_LEFT;
import static javoids.KeyMap.Action.ACTION_TURN_RIGHT;
import static javoids.KeyMap.Action.ACTION_UNKNOWN;

import java.awt.event.KeyEvent;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.net.URL;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

/**
 * A class to handle keyboard mappings of key presses to actions.
 * @author mallette
 */
/* KeyMap--------------------- */
public final class KeyMap implements Serializable
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;

  /**
   * A list of possible actions that can be mapped to key presses in the game.
   * @author mallette
   */
  public enum Action
  {
    /** a placeholder action for when no action is selected */
    ACTION_UNKNOWN,
    /** use thrust */
    ACTION_THRUST,
    /** turn left */
    ACTION_TURN_LEFT,
    /** turn right */
    ACTION_TURN_RIGHT,
    /** cycle through weapons going left */
    ACTION_CYCLE_LEFT,
    /** cycle through weapons going right */
    ACTION_CYCLE_RIGHT,
    /** fire the player's gun */
    ACTION_FIRE,
    /** flip the player to face the opposite direction */
    ACTION_FLIP,
    /** use the shield */
    ACTION_SHIELD,
    /** use the afterburner */
    ACTION_AFTERBURNER,
    /** use the jump */
    ACTION_JUMP,
    /** use a bomb */
    ACTION_BOMB1,
    /** use a bomb to destroy everything (except the player) */
    ACTION_BOMB2,
    /** select the best weapon */
    ACTION_SELECT_WEAPON_BEST,
    /** select the #01 weapon (always available) */
    ACTION_SELECT_WEAPON_01,
    /** select the #02 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_02,
    /** select the #03 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_03,
    /** select the #04 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_04,
    /** select the #05 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_05,
    /** select the #06 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_06,
    /** select the #07 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_07,
    /** select the #08 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_08,
    /** select the #09 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_09,
    /** select the #10 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_10,
    /** select the #11 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_11,
    /** select the #12 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_12,
    /** select the #13 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_13,
    /** select the #14 weapon (if available -- count > 0) */
    ACTION_SELECT_WEAPON_14,
    /** pause the game */
    ACTION_PAUSE,
    /** escape from the menu */
    ACTION_ESCAPE,
    /** bring up the menu */
    ACTION_MENU,
    /** start a new game */
    ACTION_NEWGAME,
    /** begin/end a pacman game of javoids */
    ACTION_PACMAN;
  }

  /** the maximum number of key presses combinations that can be stored */
  private static int                         MAX_KEYS       = 185;
  /** the valid key codes */
  private static HashSet<Integer>            keys           = new HashSet<Integer>(KeyMap.MAX_KEYS);
  /** the valid actions */
  private static HashSet<Action>             actions        = new HashSet<Action>(KeyMap.MAX_KEYS);
  /** a mapping of mnemonic to key code */
  private static Hashtable<String,Integer>   mnemonicKey    = new Hashtable<String,Integer>(KeyMap.MAX_KEYS);
  /** a mapping of mnemonic to an action */
  private static Hashtable<String,Action>    mnemonicAction = new Hashtable<String,Action>(KeyMap.MAX_KEYS);
  /** a mapping of key code to a mnemonic */
  protected static Hashtable<Integer,String> keyMnemonic    = new Hashtable<Integer,String>(KeyMap.MAX_KEYS);
  /** a mapping of an action to a mnemonic */
  protected static Hashtable<Action,String>  actionMnemonic = new Hashtable<Action,String>(KeyMap.MAX_KEYS);
  /** a set of pressed keys to process */
  private transient Set<Integer>             pressedKeys    = Collections.synchronizedSet(new HashSet<Integer>(KeyMap.MAX_KEYS));
  /** a set of actions to perform */
  private transient Set<Action>              pressedButtons = Collections.synchronizedSet(new HashSet<Action>(KeyMap.MAX_KEYS));
  /** a mapping of actions to key codes */
  protected Hashtable<Action,Integer>        actionKey      = new Hashtable<Action,Integer>(KeyMap.MAX_KEYS);
  /** a mapping of key codes to actions */
  protected Hashtable<Integer,Action>        keyAction      = new Hashtable<Integer,Action>(KeyMap.MAX_KEYS);

  /**
   * Constructor (default)
   */
  public KeyMap()
  {
    this.reset();
  }

  /**
   * Constructor (copy)
   * @param keyMap the keymap to copy
   */
  public KeyMap(KeyMap keyMap)
  {
    this.reset();
    this.pressedKeys = new HashSet<Integer>(keyMap.pressedKeys);
    this.pressedButtons = new HashSet<Action>(keyMap.pressedButtons);
    this.actionKey = new Hashtable<Action,Integer>(keyMap.actionKey);
    this.keyAction = new Hashtable<Integer,Action>(KeyMap.MAX_KEYS);
    this.copyValues(this,keyMap);
  }

  /**
   * Copy the source keymap to the destination keymap
   * @param destination the destination keymap
   * @param source the source keymap
   */
  public void copyValues(KeyMap destination,KeyMap source)
  {
    for (Integer key : source.pressedKeys)
      destination.pressedKeys.add(key);
    for (Action key : source.pressedButtons)
      destination.pressedButtons.add(key);
    for (Action action : source.actionKey.keySet())
      destination.actionKey.put(action,source.actionKey.get(action));
    for (Integer key : source.keyAction.keySet())
      destination.keyAction.put(key,source.keyAction.get(key));
  }

  /**
   * reset the keymap to its default values
   */
  public void reset()
  {
    if (Debug.debugKeyMap)
      System.out.printf(Messages.getString("KeyMap.ResetKeyMap")); //$NON-NLS-1$
    KeyMap.keys.clear();
    KeyMap.actions.clear();
    KeyMap.mnemonicKey.clear();
    KeyMap.keyMnemonic.clear();
    KeyMap.mnemonicAction.clear();
    KeyMap.actionMnemonic.clear();
    this.pressedKeys.clear();
    this.pressedButtons.clear();
    this.actionKey.clear();
    this.keyAction.clear();
    // add keys
    if (Debug.debugKeyMap)
      System.out.printf(Messages.getString("KeyMap.LoadKeyLiterals")); //$NON-NLS-1$
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_0));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_1));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_2));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_3));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_4));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_5));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_6));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_7));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_8));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_9));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_A));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ACCEPT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ADD));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_AGAIN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ALL_CANDIDATES));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ALPHANUMERIC));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ALT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ALT_GRAPH));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_AMPERSAND));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ASTERISK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_AT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_B));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_BACK_QUOTE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_BACK_SLASH));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_BACK_SPACE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_BRACELEFT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_BRACERIGHT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_C));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CANCEL));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CAPS_LOCK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CIRCUMFLEX));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CLEAR));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CLOSE_BRACKET));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CODE_INPUT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_COLON));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_COMMA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_COMPOSE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CONTEXT_MENU));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CONTROL));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CONVERT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_COPY));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_CUT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_D));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_ABOVEDOT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_ABOVERING));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_ACUTE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_BREVE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_CARON));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_CEDILLA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_CIRCUMFLEX));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_DIAERESIS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_DOUBLEACUTE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_GRAVE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_IOTA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_MACRON));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_OGONEK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_SEMIVOICED_SOUND));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_TILDE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DEAD_VOICED_SOUND));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DECIMAL));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DELETE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DIVIDE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DOLLAR));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_DOWN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_E));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_END));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ENTER));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_EQUALS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ESCAPE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_EURO_SIGN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_EXCLAMATION_MARK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F1));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F10));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F11));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F12));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F13));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F14));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F15));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F16));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F17));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F18));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F19));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F2));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F20));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F21));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F22));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F23));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F24));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F3));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F4));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F5));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F6));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F7));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F8));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_F9));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_FINAL));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_FIND));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_FULL_WIDTH));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_G));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_GREATER));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_H));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_HALF_WIDTH));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_HELP));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_HIRAGANA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_HOME));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_I));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_INPUT_METHOD_ON_OFF));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_INSERT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_INVERTED_EXCLAMATION_MARK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_J));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_JAPANESE_HIRAGANA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_JAPANESE_KATAKANA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_JAPANESE_ROMAN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_K));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KANA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KANA_LOCK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KANJI));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KATAKANA));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KP_DOWN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KP_LEFT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KP_RIGHT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_KP_UP));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_L));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_LEFT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_LEFT_PARENTHESIS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_LESS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_M));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_META));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_MINUS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_MODECHANGE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_MULTIPLY));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_N));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NONCONVERT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUM_LOCK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMBER_SIGN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD0));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD1));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD2));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD3));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD4));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD5));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD6));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD7));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD8));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_NUMPAD9));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_O));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_OPEN_BRACKET));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_P));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PAGE_DOWN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PAGE_UP));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PASTE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PAUSE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PERIOD));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PLUS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PREVIOUS_CANDIDATE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PRINTSCREEN));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_PROPS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_Q));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_QUOTE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_QUOTEDBL));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_R));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_RIGHT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_RIGHT_PARENTHESIS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_ROMAN_CHARACTERS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_S));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_SCROLL_LOCK));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_SEMICOLON));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_SEPARATER));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_SHIFT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_SLASH));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_SPACE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_STOP));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_SUBTRACT));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_T));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_TAB));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_U));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_UNDEFINED));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_UNDERSCORE));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_UNDO));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_UP));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_V));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_W));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_WINDOWS));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_X));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_Y));
    KeyMap.keys.add(Integer.valueOf(KeyEvent.VK_Z));
    // add actions
    if (Debug.debugKeyMap)
      System.out.printf(Messages.getString("KeyMap.LoadActionLiterals")); //$NON-NLS-1$
    KeyMap.actions.add(ACTION_FIRE);
    KeyMap.actions.add(ACTION_TURN_LEFT);
    KeyMap.actions.add(ACTION_TURN_RIGHT);
    KeyMap.actions.add(ACTION_AFTERBURNER);
    KeyMap.actions.add(ACTION_THRUST);
    KeyMap.actions.add(ACTION_FLIP);
    KeyMap.actions.add(ACTION_JUMP);
    KeyMap.actions.add(ACTION_SHIELD);
    KeyMap.actions.add(ACTION_CYCLE_LEFT);
    KeyMap.actions.add(ACTION_CYCLE_RIGHT);
    KeyMap.actions.add(ACTION_BOMB1);
    KeyMap.actions.add(ACTION_BOMB2);
    KeyMap.actions.add(ACTION_MENU);
    KeyMap.actions.add(ACTION_NEWGAME);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_BEST);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_01);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_02);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_03);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_04);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_05);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_06);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_07);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_08);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_09);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_10);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_11);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_12);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_13);
    KeyMap.actions.add(ACTION_SELECT_WEAPON_14);
    KeyMap.actions.add(ACTION_PAUSE);
    KeyMap.actions.add(ACTION_ESCAPE);
    KeyMap.actions.add(ACTION_PACMAN);
    // when I press this key, I get this action
    if (Debug.debugKeyMap)
      System.out.printf(Messages.getString("KeyMap.LoadKeyActions")); //$NON-NLS-1$
    this.putAction(Integer.valueOf(KeyEvent.VK_BACK_QUOTE),ACTION_SELECT_WEAPON_BEST);
    this.putAction(Integer.valueOf(KeyEvent.VK_1),ACTION_SELECT_WEAPON_01);
    this.putAction(Integer.valueOf(KeyEvent.VK_2),ACTION_SELECT_WEAPON_02);
    this.putAction(Integer.valueOf(KeyEvent.VK_3),ACTION_SELECT_WEAPON_03);
    this.putAction(Integer.valueOf(KeyEvent.VK_4),ACTION_SELECT_WEAPON_04);
    this.putAction(Integer.valueOf(KeyEvent.VK_5),ACTION_SELECT_WEAPON_05);
    this.putAction(Integer.valueOf(KeyEvent.VK_6),ACTION_SELECT_WEAPON_06);
    this.putAction(Integer.valueOf(KeyEvent.VK_7),ACTION_SELECT_WEAPON_07);
    this.putAction(Integer.valueOf(KeyEvent.VK_8),ACTION_SELECT_WEAPON_08);
    this.putAction(Integer.valueOf(KeyEvent.VK_9),ACTION_SELECT_WEAPON_09);
    this.putAction(Integer.valueOf(KeyEvent.VK_0),ACTION_SELECT_WEAPON_10);
    this.putAction(Integer.valueOf(KeyEvent.VK_A),ACTION_AFTERBURNER);
    this.putAction(Integer.valueOf(KeyEvent.VK_ACCEPT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ADD),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_AGAIN),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ALL_CANDIDATES),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ALPHANUMERIC),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ALT),ACTION_MENU);
    this.putAction(Integer.valueOf(KeyEvent.VK_ALT_GRAPH),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_AMPERSAND),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ASTERISK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_AT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_B),ACTION_BOMB1);
    this.putAction(Integer.valueOf(KeyEvent.VK_BACK_QUOTE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_BACK_SLASH),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_BACK_SPACE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_BRACELEFT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_BRACERIGHT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_C),ACTION_CYCLE_LEFT);
    this.putAction(Integer.valueOf(KeyEvent.VK_CANCEL),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CAPS_LOCK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CIRCUMFLEX),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CLEAR),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CLOSE_BRACKET),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CODE_INPUT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_COLON),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_COMMA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_COMPOSE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CONTROL),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CONTEXT_MENU),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CONVERT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_COPY),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_CUT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_D),ACTION_JUMP);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_ABOVEDOT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_ABOVERING),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_ACUTE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_BREVE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_CARON),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_CEDILLA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_CIRCUMFLEX),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_DIAERESIS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_DOUBLEACUTE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_GRAVE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_IOTA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_MACRON),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_OGONEK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_SEMIVOICED_SOUND),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_TILDE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DEAD_VOICED_SOUND),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DECIMAL),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DELETE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DIVIDE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DOLLAR),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_DOWN),ACTION_THRUST);
    this.putAction(Integer.valueOf(KeyEvent.VK_E),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_END),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ENTER),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_EQUALS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ESCAPE),ACTION_ESCAPE);
    this.putAction(Integer.valueOf(KeyEvent.VK_EURO_SIGN),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_EXCLAMATION_MARK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F1),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F10),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F11),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F12),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F13),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F14),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F15),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F16),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F17),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F18),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F19),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F2),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F20),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F21),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F22),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F23),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F24),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F3),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F4),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F5),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F6),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F7),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F8),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_F9),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_FINAL),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_FIND),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_FULL_WIDTH),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_G),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_GREATER),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_H),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_HALF_WIDTH),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_HELP),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_HIRAGANA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_HOME),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_I),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_INPUT_METHOD_ON_OFF),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_INSERT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_INVERTED_EXCLAMATION_MARK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_J),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_JAPANESE_HIRAGANA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_JAPANESE_KATAKANA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_JAPANESE_ROMAN),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_K),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KANA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KANA_LOCK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KANJI),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KATAKANA),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KP_DOWN),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KP_LEFT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KP_RIGHT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_KP_UP),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_L),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_LEFT),ACTION_FIRE);
    this.putAction(Integer.valueOf(KeyEvent.VK_LEFT_PARENTHESIS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_LESS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_M),ACTION_BOMB2);
    this.putAction(Integer.valueOf(KeyEvent.VK_META),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_MINUS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_MODECHANGE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_MULTIPLY),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_N),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NONCONVERT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUM_LOCK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMBER_SIGN),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD0),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD1),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD2),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD3),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD4),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD5),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD6),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD7),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD8),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_NUMPAD9),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_O),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_OPEN_BRACKET),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_P),ACTION_PAUSE);
    this.putAction(Integer.valueOf(KeyEvent.VK_PAGE_DOWN),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PAGE_UP),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PASTE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PAUSE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PERIOD),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PLUS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PREVIOUS_CANDIDATE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PRINTSCREEN),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_PROPS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_Q),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_QUOTE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_QUOTEDBL),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_R),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_RIGHT),ACTION_SHIELD);
    this.putAction(Integer.valueOf(KeyEvent.VK_RIGHT_PARENTHESIS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_ROMAN_CHARACTERS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_S),ACTION_FLIP);
    this.putAction(Integer.valueOf(KeyEvent.VK_SCROLL_LOCK),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_SEMICOLON),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_SEPARATER),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_SHIFT),ACTION_SHIELD);
    this.putAction(Integer.valueOf(KeyEvent.VK_SLASH),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_SPACE),ACTION_NEWGAME);
    this.putAction(Integer.valueOf(KeyEvent.VK_STOP),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_SUBTRACT),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_T),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_TAB),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_U),ACTION_SELECT_WEAPON_BEST);
    this.putAction(Integer.valueOf(KeyEvent.VK_UNDEFINED),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_UNDERSCORE),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_UNDO),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_UP),ACTION_AFTERBURNER);
    this.putAction(Integer.valueOf(KeyEvent.VK_V),ACTION_CYCLE_RIGHT);
    this.putAction(Integer.valueOf(KeyEvent.VK_W),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_WINDOWS),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_X),ACTION_TURN_RIGHT);
    this.putAction(Integer.valueOf(KeyEvent.VK_Y),ACTION_UNKNOWN);
    this.putAction(Integer.valueOf(KeyEvent.VK_Z),ACTION_TURN_LEFT);
    // add key mnemonics
    if (Debug.debugKeyMap)
      System.out.printf(Messages.getString("KeyMap.LoadMnemonics")); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.0"),KeyEvent.VK_0); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.1"),KeyEvent.VK_1); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.2"),KeyEvent.VK_2); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.3"),KeyEvent.VK_3); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.4"),KeyEvent.VK_4); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.5"),KeyEvent.VK_5); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.6"),KeyEvent.VK_6); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.7"),KeyEvent.VK_7); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.8"),KeyEvent.VK_8); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.9"),KeyEvent.VK_9); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.A"),KeyEvent.VK_A); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Accept"),KeyEvent.VK_ACCEPT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Add"),KeyEvent.VK_ADD); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Again"),KeyEvent.VK_AGAIN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.AllCandidates"),KeyEvent.VK_ALL_CANDIDATES); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.AlphaNumeric"),KeyEvent.VK_ALPHANUMERIC); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Alt"),KeyEvent.VK_ALT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.AltGraph"),KeyEvent.VK_ALT_GRAPH); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Ampersand"),KeyEvent.VK_AMPERSAND); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Asterisk"),KeyEvent.VK_ASTERISK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.At"),KeyEvent.VK_AT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.B"),KeyEvent.VK_B); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.BackQuoteText"),KeyEvent.VK_BACK_QUOTE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.BackQuote"),KeyEvent.VK_BACK_QUOTE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.BackSlashText"),KeyEvent.VK_BACK_SLASH); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DoubleBackSlash"),KeyEvent.VK_BACK_SLASH); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.BackSpace"),KeyEvent.VK_BACK_SPACE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.LeftBraceText"),KeyEvent.VK_BRACELEFT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.LeftBrace"),KeyEvent.VK_BRACELEFT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.RightBraceText"),KeyEvent.VK_BRACERIGHT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.RightBrace"),KeyEvent.VK_BRACERIGHT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Break"),KeyEvent.VK_PAUSE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.C"),KeyEvent.VK_C); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Cancel"),KeyEvent.VK_CANCEL); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.CapsLock"),KeyEvent.VK_CAPS_LOCK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Circumflex"),KeyEvent.VK_CIRCUMFLEX); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Clear"),KeyEvent.VK_CLEAR); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.CloseBracket"),KeyEvent.VK_CLOSE_BRACKET); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.CodeInput"),KeyEvent.VK_CODE_INPUT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Colon"),KeyEvent.VK_COLON); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.CommaText"),KeyEvent.VK_COMMA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Comma"),KeyEvent.VK_COMMA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Compose"),KeyEvent.VK_COMPOSE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Control"),KeyEvent.VK_CONTROL); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.ContextMenu"),KeyEvent.VK_CONTEXT_MENU); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Convert"),KeyEvent.VK_CONVERT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Copy"),KeyEvent.VK_COPY); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Cut"),KeyEvent.VK_CUT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.D"),KeyEvent.VK_D); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadAboveDot"),KeyEvent.VK_DEAD_ABOVEDOT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadAboveRing"),KeyEvent.VK_DEAD_ABOVERING); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadAcute"),KeyEvent.VK_DEAD_ACUTE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadBreve"),KeyEvent.VK_DEAD_BREVE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadCaron"),KeyEvent.VK_DEAD_CARON); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadCedilla"),KeyEvent.VK_DEAD_CEDILLA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadCircumflex"),KeyEvent.VK_DEAD_CIRCUMFLEX); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadDiaeresis"),KeyEvent.VK_DEAD_DIAERESIS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadDoubleAcute"),KeyEvent.VK_DEAD_DOUBLEACUTE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadGrave"),KeyEvent.VK_DEAD_GRAVE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadIota"),KeyEvent.VK_DEAD_IOTA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadMacron"),KeyEvent.VK_DEAD_MACRON); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadOgonek"),KeyEvent.VK_DEAD_OGONEK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadSemivoicedSound"),KeyEvent.VK_DEAD_SEMIVOICED_SOUND); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadTilde"),KeyEvent.VK_DEAD_TILDE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DeadVoicedSound"),KeyEvent.VK_DEAD_VOICED_SOUND); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Decimal"),KeyEvent.VK_DECIMAL); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Period"),KeyEvent.VK_DECIMAL); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Delete"),KeyEvent.VK_DELETE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DiviteText"),KeyEvent.VK_DIVIDE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Dollar"),KeyEvent.VK_DOLLAR); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Down"),KeyEvent.VK_DOWN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.E"),KeyEvent.VK_E); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.End"),KeyEvent.VK_END); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Enter"),KeyEvent.VK_ENTER); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.EqualsText"),KeyEvent.VK_EQUALS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Equals"),KeyEvent.VK_EQUALS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Escape"),KeyEvent.VK_ESCAPE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.EuroSign"),KeyEvent.VK_EURO_SIGN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.ExclamationMark"),KeyEvent.VK_EXCLAMATION_MARK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F"),KeyEvent.VK_F); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F1"),KeyEvent.VK_F1); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F10"),KeyEvent.VK_F10); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F11"),KeyEvent.VK_F11); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F12"),KeyEvent.VK_F12); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F13"),KeyEvent.VK_F13); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F14"),KeyEvent.VK_F14); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F15"),KeyEvent.VK_F15); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F16"),KeyEvent.VK_F16); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F17"),KeyEvent.VK_F17); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F18"),KeyEvent.VK_F18); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F19"),KeyEvent.VK_F19); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F2"),KeyEvent.VK_F2); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F20"),KeyEvent.VK_F20); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F21"),KeyEvent.VK_F21); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F22"),KeyEvent.VK_F22); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F23"),KeyEvent.VK_F23); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F24"),KeyEvent.VK_F24); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F3"),KeyEvent.VK_F3); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F4"),KeyEvent.VK_F4); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F5"),KeyEvent.VK_F5); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F6"),KeyEvent.VK_F6); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F7"),KeyEvent.VK_F7); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F8"),KeyEvent.VK_F8); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.F9"),KeyEvent.VK_F9); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Final"),KeyEvent.VK_FINAL); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Find"),KeyEvent.VK_FIND); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.FullWidth"),KeyEvent.VK_FULL_WIDTH); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.G"),KeyEvent.VK_G); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Greater"),KeyEvent.VK_GREATER); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.GreaterThan"),KeyEvent.VK_GREATER); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.H"),KeyEvent.VK_H); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.HalfWidth"),KeyEvent.VK_HALF_WIDTH); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Help"),KeyEvent.VK_HELP); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Hiragana"),KeyEvent.VK_HIRAGANA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Home"),KeyEvent.VK_HOME); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.I"),KeyEvent.VK_I); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.InputMethodToggle"),KeyEvent.VK_INPUT_METHOD_ON_OFF); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Insert"),KeyEvent.VK_INSERT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.InvertedExclamationMark"),KeyEvent.VK_INVERTED_EXCLAMATION_MARK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.J"),KeyEvent.VK_J); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.JapaneseHiragana"),KeyEvent.VK_JAPANESE_HIRAGANA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.JapaneseKatakana"),KeyEvent.VK_JAPANESE_KATAKANA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.JapaneseRoman"),KeyEvent.VK_JAPANESE_ROMAN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.K"),KeyEvent.VK_K); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Kana"),KeyEvent.VK_KANA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.KanaLock"),KeyEvent.VK_KANA_LOCK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Kanji"),KeyEvent.VK_KANJI); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Katakana"),KeyEvent.VK_KATAKANA); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.KeyPadDown"),KeyEvent.VK_KP_DOWN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.KeyPadLeft"),KeyEvent.VK_KP_LEFT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.KeyPadRight"),KeyEvent.VK_KP_RIGHT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.KeyPadUp"),KeyEvent.VK_KP_UP); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.L"),KeyEvent.VK_L); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Left"),KeyEvent.VK_LEFT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.LeftParenthesisText"),KeyEvent.VK_LEFT_PARENTHESIS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.LeftParenthesis"),KeyEvent.VK_LEFT_PARENTHESIS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Less"),KeyEvent.VK_LESS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.LessThan"),KeyEvent.VK_LESS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.M"),KeyEvent.VK_M); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Meta"),KeyEvent.VK_META); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.MinusText"),KeyEvent.VK_MINUS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Minus"),KeyEvent.VK_MINUS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.ModeChange"),KeyEvent.VK_MODECHANGE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.MultiplyText"),KeyEvent.VK_MULTIPLY); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Multiply"),KeyEvent.VK_MULTIPLY); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.N"),KeyEvent.VK_N); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Nonconvert"),KeyEvent.VK_NONCONVERT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumLock"),KeyEvent.VK_NUM_LOCK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumberSignText"),KeyEvent.VK_NUMBER_SIGN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad0"),KeyEvent.VK_NUMPAD0); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad1"),KeyEvent.VK_NUMPAD1); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad2"),KeyEvent.VK_NUMPAD2); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad3"),KeyEvent.VK_NUMPAD3); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad4"),KeyEvent.VK_NUMPAD4); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad5"),KeyEvent.VK_NUMPAD5); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad6"),KeyEvent.VK_NUMPAD6); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad7"),KeyEvent.VK_NUMPAD7); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad8"),KeyEvent.VK_NUMPAD8); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.NumPad9"),KeyEvent.VK_NUMPAD9); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.O"),KeyEvent.VK_O); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.OpenBracket"),KeyEvent.VK_OPEN_BRACKET); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.P"),KeyEvent.VK_P); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.PageDown"),KeyEvent.VK_PAGE_DOWN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.PageUp"),KeyEvent.VK_PAGE_UP); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Paste"),KeyEvent.VK_PASTE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Pause"),KeyEvent.VK_PAUSE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.PeriodText"),KeyEvent.VK_PERIOD); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Period"),KeyEvent.VK_PERIOD); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.PlusText"),KeyEvent.VK_PLUS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Plus"),KeyEvent.VK_PLUS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.PreviousCandidate"),KeyEvent.VK_PREVIOUS_CANDIDATE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.PrintScreen"),KeyEvent.VK_PRINTSCREEN); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Props"),KeyEvent.VK_PROPS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Q"),KeyEvent.VK_Q); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.QuoteText"),KeyEvent.VK_QUOTE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Quote"),KeyEvent.VK_QUOTE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DoubleQuoteText"),KeyEvent.VK_QUOTEDBL); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.DoubleQuote"),KeyEvent.VK_QUOTEDBL); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.R"),KeyEvent.VK_R); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Right"),KeyEvent.VK_RIGHT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.RightParenthesisText"),KeyEvent.VK_RIGHT_PARENTHESIS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.RightParenthesis"),KeyEvent.VK_RIGHT_PARENTHESIS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.RomanCharacters"),KeyEvent.VK_ROMAN_CHARACTERS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.S"),KeyEvent.VK_S); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.ScrollLock"),KeyEvent.VK_SCROLL_LOCK); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.SemiColonText"),KeyEvent.VK_SEMICOLON); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.SemiColon"),KeyEvent.VK_SEMICOLON); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Separater"),KeyEvent.VK_SEPARATER); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Shift"),KeyEvent.VK_SHIFT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.SlashText"),KeyEvent.VK_SLASH); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Slash"),KeyEvent.VK_SLASH); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Space"),KeyEvent.VK_SPACE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Stop"),KeyEvent.VK_STOP); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Subtract"),KeyEvent.VK_SUBTRACT); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.T"),KeyEvent.VK_T); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.TabText"),KeyEvent.VK_TAB); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.U"),KeyEvent.VK_U); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Undefined"),KeyEvent.VK_UNDEFINED); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.UnderscoreText"),KeyEvent.VK_UNDERSCORE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Underscore"),KeyEvent.VK_UNDERSCORE); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Undo"),KeyEvent.VK_UNDO); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Up"),KeyEvent.VK_UP); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.V"),KeyEvent.VK_V); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.W"),KeyEvent.VK_W); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Windows"),KeyEvent.VK_WINDOWS); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.X"),KeyEvent.VK_X); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Y"),KeyEvent.VK_Y); //$NON-NLS-1$
    KeyMap.putMnemonicKey(Messages.getString("KeyMap.Z"),KeyEvent.VK_Z); //$NON-NLS-1$
    // add action nemonics
    if (Debug.debugKeyMap)
      System.out.printf(Messages.getString("KeyMap.LoadMnemonic")); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Fire"),ACTION_FIRE); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.TurnLeft"),ACTION_TURN_LEFT); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.TurnRight"),ACTION_TURN_RIGHT); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.AfterBurner"),ACTION_AFTERBURNER); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Thrust"),ACTION_THRUST); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Flip"),ACTION_FLIP); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Jump"),ACTION_JUMP); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Shield"),ACTION_SHIELD); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.CycleLeft"),ACTION_CYCLE_LEFT); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.CycleRight"),ACTION_CYCLE_RIGHT); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Bomb"),ACTION_BOMB1); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Megabomb"),ACTION_BOMB2); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Menu"),ACTION_MENU); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.NewGame"),ACTION_NEWGAME); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeaponBest"),ACTION_SELECT_WEAPON_BEST); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon01"),ACTION_SELECT_WEAPON_01); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon02"),ACTION_SELECT_WEAPON_02); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon03"),ACTION_SELECT_WEAPON_03); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon04"),ACTION_SELECT_WEAPON_04); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon05"),ACTION_SELECT_WEAPON_05); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon06"),ACTION_SELECT_WEAPON_06); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon07"),ACTION_SELECT_WEAPON_07); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon08"),ACTION_SELECT_WEAPON_08); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon09"),ACTION_SELECT_WEAPON_09); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon10"),ACTION_SELECT_WEAPON_10); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon11"),ACTION_SELECT_WEAPON_11); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon12"),ACTION_SELECT_WEAPON_12); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon13"),ACTION_SELECT_WEAPON_13); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.SelectWeapon14"),ACTION_SELECT_WEAPON_14); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Pause"),ACTION_PAUSE); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Unpause"),ACTION_ESCAPE); //$NON-NLS-1$
    KeyMap.putMnemonicAction(Messages.getString("KeyMap.Pacman"),ACTION_PACMAN); //$NON-NLS-1$
  }

  /**
   * @param key the key associated with the action
   * @return the action associated with the key
   */
  public Action getAction(Integer key)
  {
    return this.keyAction.get(key);
  }

  /**
   * @param mnemonic the mnemonic associated with the keycode
   * @return the key code associated with the mnemonic
   */
  public static Integer getKey(String mnemonic)
  {
    return KeyMap.mnemonicKey.get(mnemonic);
  }

  /**
   * @param mnemonic the mnemonic associated with the action
   * @return the action associated with the mnemonic
   */
  public static Action getAction(String mnemonic)
  {
    return KeyMap.mnemonicAction.get(mnemonic);
  }

  /**
   * @param action the action associated with the mnemonic
   * @return the mnemonic associated with the action
   */
  public static String getMnemonic(Action action)
  {
    return KeyMap.actionMnemonic.get(action);
  }

  /**
   * @param key the key code to associate with an action
   * @param action the action that is associated to the key code
   */
  public void putAction(Integer key,Action action)
  {
    this.keyAction.put(key,action);
    this.actionKey.put(action,key);
  }

  /**
   * @param mnemonic the mnemonic to associate with an action
   * @param key the key code that is associated to the key code
   */
  public static void putMnemonicKey(String mnemonic,int key)
  {
    KeyMap.mnemonicKey.put(mnemonic,Integer.valueOf(key));
    KeyMap.keyMnemonic.put(Integer.valueOf(key),mnemonic);
  }

  /**
   * @param mnemonic the mnemonic to associate with an action
   * @param action the action that is associated to the key code
   */
  public static void putMnemonicAction(String mnemonic,Action action)
  {
    KeyMap.mnemonicAction.put(mnemonic,action);
    KeyMap.actionMnemonic.put(action,mnemonic);
  }

  /**
   * Add an action to the list of mouse buttons that were pressed (for processing actions)
   * @param action the action to perform
   */
  public void pressMouse(Action action)
  {
    this.pressedButtons.add(action);
  }

  /**
   * Remove an action from the list of mouse buttons that were pressed (to stop processing the action)
   * @param action the action that is not being performed anymore
   */
  public void depressMouse(Action action)
  {
    this.pressedButtons.remove(action);
  }

  /**
   * Add an action to the list of keys that were pressed (for processing actions)
   * @param key the key code to process
   */
  public void pressKey(Integer key)
  {
    this.pressedKeys.add(key);
  }

  /**
   * Remove an action from the list of keys that were pressed (to stop processing the action)
   * @param key the action that is not being performed anymore
   * @return whether the key was removed from the list of pressed keys/actions or not
   */
  public boolean depressKey(Integer key)
  {
    return this.pressedKeys.remove(key);
  }

  /**
   * @return a list of pressed keys
   */
  public Collection<Integer> pressedKeys()
  {
    return this.pressedKeys;
  }

  /**
   * @return the list of pressed mouse buttons
   */
  public Collection<Action> pressedButtons()
  {
    return this.pressedButtons;
  }

  /**
   * @return the list of available actions
   */
  public Collection<Action> getActions()
  {
    return KeyMap.actions;
  }

  /**
   * @param mnemonic the mnemonic to look for
   * @return if the mnemonic is in the mapping of mnemonics to keys
   */
  public static boolean containsMnemonicKey(String mnemonic)
  {
    return KeyMap.mnemonicKey.containsKey(mnemonic);
  }

  /**
   * @param action the mnemonic to look for
   * @return if the mnemonic is in the mapping of action to mnemonics
   */
  public static boolean containsActionMnemonic(Action action)
  {
    return KeyMap.actionMnemonic.containsKey(action);
  }

  /**
   * Store an action related to the mneonic in the key for the action in the data variables.
   * @param key the mnemonic to look up
   * @param data the mnemonic for an action
   */
  void loadMnemonic(String key,String data)
  {
    // are both mnemonics stored?
    if (KeyMap.containsMnemonicKey(key) && KeyMap.mnemonicAction.containsKey(data))
    {
      Integer keyValue = KeyMap.getKey(key);
      // is it a valid key to press with a valid action to take?
      if (KeyMap.keys.contains(keyValue))
        this.putAction(KeyMap.getKey(key),KeyMap.getAction(data));
    }
  }

  /**
   * Load a list of keys and actions
   * @param url the url to load the key bindings from
   * @return if the file was successfully loaded or not
   */
  public boolean load(final URL url) // get the key bindings from a URL
  {
    final KeyMap keyMap = new KeyMap();
    boolean success = false;

    try
    {
      Thread thread = new Thread()
      {
        @Override
        public void run()
        {
          try
          {
            if (Debug.debugKeyMap)
              System.out.printf(Messages.getString("KeyMap.LoadingUserKeyboard")); //$NON-NLS-1$
            InputStream inputStream = url.openStream();
            Properties properties = new Properties();
            String key;
            String action;
            properties.load(inputStream);
            for (Map.Entry<Object,Object> entry : properties.entrySet())
            {
              if (Debug.debugKeyMapInternal)
                System.out.print(Messages.getString("KeyMap.Period")); //$NON-NLS-1$
              key = (String)entry.getKey();
              action = (String)entry.getValue();
              keyMap.loadMnemonic(key,action);
              if (Debug.debugKeyMapInternal)
                System.out.printf(Messages.getString("KeyMap.DebugKeyValue"),key,action); //$NON-NLS-1$
            }
            if (Debug.debugKeyMap)
              System.out.printf(Messages.getString("KeyMap.Done")); //$NON-NLS-1$
          }
          catch (IllegalArgumentException illegalArgumentException)
          {
            System.out.printf(Messages.getString("KeyMap.ErrorProblemLoadingKeymap"),url,illegalArgumentException.getMessage()); //$NON-NLS-1$
          }
          catch (IOException iOException)
          {
            System.out.printf(Messages.getString("KeyMap.ErrorProblemLoadingKeymap"),url,iOException.getMessage()); //$NON-NLS-1$
          }
        }
      };
      thread.start();
      thread.join();
      success = true;
    }
    catch (InterruptedException interruptedException)
    {
      System.out.printf(Messages.getString("KeyMap.ErrorProblemWaitingForJoin"),interruptedException.getMessage()); //$NON-NLS-1$
    }
    if (success)
      this.copyValues(this,keyMap);
    return success;
  }

  /**
   * Save a list of keys and actions
   * @param url the url to save the key bindings to
   */
  public void save(final URL url) // get the key bindings from a URL
  {
    final KeyMap thisMap = this;

    try
    {
      Thread thread = new Thread()
      {
        @Override
        public void run()
        {
          try
          {
            if (Debug.debugKeyMap)
              System.out.printf(Messages.getString("KeyMap.SavingUserKeyboard")); //$NON-NLS-1$
            FileOutputStream outputStream = new FileOutputStream(url.getFile());
            Properties properties = new Properties();
            String mnemonic1;
            String mnemonic2;
            for (Integer key : thisMap.keyAction.keySet())
            {
              if (Debug.debugKeyMapInternal)
                System.out.print(Messages.getString("KeyMap.Period")); //$NON-NLS-1$
              if (KeyMap.keyMnemonic.containsKey(key) && KeyMap.containsActionMnemonic(thisMap.keyAction.get(key)))
              {
                mnemonic1 = KeyMap.keyMnemonic.get(key);
                mnemonic2 = KeyMap.actionMnemonic.get(thisMap.keyAction.get(key));
                if (Debug.debugKeyMapInternal)
                  System.out.printf(Messages.getString("KeyMap.DebugMnemonics"),mnemonic1,mnemonic2); //$NON-NLS-1$
                if (!key.equals(mnemonic2) && !mnemonic2.equals(Messages.getString("KeyMap.Undefined"))) //$NON-NLS-1$
                {
                  properties.put(mnemonic1,mnemonic2);
                  if (Debug.debugKeyMapInternal)
                    System.out.printf(Messages.getString("KeyMap.DebugKeyValue"),key,mnemonic2); //$NON-NLS-1$
                }
              }
            }
            properties.store(outputStream,null);
            if (Debug.debugKeyMap)
              System.out.printf(Messages.getString("KeyMap.Done")); //$NON-NLS-1$
          }
          catch (IOException iOException)
          {
            System.out.printf(Messages.getString("KeyMap.ErrorProblemReadingFile"),url,iOException.getMessage()); //$NON-NLS-1$
          }
        }
      };
      thread.start();
      thread.join();
    }
    catch (InterruptedException interruptedException)
    {
      System.out.printf(Messages.getString("KeyMap.ErrorProblemWaitingForJoin"),interruptedException.getMessage()); //$NON-NLS-1$
    }
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("KeyMap.DebugKeyMap"),KeyMap.keys,KeyMap.actions,KeyMap.mnemonicKey,KeyMap.keyMnemonic,KeyMap.mnemonicAction,KeyMap.actionMnemonic,this.pressedKeys); //$NON-NLS-1$
  }
}
/* KeyMap--------------------- */
