/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.Health.DurationType.AGELESS;
import static javoids.ImageMap.Name.JAVOID1;
import static javoids.ImageMap.Name.PACMAN_GHOST;
import static javoids.Shapes.Shape.JAVOID;
import static javoids.Shapes.Shape.POINT;
import static javoids.SoundMap.Sound.JAVOIDDIE;
import static javoids.SoundMap.Sound.JAVOIDHIT;
import static javoids.SoundMap.Sound.PACMAN_EATDOT;
import static javoids.SoundMap.Sound.PACMAN_EATGHOST;

/* Javoid--------------------- */
/**
 * A javoid (asteroid like) object.
 * @author mallette
 */
public final class Javoid extends ShieldedSprite
{
  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long   serialVersionUID       = 1L;
  /** The maximum number of these sprites allowed in the game */
  protected final static int  MAX_NUMBER             = 250;
  /** the maximum number of parts the javoid will break into when exploded */
  protected final static int  MAX_PARTS              = 5;
  /** the maximum size */
  protected final static int  MAX_SIZE               = 100;
  /** the minimum size */
  protected final static int  MIN_SIZE               = 20;
  /** the default size */
  protected final static int  DEFAULT_SIZE           = 100;  // This should match the size of the image!
  /** the chance this will be a homing sprite */
  private final static double CHANCE_HOMING          = 0.01;
  /** the chance this sprite will rotate */
  private final static double CHANCE_ROTATE          = 0.35;
  /** the chance this javoid will be irregularly shaped */
  private final static double CHANCE_IRREGULAR_SHAPE = 0.25;
  /** the chance this javoid will be round (but not an image) */
  private final static double CHANCE_ROUND           = 0.035;
  /** the amount to rotate */
  private final static double ROTATION               = 30;

  /**
   * Constructor
   * @param parent the parent of this sprite
   * @param health the health information
   * @param move the movment information
   * @param _size the size this should be
   */
  public Javoid(BasicSprite parent,Health health,Move move,int _size)
  {
    super(parent,health,move,null,_size,JAVOID1,Shapes.getAreas(POINT,_size,Javoid.MAX_SIZE),Shapes.getColors(POINT));
    this.setDisplayAreas(false);
    if (parent == null)
    {
      if (Math.random() < Javoid.CHANCE_ROUND) // let some javoids be brown balls
      {
        this.setShape(POINT);
        this.setColors();
        this.setDisplayAreas(true);
      }
      else if (Math.random() < Javoid.CHANCE_IRREGULAR_SHAPE) // irregular javoid shape
      {
        this.setShape(JAVOID);
        this.setDisplayAreas(true);
      }
      else
      {
        this.setShape(POINT);
        this.setDisplayAreas(false);
      }
    }
    else
    {
      this.setImage(((Javoid)parent).getImageNumber());
      this.setShape(((Javoid)parent).getShape());
      this.setAreas(((Javoid)parent).getAreas());
      this.setColors(((Javoid)parent).getColors());
    }
    if (Math.random() < Javoid.CHANCE_HOMING)
      this.setHoming(true);
    if (Math.random() < Javoid.CHANCE_ROTATE)
      this.setRotation(Math.random() * Javoid.ROTATION * (Math.random() < 0.5 ? -1 : 1));
    if (Debug.debugJavoid)
    {
      this.setHealth(health);
      this.setMove(new Move(move.getScreen(),100,100,100,100));
    }
    this.accelerate(move.getMaxVelocity());
  }

  /**
   * @return the maximum size
   */
  public int getMaximumSize()
  {
    return Javoid.MAX_SIZE;
  }

  /**
   * @return the minimum size
   */
  public int getMinimumSize()
  {
    return Javoid.MIN_SIZE;
  }

  /**
   * @return the default size
   */
  public int getDefaultSize()
  {
    return Javoid.DEFAULT_SIZE;
  }

  /**
   * @param _size the size to set the sprite's size to
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#setSize(int)
   */
  @Override
  public void setSize(int _size)
  {
    super.setSize(_size >= Javoid.MIN_SIZE ? _size <= Javoid.MAX_SIZE ? _size : Javoid.MAX_SIZE : Javoid.MIN_SIZE);
  }

  /**
   * @param _shape the shape to use for drawing or area checking
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#setShape(javoids.Shapes.Shape)
   */
  @Override
  public void setShape(Shapes.Shape _shape)
  {
    super.setShape(_shape);
    if (JAVOID.equals( _shape ))
      this.setDisplayAreas(true);
    else
      this.setDisplayAreas(false);
  }

  /**
   * @param _damage the damage caused to this sprite
   * @return the new damage level
   */
  /*
   * (non-Javadoc)
   * @see javoids.BasicSprite#modifyDamage(int)
   */
  @Override
  public int modifyDamage(int _damage)
  {
    this.setPoints(this.getDamage());
    int damage = super.modifyDamage(_damage);
    if (this.isAlive())
      Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATDOT : JAVOIDHIT);
    else
      Media.play(BasicSprite.getPacmanGame() ? PACMAN_EATGHOST : JAVOIDDIE);
    return damage;
  }

  /**
   * @return a list of newly created javoids resulting from an explosion
   */
  public SpriteVector<Javoid> breakApart()
  {
    SpriteVector<Javoid> javoids = new SpriteVector<Javoid>(Javoid.MAX_PARTS);

    if (!this.isAlive() && !super.isExploded())
    {
      double maxVelocity;
      int n;
      int damage;
      int _size;
      int newSize;
      _size = this.getSize();
      n = 0;

      super.explode();
      while (_size > Javoid.MIN_SIZE && n < Javoid.MAX_PARTS)
      {
        n++;
        newSize = (int)(Math.random() * _size * 0.8) + 1; // find out how large the new Javoid is based on the remaining size
        _size -= newSize; // remove the new Javoid(chunk) from the original!
        if (newSize >= Javoid.MIN_SIZE) // if the roid is too small it was disintigrated in explosion
        {
          damage = newSize;
          maxVelocity = Math.random() * Move.MAX_VELOCITY;
          if (!Debug.debug)
          {
            Javoid javoid = new Javoid(this,new Health(1,damage,Health.DEFAULT_DURATION,Health.DEFAULT_AGING_RATE,AGELESS),new Move(this.getScreen(),this.getX(),this.getY(),Math.random() * 360,maxVelocity),newSize);
            javoid.move();
            javoids.add(javoid);
          }
        }
      }
    }
    return javoids;
  }

  /**
   * @return the image name currently being used
   */
  /*
   * (non-Javadoc)
   * @see javoids.Sprite#getImageNumber()
   */
  @Override
  public ImageMap.Name getImageNumber()
  {
    return BasicSprite.getPacmanGame() ? PACMAN_GHOST : this.getOriginalImage();
  }
}
/* Javoid--------------------- */
