/*-
 * Javoids -- Javoids is an asteroids based game (that look nothing like the original).
 * 
 * Copyright (C) 1999-2006 Patrick Mallette
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * I can be reached at parickmallette@rogers.com
 */
package javoids;

import static javoids.BasicSprite.Collision.COLLISION;
import static javoids.BasicSprite.Collision.NO_COLLISION;
import static javoids.BasicSprite.Gravity.NONE;
import static javoids.BasicSprite.Gravity.SINK;
import static javoids.BasicSprite.Gravity.SOURCE;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.io.Serializable;
import java.util.Collection;
import java.util.Vector;

/* BasicSprite---------------- */
/**
 * Common anscestor for all game sprites
 * @author mallette
 */
public abstract class BasicSprite extends Object implements Serializable,Sizes
{
  /** Type of collision interaction */
  public enum Collision
  {
    /** a basic collision occurred */
    COLLISION,
    /** nothing collided */
    NO_COLLISION,
    /** a shield to shield collision occurred */
    SHIELD_SHIELD,
    /** a shield to sprite collision occurred */
    SHIELD_SPRITE,
    /** a sprite to shield collision occurred */
    SPRITE_SHIELD,
    /** a sprite to sprite collision occurred */
    SPRITE_SPRITE;
  }

  /** Type of gravity interaction */
  public enum Gravity
  {
    /** no gravity exerted */
    NONE,
    /** gravity from this object attracts other objects */
    SINK,
    /** gravity from this object repels other objects */
    SOURCE;
  }

  /** This is the version used for serializing/deserializing (storing/retrieving) this object */
  private static final long serialVersionUID = 1L;
  /** should the actual object's area be used for collision detection rather than a bounding circle? */
  private static boolean    isAreaChecking = false;
  /** is this a special pacman game? */
  private static boolean    isPacmanGame = false;
  /** was the sprite exploded (destroyed) */
  private transient boolean exploded;
  /** the sprite's health information */
  private Health            health;
  /** is the sprite supposed to move automatically? */
  private boolean           automaticMove;
  /** should the areas be displayed (drawn) for this sprite */
  private boolean           displayAreas;
  /** the gravity type of the sprite */
  private Gravity gravity;
  /** is the sprite a homing sprite? */
  private boolean           homing;
  /** the movement information */
  private Move              move;
  /** what is the parent sprite to this sprite? */
  private BasicSprite       parent;
  /** is this sprite a player? */
  private boolean           player;
  /** the number of points this sprite is worth when shot */
  private int               points;
  /** theamount to rotate when turning */
  private double            rotationAngle;        // number of degrees object rotates
  /** the sprite size */
  protected int             size;

  /**
   * @param _parent null if this is a top level sprite. pointer to this sprite's parent sprite otherwise (it's a bullet that belong to the parent)
   * @param _health initial value of the sprite's health and life expectency
   * @param _move initial location, direction etc.
   * @param _size initial size of sprite
   */
  public BasicSprite(BasicSprite _parent,Health _health,Move _move,int _size)
  {
    this.parent = _parent;
    this.health = _health;
    this.move = _move;
    this.setSize(_size);
    this.setPoints(0);
    this.homing = false;
    this.gravity = NONE;
    this.player = false;
    this.automaticMove = false;
    this.displayAreas = false;
    this.exploded = false;
  }

  /**
   * @param acceleration how much to increase the velocity by
   */
  public void accelerate(double acceleration)
  {
    this.move.accelerate(acceleration);
  }

  /**
   * make the sprite die
   */
  public void kill()
  {
    this.health.setDamage(this.health.getMaxDamage());
  }

  /**
   * make the sprite expire (maximum duration reached)
   */
  public void expire()
  {
    this.health.setDuration(this.health.getMaxDuration());
  }

  /**
   * make the sprite age
   */
  public void age()
  {
    this.health.age();
  }

  /**
   * @return was the sprite blown up (exploded) or not?
   */
  public boolean isExploded()
  {
    return this.exploded;
  }

  /**
   * @return whether or not the object can be restored (i.e. it has lives left)
   */
  public boolean isRestorable()
  {
    return this.getHealth().isRestorable();
  }

  /**
   * @param sprite The sprite that is colliding into this sprite.
   * @param collisionType The type of interaction required for this collision (shield on shield etc).
   * @return the number of points obtained from the collision
   */
  public int collide(BasicSprite sprite,BasicSprite.Collision collisionType)
  {
    int returnValue = 0;
    double relativeVelocity = this.getRelativeVelocity(sprite);
    
    switch (collisionType)
    {
      case NO_COLLISION :
      case SHIELD_SHIELD :
      case SHIELD_SPRITE :
      case SPRITE_SHIELD :
        System.out.printf(Messages.getString("BasicSprite.ErrorCollission"),collisionType); //$NON-NLS-1$
        break;
      default : // covers COLLISION (non-shielded sprites) and sprites with no shields
        this.modifyDamage((int)(relativeVelocity * sprite.getMass() + 1));
        sprite.modifyDamage((int)(relativeVelocity * this.getMass() + 1));
        returnValue = this.getPoints();
        break;
    }
    return returnValue;
  }

  /**
   * @param sprite did this sprite collide with another sprite?
   * @return the type of collision (none, shield to shield, shield to sprite, sprite to sprite)
   */
  public BasicSprite.Collision collisionDetected(BasicSprite sprite)
  {
    boolean collision = false;
    if (this.isAlive() && sprite.isAlive())
    {
      double dx = Math.abs(this.getX() - sprite.getX());
      double dy = Math.abs(this.getY() - sprite.getY());
      double distance1 = Math.sqrt(Common.pow2(dx) + Common.pow2(dy));
      double distance2;
      double crash_distance = this.getSize() / 2.0 + sprite.getSize() / 2.0;
      if (dx > this.getScreen().width / 2)
        dx = this.getScreen().width - dx;
      if (dy > this.getScreen().height / 2)
        dy = this.getScreen().height - dy;
      distance2 = Math.sqrt(Common.pow2(dx) + Common.pow2(dy));
      collision = distance1 <= crash_distance;
      collision = collision || distance2 <= crash_distance;
      if (collision && BasicSprite.isAreaChecking)
      {
        boolean xlt = this.getX() - this.getSize() / 2 < this.getScreen().x;
        boolean ylt = this.getY() - this.getSize() / 2 < this.getScreen().y;
        boolean xgt = this.getX() + this.getSize() / 2 > this.getScreen().x + this.getScreen().width;
        boolean ygt = this.getY() + this.getSize() / 2 > this.getScreen().y + this.getScreen().height;
        int screenXTotal = this.getScreen().x + this.getScreen().width;
        int screenYTotal = this.getScreen().y + this.getScreen().height;
        int modifierX = xlt ? screenXTotal : xgt ? -screenXTotal : 0;
        int modifierY = ylt ? screenYTotal : ygt ? -screenYTotal : 0;
        int totalX = (int)this.getX() + modifierX;
        int totalY = (int)this.getY() + modifierY;
        AffineTransform transform = new AffineTransform();
        Area originalArea1 = new Area();
        Area originalArea2 = new Area();
        Area area1 = null;
        Area area2 = null;
        for (Area area : this.getAreas())
          originalArea1.add(new Area(area));
        for (Area area : sprite.getAreas())
          originalArea2.add(new Area(area));
        transform.setToIdentity();
        transform.translate(this.getX(),this.getY());
        transform.rotate(this.getDirection() * Common.toRadians);
        area1 = new Area(originalArea1);
        area1.transform(transform);
        transform.setToIdentity();
        transform.translate(sprite.getX(),sprite.getY());
        transform.rotate(sprite.getDirection() * Common.toRadians);
        area2 = new Area(originalArea2);
        area2.transform(transform);
        area1.intersect(area2);
        collision = !area1.isEmpty();
        if (!collision && (xlt && ylt || xlt && ygt || xgt && ylt || xgt && ygt))
        {
          transform.setToIdentity();
          transform.translate(totalX,totalY);
          transform.rotate(this.getDirection() * Common.toRadians);
          area1 = new Area(originalArea1);
          area1.transform(transform);
          transform.setToIdentity();
          transform.translate(sprite.getX(),sprite.getY());
          transform.rotate(sprite.getDirection() * Common.toRadians);
          area2 = new Area(originalArea2);
          area2.transform(transform);
          area1.intersect(area2);
          collision = !area1.isEmpty();
        }
        if (!collision && (xlt || xgt))
        {
          transform.setToIdentity();
          transform.translate(totalX,this.getY());
          transform.rotate(this.getDirection() * Common.toRadians);
          area1 = new Area(originalArea1);
          area1.transform(transform);
          transform.setToIdentity();
          transform.translate(sprite.getX(),sprite.getY());
          transform.rotate(sprite.getDirection() * Common.toRadians);
          area2 = new Area(originalArea2);
          area2.transform(transform);
          area1.intersect(area2);
          collision = !area1.isEmpty();
        }
        if (!collision && (ylt || ygt))
        {
          transform.setToIdentity();
          transform.translate(this.getX(),totalY);
          transform.rotate(this.getDirection() * Common.toRadians);
          area1 = new Area(originalArea1);
          area1.transform(transform);
          transform.setToIdentity();
          transform.translate(sprite.getX(),sprite.getY());
          transform.rotate(sprite.getDirection() * Common.toRadians);
          area2 = new Area(originalArea2);
          area2.transform(transform);
          area1.intersect(area2);
          collision = !area1.isEmpty();
        }
      }
    }
    return collision ? COLLISION : NO_COLLISION;
  }

  /**
   * @param g2d the graphics context
   * @param foregroundImage the buffer where drawing is performed
   */
  public abstract void draw(Graphics2D g2d,Graphics2D foregroundImage);

  /**
   * @return the rate that the sprite ages
   */
  public int getAgingRate()
  {
    return this.health.getAgingRate();
  }

  /**
   * Override this method to have a larger object break up into more pieces. You must call super.explode() right after you check to see if it was exploded or not or you may have numerous explosions happening due to thread interactions.
   * @param <E> the type of sprite being exploded
   * @return null (this should be overridden)
   */
  public <E extends BasicSprite> Collection<E> explode()
  {
    this.exploded = true;
    return null;
  }

  /**
   * @return the health status of this sprite
   */
  public Health getHealth()
  {
    return this.health;
  }

  /**
   * Is the game using area checking?
   * @return true/false
   */
  public static boolean getAreaChecking()
  {
    return BasicSprite.isAreaChecking;
  }

  /**
   * @return The areas that make up the sprite's shape.
   */
  public abstract Vector<Area> getAreas();

  /**
   * @return the color of the sprite should use to draw something (changes based on damage and duration remaining)
   */
  public Color getColor()
  {
    int damageRating = (this.getMaxDamage() - this.getDamage()) * 10 / this.getMaxDamage();
    int durationRating = (this.getMaxDuration() - this.getDuration()) * 10 / this.getMaxDuration();
    Color color;

    switch (damageRating < durationRating ? damageRating : durationRating)
    {
      case 10 :
      case 9 :
      case 8 :
      case 7 :
        color = Color.green;
        break;
      case 6 :
      case 5 :
      case 4 :
      case 3 :
        color = Color.yellow;
        break;
      default :
        color = Color.red;
        break;
    }
    return color;
  }

  /** @return The colors that match each shape that make up the sprite's shape. */
  public abstract Vector<Color> getColors();

  /** @return the direction (0 (and 360) are up on screen, 90 is right on screen, 180 is down on screen, 270 is left on screen) */
  public double getDirection()
  {
    return this.move.getDirection();
  }

  /**
   * @return what type of gravity this sprite has ( sink, source, none)
   */
  public Gravity getGravity()
  {
    return this.gravity;
  }

  /**
   * @return the sprite's damage level (0 is dead currently)
   */
  public int getDamage()
  {
    return this.health.getDamage();
  }

  /**
   * @return return the spite's liftime remaining
   */
  public int getDuration()
  {
    return this.health.getDuration();
  }

  /**
   * @return return the number of lives the sprite has left
   */
  public int getDeaths()
  {
    return this.health.getDeaths();
  }

  /**
   * @return return the mass of the sprite (used for calculating damage)
   */
  public double getMass()
  {
    return this.getMaxDamage() / 10.0;
  }

  /**
   * @return the maximum damage the sprite can take before being destroyed
   */
  public int getMaxDamage()
  {
    return this.health.getMaxDamage();
  }

  /**
   * @return the maximum amount of ticks the sprite can live
   */
  public int getMaxDuration()
  {
    return this.health.getMaxDuration();
  }

  /**
   * @return the maximum number of lives this sprite can ever have
   */
  public int getMaxDeaths()
  {
    return this.health.getMaxDeaths();
  }

  /**
   * @return the maximum velocity the sprite can attain
   */
  public double getMaxVelocity()
  {
    return this.move.getMaxVelocity();
  }

  /**
   * @return the position and velocity information for the sprite
   */
  public Move getMove()
  {
    return this.move;
  }

  /**
   * @return the scaling factor to use for drawing the sprite
   */
  public abstract double getMultiplier();

  /**
   * Is the game running in pacman mode?
   * @return true/false
   */
  public static boolean getPacmanGame()
  {
    return BasicSprite.isPacmanGame;
  }

  /**
   * @return the owning sprite of this sprite
   */
  public BasicSprite getParent()
  {
    return this.parent;
  }

  /**
   * @return the number of points the sprite is worth
   */
  public int getPoints()
  {
    return this.points;
  }

  /**
   * @param basicSprite the other sprite that the velocity is being measured relative to
   * @return the relative velocity between the two sprites
   */
  public double getRelativeVelocity(BasicSprite basicSprite)
  {
    return Math.sqrt(Common.pow2(this.move.getVelocity().x - basicSprite.getMove().getVelocity().x) + Common.pow2(this.move.getVelocity().y - basicSprite.getMove().getVelocity().y));
  }

  /**
   * @return the angle to rotate when turning
   */
  public double getRotation()
  {
    return this.rotationAngle;
  }

  /**
   * @return the screen size
   */
  public Rectangle getScreen()
  {
    return this.move.getScreen();
  }

  /**
   * @return the size of this sprite
   */
  public int getSize()
  {
    return this.size;
  }

  /**
   * @return the horizontal position of the sprite on the screen (x-axis)
   */
  public double getX()
  {
    return this.move.getX();
  }

  /**
   * @return the vertical position of the sprite on the screen (y-axis)
   */
  public double getY()
  {
    return this.move.getY();
  }

  /**
   * is the sprite health
   * @return is the sprite health?
   */
  public boolean isAlive()
  {
    return this.health.isAlive();
  }

  /**
   * is the sprite moving by itself?
   * @return is the sprite moving by itself?
   */
  public boolean isAutomaticMove()
  {
    return this.automaticMove;
  }

  /**
   * is the sprite supposed to display it's areas?
   * @return should the game display the areas that make up drawn sprites?
   */
  public boolean isDisplayAreas()
  {
    return this.displayAreas;
  }

  /**
   * is the sprite a gravity source
   * @return is the sprite a gravity source (repells objects)?
   */
  public boolean isGravitySource()
  {
    return SOURCE.equals( this.gravity );
  }

  /**
   * is the sprite a gravity well
   * @return is the sprite a gravity well (attracts objects)?
   */
  public boolean isGravityWell()
  {
    return SINK.equals( this.gravity );
  }

  /**
   * is the sprite a homing sprite
   * @return does the sprite seek out a target?
   */
  public boolean isHoming()
  {
    return this.homing;
  }

  /**
   * is the sprite a player?
   * @return is the sprite a player?
   */
  public boolean isPlayer()
  {
    return this.player;
  }

  /**
   * modify the sprite's damage level
   * @param damage the amount to modify the damage level by
   * @return the new damage level
   */
  public int modifyDamage(int damage)
  {
    return this.health.modifyDamage(damage);
  }

  /**
   * modify the sprite's duration
   * @param duration the amount to modify the duration by
   * @return the new duration
   */
  public int modifyDuration(int duration)
  {
    return this.health.modifyDuration(duration);
  }

  /**
   * modify the sprite's lives
   * @param deaths the amount to modify the number of lives by
   * @return the new number of lives
   */
  public int modifyDeaths(int deaths)
  {
    return this.health.modifyDeaths(deaths);
  }

  /**
   * modify the sprite's point total
   * @param _points the number of points to modify the sprite's point value by
   * @return the new point total
   */
  public int modifyPoints(int _points)
  {
    return this.points += _points * (Javoids.getDifficulty() + 1) / 2;
  }

  /**
   * move the sprite (apply changes based on the velocity, ageing rate, and any rotation)
   */
  public void move()
  {
    if (this.getRotation() != 0)
      this.turn(this.getRotation() > 0 ? Move.RIGHT : Move.LEFT,Math.abs(this.getRotation()));
    this.move.move();
    this.health.age();
  }

  /**
   * set the sprite's aging rate
   * @param agingRate the amount to age the sprite by
   */
  public void setAgingRate(int agingRate)
  {
    this.health.setAgingRate(agingRate);
  }

  /**
   * set the sprite's health information
   * @param _health the new values for the sprite's health information
   */
  public void setHealth(Health _health)
  {
    this.health = new Health(_health);
  }

  /**
   * Set area checking value (should the game check a sprite's actual area rather than a bounding circle)
   * @param _isAreaChecking is the game using area checking for collision detection?
   * @return true/false
   */
  public static boolean setAreaChecking(boolean _isAreaChecking)
  {
    BasicSprite.isAreaChecking = _isAreaChecking;
    return BasicSprite.isAreaChecking;
  }

  /**
   * set the sprite's value to say if it should move automatically
   * @param _automaticMove set the sprite to move automatically, if true, or disable it if false.
   */
  public void setAutomaticMove(boolean _automaticMove)
  {
    this.automaticMove = _automaticMove;
  }

  /**
   * set the sprite's direction
   * @param direction setthe direction (0 (and 360) are up on screen, 90 is right on screen, 180 is down on screen, 270 is left on screen)
   */
  public void setDirection(double direction)
  {
    this.move.setDirection(direction);
  }

  /**
   * set the sprite's value to say if it should display the sprite's areas when drawing an image (for debugging)
   * @param _displayAreas set if the spr
   */
  public void setDisplayAreas(boolean _displayAreas)
  {
    this.displayAreas = _displayAreas;
  }

  /**
   * set the sprite's gravity
   * @param _gravity the type of gravity this sprite has (source, sink, none)
   */
  public void setGravity(Gravity _gravity)
  {
    this.gravity = NONE.equals( _gravity ) || SINK.equals( _gravity ) || SOURCE.equals( _gravity ) ? _gravity : NONE;
  }

  /**
   * set the sprite's value to determine if it is a homing sprite or not
   * @param _homing the homing value (is it a homing sprite or not)
   */
  public void setHoming(boolean _homing)
  {
    this.homing = _homing;
  }

  /**
   * set the sprite's damage level
   * @param damage the sprite's new damage level
   */
  public void setDamage(int damage)
  {
    this.health.setDamage(damage);
  }

  /**
   * set the sprite's duration
   * @param duration the sprite's new duration
   */
  public void setDuration(int duration)
  {
    this.health.setDuration(duration);
  }

  /**
   * set the sprite's lives
   * @param lives the sprite's new number of lives
   */
  public void setDeaths(int lives)
  {
    this.health.setDeaths(lives);
  }

  /**
   * set the sprite's maximum damage level
   * @param maxDamage the sprite's new maximum damage level
   */
  public void setMaxDamage(int maxDamage)
  {
    this.health.setMaxDamage(maxDamage);
  }

  /**
   * set the sprite's maximum duration
   * @param maxDuration the sprite's new maximum duration
   */
  public void setMaxDuration(int maxDuration)
  {
    this.health.setMaxDuration(maxDuration);
  }

  /**
   * set the sprite's maximum lives
   * @param maxLives the sprite's new maximum number of lives
   */
  public void setMaxLives(int maxLives)
  {
    this.health.setMaxDeaths(maxLives);
  }

  /**
   * set the sprite's maximum velocity
   * @param maxVelocity the sprite's new maximum velocity
   */
  public void setMaxVelocity(double maxVelocity)
  {
    this.move.setMaxVelocity(maxVelocity);
  }

  /**
   * set the sprite's movement information
   * @param _move the sprite's new movment information
   */
  public void setMove(Move _move)
  {
    this.move = new Move(_move);
  }

  /**
   * Set pacman game value
   * @param _isPacmanGame is the game using pacman settings?
   * @return true/false
   */
  public static boolean setPacmanGame(boolean _isPacmanGame)
  {
    BasicSprite.isPacmanGame = _isPacmanGame;
    return BasicSprite.isPacmanGame;
  }

  /**
   * set the sprite's parent
   * @param _parent the sprite's new parent
   */
  public void setParent(BasicSprite _parent)
  {
    this.parent = _parent;
  }

  /**
   * set the sprite's value to say if it is a player or not
   * @param _player is this sprite a player?
   */
  public void setPlayer(boolean _player)
  {
    this.player = _player;
  }

  /**
   * set the sprite's point value
   * @param _points the sprite's new point total
   */
  public void setPoints(int _points)
  {
    this.points = _points;
  }

  /**
   * set the sprite's rotation amount
   * @param _rotationAngle the sprite's new amount to rotate by when turning
   */
  public void setRotation(double _rotationAngle)
  {
    this.rotationAngle = _rotationAngle % 360.0;
  }

  /**
   * set the sprite's screen size
   * @param screen the sprite's new screen size
   */
  public void setScreen(Rectangle screen)
  {
    this.move.setScreen(screen);
  }

  /**
   * set the sprite's size
   * @param _size the sprite's new size
   */
  public abstract void setSize(int _size);

  /**
   * set the sprite's x coordinate
   * @param x the sprite's new x xoordinate
   */
  public void setX(double x)
  {
    this.move.setX(x);
  }

  /**
   * set the sprite's y coordinate
   * @param y the sprite's new y xoordinate
   */
  public void setY(double y)
  {
    this.move.setY(y);
  }

  /**
   * Provide a String representation of this object.
   * @return String A representation of the object for debugging.
   */
  @Override
  public String toString()
  {
    return String.format(Messages.getString("BasicSprite.ToString"),this.health,this.move,Integer.valueOf(this.points),Integer.valueOf(this.size),this.gravity,Boolean.valueOf(this.homing),Boolean.valueOf(this.player),Boolean.valueOf(this.automaticMove),Boolean.valueOf(BasicSprite.isPacmanGame)); //$NON-NLS-1$
  }

  /**
   * turn the sprite to a new direction
   * @param turn_direction the direction to turn
   * @param n the amount to turn by
   */
  public void turn(double turn_direction,double n)
  {
    this.move.turn(turn_direction,n);
  }
}
/* BasicSprite---------------- */
