/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) AttributeConverter.java 1.25 - last change made 07/30/96
 */

package sunsoft.jws.visual.rt.type;

import sunsoft.jws.visual.rt.base.*;
import java.util.Enumeration;

/**
 * This class converts Attributes to strings and back again.
 *
 * @see Attribute
 * @version 1.25, 07/30/96
 */
public class AttributeConverter extends Converter {

  /**
   * Converts an attribute to a string.
   *
   * @param obj Attribute instance to convert
   * @param buf buffer to which to add the string
   * @return string representation of an attribute (type, name, and value)
   */
  public void convertToString(Object obj, StringBuffer buf) {
    if (obj != null) {
      Attribute a = (Attribute) obj;

      String type = a.getType();
      // Special case for GBConstraints to make the save file neater
      if (type.equals("sunsoft.jws.visual.rt.awt.GBConstraints"))
	type = "GBC";
      ListParser.quote(type, buf, false);
      buf.append(' ');

      ListParser.quote(a.getName(), buf, false);
      buf.append(' ');

      Converter c = getConverter(a.getType());
      if (c == null) {
	buf.append("null");
      }
      else {
	ListParser.quote(c.convertToString(a.getValue()), buf, false);
      }
    }
  }

  /**
   * Call the convertFromString method that takes additionhal
   * arguments. An AttributeManager object is needed to operate on.
   *
   * @exception Error when called
   */
  public Object convertFromString(String s) {
    throw new Error("AttributeConverter needs a shadow object " +
		    "argument to operate");
  }

  /**
   * Converts a string (type, name, and value) to an Attribute.
   *
   * @param version description file version
   * @param mgr object in which the attribute will be used
   * @param type type of the attribute
   * @param key name of the attribute
   * @param value value of the attribute
   * @return a new instance of Attribute
   * @exception ParseError when there is an error in one of the strings
   */
  public Object convertFromString(double version, AttributeManager mgr,
				  String type, String key, String value) {

    // get the targeted attribute from the shadow object
    if (!mgr.hasAttribute(key)) {
      throw new ParseException(Global.newline() +
			       "    Unknown attribute \"" + key + "\"" +
			       Global.newline() +
			       "      class = " + mgr.getClass().getName() +
			       Global.newline() + "      type = " + type +
			       Global.newline() + "      key = " + key +
			       Global.newline() + "      value = " + value);
    }

    // get type of attribute and convert and set the value
    String mgrType = mgr.getType(key);

    if (version >= 3) {
      if (!type.equals(mgrType) && !type.equals("GBC")) {
	throw new ParseException(Global.newline() +
		 "    Type in gui file \"" + type + "\" does not match" +
		 Global.newline() + "    expected type \"" + mgrType + "\".");
      }
    }

    Converter valueConverter = getConverter(mgrType);
    if (valueConverter == null)
      throw new ParseException("Unknown attribute type " + mgrType);

    mgr.set(key, valueConverter.convertFromString(value));
    return null;
  }

  /**
   * Converts an attribute to the code that would set it for an
   * AttributeManager object.
   *
   * @param amName name of the AttributeManager
   * @param a attribute (which contains type, name, and value)
   * @param indent number of spaces to indent the code line(s)
   * @param buf buffer to which to add the code
   */
  public void convertToCodeBlock(String amName, Attribute a,
				 int indent, StringBuffer buf) {

    Converter c = getConverter(a.getType());
    c.convertToCodeBlock(amName, a, indent, buf);
  }

  /**
   * Call the convertToCode method that takes more arguments instead.
   *
   * @exception Error when called
   */
  public String convertToCode(Object obj) {
    throw new Error("internal error - " +
		    "convertToCode with minimal arguments cannot be called");
  }
}
