/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) @(#) DialogShadow.java 1.54 - last change made 08/02/96
 */

package sunsoft.jws.visual.rt.shadow.java.awt;

import sunsoft.jws.visual.rt.base.DesignerAccess;
import sunsoft.jws.visual.rt.base.*;
import sunsoft.jws.visual.rt.type.AMRef;
import sunsoft.jws.visual.rt.awt.RootDialog;

import java.awt.*;
import java.util.Enumeration;

// This class makes the assumption that dialogClass is either null or a
// subclass of Frame.  No error checking is done here to enforce this.

/**
 * Wraps an AWT widget.  The attributes available for this
 * class are listed below.  In the type column, type names beginning
 * with "sunsoft.jws.visual.rt" have been abbreviated to begin with "rt".
 *
 * <pre>
    name            type                      default value
    -----------------------------------------------------------------------
    frame           rt.type.AMRef             null
    modal           java.lang.Boolean         false 
    resizable       java.lang.Boolean         true 
    title           java.lang.String          "Unnamed Dialog" 
 * </pre>
 *
 * modal: when set to true, the dialog's presence (its being visible)
 * keeps the user from doing anything else in the generated application
 * until the dialog is popped down.
 * <p>
 * Check the super class for additional attributes.
 *
 * @see Dialog
 * @version 	1.54, 08/02/96
 */
public class DialogShadow extends WindowShadow {

  protected Frame dialogFrame;

  public DialogShadow() {
    attributes.add("frame", "sunsoft.jws.visual.rt.type.AMRef",
		   null, CONSTRUCTOR | NONBODY | HIDDEN);
    attributes.alias("text", "title");
    attributes.add("title", "java.lang.String", "Unnamed Dialog", NOEDITOR);
    attributes.add("modal", "java.lang.Boolean", Boolean.FALSE, CONSTRUCTOR);
    attributes.add("resizable", "java.lang.Boolean", Boolean.TRUE, 0);
  }

  //
  // getOnBody
  //
  protected Object getOnBody(String key) {
    if (body instanceof Dialog)
      return getOnDialogBody(key);
    else
      return getOnFrameBody(key);
  }

  private Object getOnDialogBody(String key) {
    Dialog dialog = (Dialog)body;

    if (key.equals("modal")) {
      if (isLive())
	return new Boolean(dialog.isModal());
      else
	return getFromTable("modal");
    }
    else if (key.equals("resizable"))
      return(new Boolean(dialog.isResizable()));
    else if (key.equals("title"))
      return(dialog.getTitle());
    else
      return(super.getOnBody(key));
  }

  private Object getOnFrameBody(String key) {
    Frame frame = (Frame)body;

    if (key.equals("modal"))
      return(getFromTable("modal"));
    else if (key.equals("resizable"))
      return(new Boolean(frame.isResizable()));
    else if (key.equals("title"))
      return(frame.getTitle());
    else
      return(super.getOnBody(key));
  }

  //
  // setOnBody
  //
  protected void setOnBody(String key, Object value) {
    if (body instanceof Dialog)
      setOnDialogBody(key, value);
    else
      setOnFrameBody(key, value);
  }

  private void setOnDialogBody(String key, Object value) {
    Dialog dialog = (Dialog)body;

    if (key.equals("modal"))
      return; // cannot set modal once dialog is created
    else if (key.equals("resizable"))
      dialog.setResizable(((Boolean)value).booleanValue());
    else if (key.equals("title"))
      dialog.setTitle((String)value);
    else
      super.setOnBody(key, value);
  }

  private void setOnFrameBody(String key, Object value) {
    Frame frame = (Frame)body;

    if (key.equals("modal"))
      return; // cannot set modal once dialog is created
    else if (key.equals("resizable"))
      frame.setResizable(((Boolean)value).booleanValue());
    else if (key.equals("title"))
      frame.setTitle((String)value);
    else
      super.setOnBody(key, value);
  }

  public void create() {
    if (!isCreated() && Global.isWindows()) {
      createBody();
      if (!(body instanceof Dialog)) {
	if (attributes.get("background").flagged(DEFAULT)) {
	  attributes.add("background", "java.awt.Color",
			 Color.lightGray, DONTFETCH);
	}
	if (attributes.get("font").flagged(DEFAULT)) {
	  attributes.add("font", "java.awt.Font",
			 new Font("Dialog", Font.PLAIN, 12), DONTFETCH);
	}
      }
    }

    super.create();
  }

  public void createBody() {
    dialogFrame = getFrame();

    // Create the dialog
    if (!inDesignerRoot()) {
      boolean modal;
      if (isLive())
	modal = ((Boolean) getFromTable("modal")).booleanValue();
      else
	modal = false;

      String title = (String) getFromTable("title");

      Dialog dialog = new RootDialog(dialogFrame, title, modal);
      body = dialog;
    }
    else {
      try {
	body = DesignerAccess.getDialogClass().newInstance();
      } catch (InstantiationException ex) {
	throw new Error(ex.toString());
      } catch (IllegalAccessException ex) {
	throw new Error(ex.toString());
      }
    }
  }

  public void showComponent() {
    // Call addNotify on our frame if necessary.  Need to check for
    // null because we might not be created yet.
    if (dialogFrame != null && dialogFrame.getPeer() == null)
      dialogFrame.addNotify();

    super.showComponent();
  }

  protected Frame getFrame() {
    AMRef ref;
    FrameShadow frameShadow;
    Frame frame;

    // Determine the frame that this dialog hangs off of
    ref = (AMRef) getFromTable("frame");
    if (ref != null) {
      AttributeManager scope = getForwardingGroup("frame");
      if (scope == null)
	scope = this;

      frameShadow = (FrameShadow)ref.getRef(scope);

      if (frameShadow == null)
	throw new Error("FrameShadow reference is null");

      frame = getFrameBody(frameShadow);
    }
    else {
      frame = lookupFrame();
    }

    if (frame == null)
      throw new Error("frame is null");

    return frame;
  }

  private Frame lookupFrame() {
    Root root = getRoot();

    // Try the main container
    AttributeManager mgr = root.getMainChild();
    if (mgr instanceof Group)
      mgr = DesignerAccess.getContainer((Group)mgr);

    if (mgr instanceof FrameShadow)
      return getFrameBody((FrameShadow)mgr);

    // Try any other child of the root if we aren't the main container
    if (mgr != this) {
      Enumeration e = root.getChildList();
      while (e.hasMoreElements()) {
	mgr = (AttributeManager)e.nextElement();
	if (mgr instanceof FrameShadow) {
	  Boolean v = (Boolean)mgr.get("visible");
	  if (v.booleanValue())
	    return getFrameBody((FrameShadow)mgr);
	}
      }
    }

    // Try for a shadow parent of the root
    FrameShadow frameShadow = lookupGroupFrame(root.getGroup());
    if (frameShadow != null) {
      Boolean v = (Boolean)mgr.get("visible");
      if (v.booleanValue())
	return getFrameBody(frameShadow);
    }

    // Try the toplevel
    return getGroup().getTopLevel();
  }

  private Frame getFrameBody(FrameShadow frameShadow) {
    Frame frame = (Frame)frameShadow.getBody();

    if (frame == null) {
      frameShadow.createBody();
      frame = (Frame)frameShadow.getBody();
    }

    return frame;
  }

  private FrameShadow lookupGroupFrame(Group group) {
    if (group == null)
      return null;

    AttributeManager mgr = DesignerAccess.getContainer(group);
    if (mgr instanceof FrameShadow)
      return (FrameShadow)mgr;
    else
      return lookupGroupFrame(group.getParentGroup());
  }

  // Overridden AMContainer interfaces

  public void addChildBody(Shadow child) {
    if (body != null) {
      if ((((Container) body).getLayout() instanceof BorderLayout)
	  && getChildCount() == 1) {
	// most frames will have a single central child
	((Container) body).add("Center", (Component) child.getBody());
      }
      else {
	super.addChildBody(child);
      }
    }
  }
}
