/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) TextView.java 1.17 - last change made 07/10/96
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.base.Global;

import java.awt.*;
import java.util.Vector;
import java.util.Hashtable;

public class TextView extends Canvas implements Scrollable {
  private static final int RIGHT_MOUSE = 4;

  protected Vector items;

  protected int fontHeight, lineWidth, lineHeight;
  protected FontMetrics fontMetrics;
  protected int minrows = 10;
  protected int mincolumns = 15;

  static protected final int textIndent = 6;
  static protected final int textBorder = 2;
  static protected final int viewBorder = 0;
  static protected final int viewIPad = 2;

  protected int selection = -1;
  protected int minWidth = 0;

  private int scrollx = 0;
  private int scrolly = 0;
  private Image buffer;

  private Hashtable stringWidthTable;

  public TextView() {
    stringWidthTable = new Hashtable();
    setBackground(Color.white);
  }

  //
  // Accessor methods.  These are forwarded from the TextList class.
  //
  public void setMinimumRows(int num) {
    minrows = num;
  }

  public int getMinimumRows() {
    return minrows;
  }

  public void setMinimumColumns(int num) {
    mincolumns = num;
  }

  public int getMinimumColumns() {
    return mincolumns;
  }

  public int getRows() {
    if (lineHeight == 0)
      return 0;

    Dimension size = size();
    int h = size.height - (viewBorder + viewIPad);
    return ((h+lineHeight-1)/lineHeight);
  }

  public void updateView() {
    if (selection >= items.size())
      selection = -1;

    cacheMinWidth();
    repaint();
  }

  public void select(int index) {
    if (index >= items.size())
      return;
    if (index < -1)
      return;

    if (selection != index) {
      selection = index;
      repaint();
    }
  }

  public void deselect(int index) {
    if (index != selection)
      return;

    selection = -1;
    repaint();
  }

  public int getSelectedIndex() {
    return selection;
  }

  public Object getSelectedItem() {
    if (selection == -1)
      return null;
    else
      return items.elementAt(selection);
  }

  //
  // Package private accessor methods
  //
  void items(Vector items) {
    this.items = items;
  }

  //
  // Component methods
  //
  public Dimension minimumSize() {
    int bd = 2 * (viewBorder + viewIPad);
    return new Dimension(minWidth + bd, (minrows * lineHeight) + bd);
  }

  public Dimension preferredSize() {
    return minimumSize();
  }

  //
  // Scrollable methods
  //
  public void scrollX(int x) {
    scrollx = x;
    repaint();
  }

  public void scrollY(int y) {
    scrolly = y;
    repaint();
  }

  public Dimension scrollSize() {
    int bd = 2*(viewBorder + viewIPad);
    return new Dimension(minWidth+bd, items.size()*lineHeight+bd);
  }

  public Dimension viewSize() {
    Dimension size = size();
    int bd = 2 * (viewBorder + viewIPad);
    size.width -= bd;
    size.height -= bd;
    return size;
  }

  public int lineHeight() {
    return lineHeight;
  }

  //
  // Event handling for selections
  //
  public boolean mouseDown(Event e, int x, int y) {
    selectY(e);
    return true;
  }

  public boolean mouseDrag(Event e, int x, int y) {
    // Workaround for bug observed on WindowsNT where you get spurious
    // mouse drag events when pressing the mouse.  The spurious event
    // has coordinates x=-1 and y=-1.
    if (!Global.isWindows() || e.y != -1)
      selectY(e);

    return true;
  }

  public boolean mouseUp(Event e, int x, int y) {
    return true;
  }

  private int prevX = 0;
  private int prevY = 0;

  private void selectY(Event e) {
    e.y += scrolly - (viewBorder + viewIPad);
    int index = e.y/lineHeight;
    int size = items.size();

    if (index < 0 || index >= size)
      return;

    int id = Event.LIST_SELECT;

    // Workaround for windows bug where the click count is set to 2, even
    // if the two clicks were separated by a lot of distance.
    if (e.id == Event.MOUSE_DOWN && Global.isWindows()) {
      if (Math.abs(prevX - e.x) + Math.abs(prevY - e.y) > 4)
	e.clickCount = 1;

      prevX = e.x;
      prevY = e.y;
    }

    //
    // Ignore double-clicks on Windows because they are sent spuriously.
    //
    if ((e.clickCount == 2 && !Global.isWindows()) ||
	e.modifiers == RIGHT_MOUSE) {
      id = Event.ACTION_EVENT;
    }

    if (selection != index) {
      selection = index;
      repaint();
      postEvent(new Event(getParent(), id, items.elementAt(index)));
    }
    else if (e.id == Event.MOUSE_DOWN) {
      postEvent(new Event(getParent(), id, items.elementAt(index)));
    }
  }

  //
  // Painting
  //
  public void reshape(int x, int y, int width, int height) {
    super.reshape(x, y, width, height);
    cacheLineWidth();

    if (width <= 0 || height <= 0)
      return;

    // Create the image used for double-buffering
    if (buffer == null ||
	(width != buffer.getWidth(this) || height != buffer.getHeight(this)))
      buffer = createImage(width, height);
  }

  public void update(Graphics g) {
    paint(g);
  }

  public void paint(Graphics g) {
    if (buffer == null)
      return;

    g = buffer.getGraphics();
    g.setFont(getFont());

    Dimension d = size();

    g.setColor(getBackground());
    g.fillRect(0, 0, d.width, d.height);

    if (isEnabled())
      g.setColor(getForeground());
    else
      g.setColor(getBackground().darker());
    drawItems(g);

    g.setColor(getBackground());
    drawBorder(g);

    g = getGraphics();
    g.drawImage(buffer, 0, 0, this);
  }

  private void drawItems(Graphics g) {
    Dimension d = size();
    int size = items.size();

    int viewTop, viewBottom, lineTop, lineBottom;
    int bd = viewBorder + viewIPad;
    int yoff;

    viewTop = scrolly;
    viewBottom = scrolly + d.height;

    for (int i=0; i<size; i++) {
      lineTop = i*lineHeight;
      lineBottom = lineTop + lineHeight;

      if (lineTop > viewBottom || lineBottom < viewTop)
	continue;

      yoff = lineTop - viewTop + bd;
      drawLine(g, i, -scrollx+bd, yoff);
    }
  }

  protected void drawLine(Graphics g, int index, int xoff, int yoff) {
    String name = (String)items.elementAt(index);

    int x = textIndent;
    int y = (lineHeight + fontHeight)/2 - 1;

    if (index == selection) {
      g.setColor(new Color(0, 0, 128));
      g.fillRect(xoff, yoff, lineWidth, lineHeight);
      g.setColor(Color.white);
    }

    // Useful for pixel debugging
    // g.drawRect(xoff, yoff, lineWidth-1, lineHeight-1);

    g.drawString(name, x+xoff, y+yoff);

    if (index == selection) {
      g.setColor(getForeground());
    }
  }

  private void drawBorder(Graphics g) {
    Dimension size = size();

    for (int i=0; i<viewIPad; i++)
      g.drawRect(viewBorder+i, viewBorder+i,
		 size.width-1-2*(i+viewBorder),
		 size.height-1-2*(i+viewBorder));
  }

  public void addNotify() {
    super.addNotify();
    cacheAll();
  }

  public void setFont(Font f) {
    super.setFont(f);

    stringWidthTable.clear();
    if (getPeer() != null)
      cacheAll();
  }

  private void cacheAll() {
    cacheLineHeight();
    cacheMinWidth();
  }

  //
  // Need to call this when the list of items changes, the font changes,
  // or the mincolumns changes.  The mincolumns change should be followed
  // by a call to updateView for the change to take effect.  It is only
  // necessary to call updateView if addNotify has not yet been called.
  //
  protected void cacheMinWidth() {
    minWidth = mincolumns * getStringWidth("0");

    int count = items.size();
    for (int i=0; i<count; i++)
      minWidth = Math.max(minWidth,
			  getStringWidth((String)items.elementAt(i)));

    minWidth += textIndent * 2;
    cacheLineWidth();
  }

  protected int getStringWidth(String s) {
    if (fontMetrics == null)
      return 0;

    Integer val = (Integer)stringWidthTable.get(s);
    if (val == null) {
      val = new Integer(fontMetrics.stringWidth(s));
      stringWidthTable.put(s, val);
    }

    return val.intValue();
  }

  //
  // Need to call this when the size changes and when the minWidth changes.
  //
  protected void cacheLineWidth() {
    Dimension size = size();
    int bd = 2*(viewBorder + viewIPad);
    lineWidth = Math.max(minWidth, size.width-bd);
  }

  //
  // Need to call this when the font changes.
  //
  protected void cacheLineHeight() {
    lineHeight = 0;
    Graphics g = getGraphics();
    if (g == null)
      return;

    Font f = getFont();
    if (f == null)
      return;

    fontMetrics = g.getFontMetrics(f);
    fontHeight = fontMetrics.getMaxAscent();

    lineHeight = fontHeight + 2*textBorder;
  }
}
