/*
 *        Copyright (C) 1996  Active Software, Inc.
 *                  All rights reserved.
 *
 * @(#) ScrollPanel.java 1.26 - last change made 08/02/96
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.awt.*;
import sunsoft.jws.visual.rt.base.Util;
import java.awt.*;

public class ScrollPanel extends VJPanel {
  protected VJScrollbar vbar, hbar;

  private Component comp;
  private boolean vshow, hshow;
  private GBLayout gridbag;

  private int scrollAreaWidth = 0;
  private int scrollAreaHeight = 0;
  private Insets scrollAreaInsets;

  public ScrollPanel() {
    super(Util.WIN95_FIELD_BORDER);
    setInternalInsets(new Insets(0, 0, 0, 0));

    GBConstraints c = new GBConstraints();
    gridbag = new GBLayout();
    setLayout(gridbag);

    hbar = new VJScrollbar(VJScrollbar.HORIZONTAL);
    vbar = new VJScrollbar(VJScrollbar.VERTICAL);

    c.gridx = 1;
    c.gridy = 0;
    c.shrinkx = false;
    c.shrinky = true;
    c.fill = GBConstraints.VERTICAL;
    gridbag.setConstraints(super.add(vbar), c);

    c.gridx = 0;
    c.gridy = 1;
    c.shrinkx = true;
    c.shrinky = false;
    c.fill = GBConstraints.HORIZONTAL;
    gridbag.setConstraints(super.add(hbar), c);

    // Have the initial minimumSize include the vertical scrollbar,
    // but not the horizontal scrollbar.
    vshow = true;
    hbar.hide();
  }

  public Component add(Component comp) {
    if (!(comp instanceof Scrollable))
      throw new Error("Can only add an instance of " +
		      "Scrollable to the ScrollPanel");

    if (this.comp != null)
      remove(this.comp);

    this.comp = comp;

    GBConstraints c = new GBConstraints();
    c.gridx = 0;
    c.gridy = 0;
    c.weightx = 1;
    c.weighty = 1;
    c.shrinkx = true;
    c.shrinky = true;
    c.fill = GBConstraints.BOTH;
    c.insets = scrollAreaInsets;
    gridbag.setConstraints(super.add(comp), c);

    return comp;
  }

  public int getScrollAreaWidth() {
    return scrollAreaWidth;
  }

  public void setScrollAreaWidth(int scrollAreaWidth) {
    this.scrollAreaWidth = scrollAreaWidth;
  }

  public int getScrollAreaHeight() {
    return scrollAreaHeight;
  }

  public void setScrollAreaHeight(int scrollAreaHeight) {
    this.scrollAreaHeight = scrollAreaHeight;
  }

  public Insets getScrollAreaInsets() {
    return scrollAreaInsets;
  }

  public void setScrollAreaInsets(Insets insets) {
    scrollAreaInsets = insets;

    if (comp != null) {
      GBConstraints c = gridbag.getConstraints(comp);
      c.insets = insets;
      gridbag.setConstraints(comp, c);
    }
  }

  public int getVisibleIndex() {
    if (!vshow)
      return 0;

    int lineHeight = ((Scrollable)comp).lineHeight();
    int scrolly = vbar.getValue();

    return (scrolly+lineHeight-1)/lineHeight;
  }

  public void makeVisible(int index) {
    if (!vshow)
      return;

    if (index == -1)
      return;

    Scrollable scrollable = (Scrollable)comp;

    Dimension viewSize = scrollable.viewSize();
    int lineHeight = scrollable.lineHeight();
    int y = lineHeight*index;
    int scrolly = vbar.getValue();

    if (y < scrolly) {
      vbar.setValue(y);
      scrollable.scrollY(y);
    }
    else if ((y + lineHeight) > scrolly + viewSize.height) {
      vbar.setValue(y);
      scrollable.scrollY(y);
    }
  }

  public boolean handleEvent(Event e) {
    if (e.target == hbar)
      ((Scrollable)comp).scrollX(hbar.getValue());
    else if (e.target == vbar)
      ((Scrollable)comp).scrollY(vbar.getValue());
    else
      return super.handleEvent(e);

    return true;
  }

  private Dimension insetSize() {
    Dimension size = size();

    // take out our insets
    Insets insets = insets();
    Insets scrollAreaInsets = getScrollAreaInsets();
    if (scrollAreaInsets == null)
      scrollAreaInsets = new Insets(0, 0, 0, 0);

    size.width -= (insets.left + insets.right +
      scrollAreaInsets.left + scrollAreaInsets.right);
    size.height -= (insets.top + insets.bottom +
      scrollAreaInsets.top + scrollAreaInsets.bottom);

    return size;
  }

  // Always leave space for the scrollbars
  public Dimension minimumSize() {
    Dimension compMin;
    Dimension hbarMin = hbar.minimumSize();
    Dimension vbarMin = vbar.minimumSize();

    if (comp != null)
      compMin = comp.minimumSize();
    else
      compMin = new Dimension(0, 0);

    return calcSize(compMin, hbarMin, vbarMin);
  }

  // Always leave space for the scrollbars
  public Dimension preferredSize() {
    Dimension compPref;
    Dimension hbarPref = hbar.preferredSize();
    Dimension vbarPref = vbar.preferredSize();

    if (comp != null)
      compPref = comp.preferredSize();
    else
      compPref = new Dimension(0, 0);

    return calcSize(compPref, hbarPref, vbarPref);
  }

  private Dimension calcSize(Dimension compMin,
			     Dimension hbarMin, Dimension vbarMin) {
    Insets insets = insets();
    Insets scrollAreaInsets = getScrollAreaInsets();
    if (scrollAreaInsets == null)
      scrollAreaInsets = new Insets(0, 0, 0, 0);

    if (scrollAreaWidth != 0)
      compMin.width = scrollAreaWidth;
    if (scrollAreaHeight != 0)
      compMin.height = scrollAreaHeight;

    int insetsWidth = insets.left + insets.right +
      scrollAreaInsets.left + scrollAreaInsets.right;
    compMin.width += (vbarMin.width + insetsWidth);
    compMin.width = Math.max(compMin.width, (hbarMin.width + insetsWidth));

    int insetsHeight = insets.top + insets.bottom +
      scrollAreaInsets.top + scrollAreaInsets.bottom;
    compMin.height += (hbarMin.height + insetsHeight);
    compMin.height = Math.max(compMin.height, (vbarMin.height + insetsHeight));

    return compMin;
  }

  public void layout() {
    Dimension size = insetSize();
    if (size.width <= 0 || size.height <= 0) {
      super.layout();
      return;
    }

    Dimension d;
    Dimension min = ((Scrollable)comp).scrollSize();
    boolean needHShow = false, needVShow = false;
    boolean revalidate = false;

    if (size.width < min.width)
      needHShow = true;
    if (size.height < min.height)
      needVShow = true;

    if ((needHShow || needVShow) && !(needHShow && needVShow)) {
      if (needVShow) {
	d = vbar.minimumSize();
	if (size.width < (min.width + d.width))
	  needHShow = true;
      }
      else if (needHShow) {
	d = hbar.minimumSize();
	if (size.height < (min.height + d.height))
	  needVShow = true;
      }
    }

    if (needHShow) {
      if (!hshow) {
	hbar.show();
	hshow = true;
	revalidate = true;
      }
    }
    else {
      if (hshow) {
	hbar.hide();
	hshow = false;
	revalidate = true;
	((Scrollable)comp).scrollX(0);
      }
    }

    if (needVShow) {
      if (!vshow) {
	vbar.show();
	vshow = true;
	revalidate = true;
      }
    }
    else {
      if (vshow) {
	vbar.hide();
	vshow = false;
	revalidate = true;
	((Scrollable)comp).scrollY(0);
      }
    }

    if (hshow)
      updateHScrollbar();
    if (vshow)
      updateVScrollbar();

    if (revalidate)
      validate();
    else
      super.layout();
  }

  public void updateWindow() {
    invalidate();
    validate();
  }

  protected void updateHScrollbar() {
    Dimension size = insetSize();
    Dimension min = ((Scrollable)comp).scrollSize();

    if (vshow) {
      Dimension d = vbar.minimumSize();
      size.width -= d.width;
    }

    // (size.width-d.width) can become negative during initialization
    if (size.width < 1)
      return;

    hbar.setValues(hbar.getValue(), size.width, 0, min.width-size.width);

    Scrollable scrollable = (Scrollable)comp;
    int lineWidth = Math.max(10, (min.width - size.width)/6);
    int pageSize = Math.max(lineWidth, min.width - size.width);

    hbar.setLineIncrement(lineWidth);
    hbar.setPageIncrement(pageSize);

    scrollable.scrollX(hbar.getValue());
  }

  protected void updateVScrollbar() {
    Dimension size = insetSize();
    Dimension min = ((Scrollable)comp).scrollSize();

    if (hshow) {
      Dimension d = hbar.minimumSize();
      size.height -= d.height;
    }

    // (size.height-d.height) can become negative during initialization
    if (size.height < 1)
      return;

    vbar.setValues(vbar.getValue(), size.height, 0, min.height-size.height);

    Scrollable scrollable = (Scrollable)comp;
    int lineHeight = scrollable.lineHeight();
    int pageSize = Math.max(lineHeight, min.height - size.height - lineHeight);

    vbar.setLineIncrement(lineHeight);
    vbar.setPageIncrement(pageSize);

    scrollable.scrollY(vbar.getValue());
  }
}
