/**
 * ColumnListCanvas.java
 *
 * Copyright 1995-1996 Active Software Inc.
 *
 * @version @(#)ColumnListCanvas.java 1.45 96/08/01
 * @author  Tilman Sporkert
 */

package sunsoft.jws.visual.rt.awt;

import sunsoft.jws.visual.rt.base.DesignerAccess;
import sunsoft.jws.visual.rt.base.Global;

import java.awt.Graphics;
import java.util.Vector;
import java.awt.*;


/**
 * An internal widget for the ColumnList widget
 *
 * @author  Tilman Sporkert
 */
class ColumnListCanvas extends Panel implements Scrollable {
    ColumnList parent;
    Frame      ourFrame = null;
    int        columns;         // # of data columns
                                // there are actual columns + 1 columns managed
                                // but only columns are visible. The last 
                                // column can be accessed only by addItem(),
                                // getItem(), and putItem(), and can be used to
                                // manage a random object together with the 
                                // data row
    int        charWidth;
    int        rowHeight;      // height of data row
    Component  hasComponents = null;
    int        componentHeight = 0; // height of largest component in list
    int        rowAscent;
    int        headerHeight;
    int        totalWidth;     // total width of all colums
    boolean    dragging = false;// for resizing columns
    int        drag_start;      // x position where drag started
    int        drag_column;     // which column is getting resized?

    String[]   headers;
    boolean    showHeaders = true;
    int[]      format = null;   // Formatting mode for columns
    int[]      dimensions;      // the dimensions of the headers
    boolean    autoWidth = true; // automatically expand columns to fit?
    int        records;         // # of active records
    Vector     labels;          // the actual data, as Label[]
    Vector     row_colors;      // Attributes of a record

    int        requested_rows = 5;  // requested # of rows to show
    int        requested_chars = 0; // request # of chars to display horizontally
    int        disp_rows;       // # of rows to show
    int        visibleRows = -1;    // this can be one more than disp_rows

    int        scrollx = 0;
    int        scrollrow = 0;
    int        scrolly = 0;

    // the selection
    boolean    highlight_items = false; // highlight new items?
    Thread     colorThread = null;
    boolean    selectable = false;
    int        selected_row = -1;
    Color      selectColor;

    private final static int COLOR_NONE = 0;
    private final static int COLOR_FIRST = 100;

    /**
     * create a new ColumnListCanvas. Before it can be used
     * in any reasonable fashion, setHeaders() should be called, followed
     * by an optional call to setFormat()!
     *
     * @param parent_in   The parent ColumnList
     */
    public ColumnListCanvas(ColumnList parent_in) {
	setLayout(null);
	parent = parent_in;
	
	headers = new String[0];
	columns = 0;
	
	labels = new Vector();
	row_colors = new Vector();
	
	dimensions = null;
	
	disp_rows = 0;
	records = 0;
	
	selectColor = new Color(0, 0, 128);
    }

    protected void finalize() {
	if (colorThread != null)
	    colorThread.stop();
    }


    /**
     * set the headers for the columns. THis also defines the
     * # of columns. If the # of header columns changes, all current rows
     * will be deleted, and the formatting options get reset
     *
     * @param headers_in  array of column headers
     */
    public void setHeaders(String[] headers_in) {
	int new_columns;
	
	if (headers_in == null)
	    new_columns = 0;
	else
	    new_columns = headers_in.length;
	
	if (new_columns != columns) {
	    // # of columns change -> get rid of everything
	    delItems();
	    labels = new Vector();
	    row_colors = new Vector();
	    format = null;
	    dimensions = null;
	}
	
	if (headers_in == null)
	    columns = 0;
	else
	    columns = headers_in.length;
	headers = headers_in;
	repaint();
    }


    // ///////////////////////////////////////////////////////////////
    // setFormat -- set the column formatting. This is currently ignored, but
    //  should contain things like currency, left_align, right_align etc.
    public void setFormat(int[] format_in) {
	format = format_in;
    }

    
    public void setSelectable(boolean selectable_in) {
	selectable = selectable_in;
    }

    
    public void setHighlightItems(boolean highlight_items_in) {
	highlight_items = highlight_items_in;
    }

    public void setShowHeaders(boolean showHeaders) {
	this.showHeaders = showHeaders;
	repaint();
    }

    public void setAutoWidth(boolean autoWidth) {
	this.autoWidth = autoWidth;
    }

    // Scrollable
    public void scrollX(int x) {
	scrollx = x;
	repaint();
    }
    
    public void scrollY(int y) {
	scrolly = y;
	if (rowHeight != 0)
	    scrollrow = (y + rowHeight - 1) / rowHeight;
	else
	    scrollrow = 0;
	repaint();
    }
    
    public Dimension scrollSize() {
	cacheDimensions();
	if (dimensions == null)
	    return new Dimension(100, 100);
	else
	    return new Dimension(totalWidth,
				 headerHeight + rowHeight * records);
    }
    
    public Dimension viewSize() {
	cacheDimensions();
	Dimension size = size();
	size.height -= headerHeight;
	return size;
    }
    
    public int lineHeight() {
	cacheDimensions();
	return rowHeight;
    }
    
    private void add_more_labels() {
	int c;
	// have one extra for the "hidden" object
	Object[] new_labels = new Object[columns + 1];
	labels.addElement(new_labels);
	row_colors.addElement(new Integer(COLOR_NONE));
	for (c = 0; c < columns; c++) 
	    new_labels[c] = new String("");
	new_labels[columns] = null;
    }
    
    public synchronized void insertItem(Object[] values, int row) {
	int c;
	boolean newWidth = false;
	
	// make sure we have enough labels...
	while (row >= labels.size())
	    add_more_labels();

	// initialize all intermediate records
	while (row > records) {
	    Object[] label_row = (Object[]) labels.elementAt(records);
	    for (c = 0; c < columns; c++) 
		label_row[c] = new String("");
	    label_row[columns] = null;
	    records++;
	}

        // Make one extra entry
        add_more_labels(); // Make one extra
	Object[] label_row = (Object[]) labels.elementAt(records);
	for (c = 0; c < columns; c++) 
	    label_row[c] = new String("");
	label_row[columns] = null;
	records++;

        // Move other rows out of the way
        for(int pos=records-2; row<=pos; --pos) {
            labels.setElementAt(labels.elementAt(pos),pos+1);
            row_colors.setElementAt(row_colors.elementAt(pos),pos+1);
        }
        label_row = new Object[columns+1];
	for (c = 0; c < columns; c++) 
	    label_row[c] = new String("");
	label_row[columns] = null;
        labels.setElementAt(label_row,row);

        // Fix selected row
        if (selected_row >= row)
            ++selected_row;

        // Call addItem to replace things
        addItem(values,row);
    }
    
    public synchronized void addItem(Object[] values) {
	addItem(values, records);
    }
    
    public synchronized void addItem(Object[] values, int row) {
	int c;
	boolean newWidth = false;
	
	// make sure we have enough labels...
	while (row >= labels.size())
	    add_more_labels();

	// initialize all intermediate recors
	while (row > records) {
	    Object[] label_row = (Object[]) labels.elementAt(records);
	    for (c = 0; c < columns; c++) 
		label_row[c] = new String("");
	    label_row[columns] = null;
	    records++;
	}
	
	if (autoWidth)
	    cacheDimensions();
	int usableWidth = -3;
	Object[] label_row = (Object[]) labels.elementAt(row);
	for (c = 0; c < columns; c++) {
	    label_row[c] = values[c];
	    if (label_row[c] == null)
		continue;
	    if (autoWidth && (dimensions != null))
		usableWidth = dimensions[c] - 2;
	    if (label_row[c] instanceof Component) {
		if (hasComponents == null)
		    hasComponents = (Component) label_row[c];
		add((java.awt.Component) label_row[c]);
		((java.awt.Component) label_row[c]).hide();
		Dimension size =
		    ((java.awt.Component) label_row[c]).preferredSize();
		if (size.height > componentHeight) {
		    componentHeight = size.height;
		    if (componentHeight > rowHeight) {
			rowAscent += (componentHeight - rowHeight) / 2;
			rowHeight = componentHeight;
		    }
		}
		if ((usableWidth != -3) && (size.width > usableWidth)) {
		    totalWidth += size.width - usableWidth;
		    dimensions[c] += size.width - usableWidth;
		    newWidth = true;
		}
	    } else if (usableWidth != -3) {
		String val;
		if (label_row[c] instanceof String)
		    val = (String) label_row[c];
		else
		    val = label_row[c].toString();
		getGraphics().setFont(getFont());
		FontMetrics fm = getGraphics().getFontMetrics();
		int w = fm.stringWidth(val) + 6;
		if (label_row[c] instanceof CLCheckbox)
		    w += rowHeight;
		if (w > usableWidth) {
		    totalWidth += w - usableWidth;
		    dimensions[c] += w - usableWidth;
		    newWidth = true;
		}
		
	    }
	}
	
	if (values.length > columns)
	    label_row[columns] = values[columns];
	else
	    label_row[columns] = null;
	
	if (highlight_items == true) {
	    row_colors.setElementAt(new Integer(COLOR_FIRST), row);
	    if (colorThread == null) {
		colorThread = new ColumnListThread(this);
		colorThread.start();
	    } else
		colorThread.resume();
	}
	else
	    row_colors.setElementAt(new Integer(COLOR_NONE), row);
	if (row >= records)
	    records++;
	/*
	if (row_visible(row) || newWidth)
	    repaint();
	    */
    }
    
    public synchronized void delItems() {
	delItems(0, records - 1);
	selected_row = -1;
    }
    
    public synchronized void delItems(int start, int end) {
	if ((start < 0) || (start >= records))
	    return;
	if (end < start)
	    return;
	if (end > records)
	    end = records - 1;
	int r, c;
	int diff = end - start + 1;
	for (r = start; r <= end; r++) {
	    Object[] label_row = (Object[]) labels.elementAt(start);
	    for (c = 0; c < columns; c++) {
		if (label_row[c] instanceof Component)
		    remove((java.awt.Component) label_row[c]);
	    }
	    
	    labels.removeElementAt(start);
	    row_colors.removeElementAt(start);
	}
	records -= diff;
	
	if (selected_row > end)
	    selected_row -= diff;
	else if (selected_row > start)
	    selected_row = start - 1;
	
	// repaint all the time... could be optimized
	repaint();
    }
    
    
    protected Object getItem(int row, int column) {
	if ((row > records) || (row < 0) || (column > columns) || (column < 0))
	    return null;
	return ((Object []) labels.elementAt(row))[column];
    }


    protected boolean putItem(int row, int column, Object value) {
	if ((row > records) || (row < 0) || (column > columns) || (column < 0))
	    return false;
	Object[] data = (Object []) labels.elementAt(row);
	data[column] = value;
	repaint();
	return true;
    }
    
    
    protected boolean swapItems(int row1, int row2) {
	if ((row1 > records) || (row1 < 0) || (row2 > records) || (row2 < 0))
	    return false;
	if (row1 == row2)
	    return true;
	Object[] data1 = (Object []) labels.elementAt(row1);
	Object[] data2 = (Object []) labels.elementAt(row2);
	labels.setElementAt(data2, row1);
	labels.setElementAt(data1, row2);
	repaint();
	return true;
    }
    
    
    
    // drawString - calls g.drawString(str, x, y), but makes sure that str
    //              will fit into width
    private void drawString(Graphics g, String str, int x, int y, int width) {
        g.setFont(getFont());
	FontMetrics fm = g.getFontMetrics();
	int w = fm.stringWidth(str);
	if (w <= width) {
	    g.drawString(str, x, y);
	    return;
	}

	// cut off chars at the end to make the string fit..
	
	// do a best guess...
	int l = width * str.length() / w;
	String sub_str = str.substring(0, l);
	w = fm.stringWidth(sub_str);
	
	// now make some minor adjustments...
	if (w > width) {
	while (w > width) {
	    l--;
	    sub_str = str.substring(0, l);
	    w = fm.stringWidth(sub_str);
	}
    } else if (w < width) {
	while (w < width) {
	    l++;
	    sub_str = str.substring(0, l);
	    w = fm.stringWidth(sub_str);
	}
	sub_str = str.substring(0, l - 1);
    }
    g.drawString(sub_str, x, y);
}

public void setFont(Font font) {
    super.setFont(font);
    dimensions = null;
    cacheDimensions();
}
    

public void update(Graphics g) {
    paint(g);
}

private void cacheDimensions() {
    if (dimensions == null) {
        Graphics g = getGraphics();
	if (g != null) {
	    g.setFont(getFont());
	    FontMetrics fm = g.getFontMetrics();
	    dimensions = new int[(columns == 0) ? 1 : columns];
	    totalWidth = 0;
	    int c;
	    for (c = 0; c < columns; c++) {
		dimensions[c] = fm.stringWidth(headers[c]) + 10;
		totalWidth += dimensions[c];
	    }
	    // a row is a label without PanelFrame
	    rowHeight = fm.getHeight() + 2;
	    rowAscent = fm.getAscent() + 1;
	    charWidth = fm.charWidth('X');

	    if ((componentHeight == 0) && (hasComponents != null))
		componentHeight = hasComponents.preferredSize().height;

	    if (rowHeight < componentHeight) {
		rowAscent += (componentHeight - rowHeight) / 2;
		rowHeight = componentHeight;
	    }
	    if (showHeaders)
		headerHeight = rowHeight + 6;
	    else
		headerHeight = 0;
	}
    }
}

    
public void paint(Graphics g) {

    synchronized (DesignerAccess.mutex) {
	
	// first time? get the dimensions of the header panel
	cacheDimensions();
	
	// how many rows fit?
	Dimension canvas_size = size();
	int usable_height = canvas_size.height - headerHeight;
	int usable_width = canvas_size.width;
	disp_rows = usable_height / rowHeight;

	// visibleRows can be one more than disp_rows (last row partially visible)
	visibleRows = (usable_height + rowHeight - 1) / rowHeight;
	  // make sure we have enough labels...
	int hor_offset = scrollx;
	int vert_offset = scrollrow;
	while ((visibleRows + vert_offset) > labels.size()) 
	    add_more_labels();

	g.setColor(getForeground());
	Color orig_color = g.getColor();

	// draw the column headers
	int x = 0;
	int col_width;
	if (showHeaders) {
	    for (int c = 0; c <= columns; c++) {
		
		if (c == columns)
		    // last column fills up the rest
		    col_width = usable_width - x;
		else
		    col_width = dimensions[c];
		
		int left = x - hor_offset;
		int right = left + col_width - 1;
		int bottom = headerHeight - 1;
		
		// is the column for real, and some of it visible?
		if ((col_width > 0) && ((right > 0) || (left < usable_width))) {
		    // now draw the thing...
		    g.setColor(getBackground());
		    g.drawLine(left, 0, right, 0); // top
		    g.drawLine(left, 0, left, bottom); // left
		    
		    g.setColor(getBackground().darker().darker());
		    g.drawLine(left, bottom, right, bottom); // bottom
		    g.drawLine(right, 0, right, bottom); // right
		    
		    g.setColor(getBackground().darker());
		    g.fillRect(left + 1, 1, col_width - 2, headerHeight - 2);
		    
		    g.setColor(orig_color);
		    
		    // last column has no strings
		    if (c != columns)
			drawString(g, headers[c], left + 3,
				   headerHeight - 5, col_width - 2);
		}
		
		if (c != columns)
		    x += dimensions[c];
	    }
	}
        
	// now paint the data
	int y = headerHeight;
	for (int r = 0; r < labels.size(); r++) {
	    if ((r >= vert_offset) && (r < (visibleRows + vert_offset))) {
		x = 0;
		Object[] row = (Object[]) labels.elementAt(r);
		int row_color = ((Integer) row_colors.elementAt(r)).intValue();

		Color stringForeground =
		    getStringForeground(orig_color, r, row_color);
		Color stringBackground =
		    getStringBackground(r, row_color);

		g.setColor(stringBackground);
		g.fillRect(0, y, usable_width, rowHeight);
		g.setColor(stringForeground);
		
		for (int c = 0; c < columns; c++) {
		    col_width = dimensions[c];
		    int left = x - hor_offset;
		    int right = left + col_width - 1;
		    
		    // is the column for real, and some of it visible?
		    if ((col_width > 0) &&
			((right > 0) || (left < usable_width))) {
			if (row[c] != null)
			    if (row[c] instanceof String) 
				drawString(g, (String) row[c], left + 2,
					   y + rowAscent, col_width - 2);
			    else if (row[c] instanceof CLCheckbox) {
				CLCheckbox box = (CLCheckbox) row[c];
				g.drawRect(left + 3, y + 1,
					   rowHeight - 5, rowHeight - 5);
				if (box.getState()) {
				    g.drawLine(left + 5,
					       y + rowHeight / 2 - 2,
					       left + rowHeight / 2,
					       y + rowHeight - 7);
				    g.drawLine(left + rowHeight / 2,
					       y + rowHeight - 7, 
					       left + rowHeight - 4,
					       y + 4);
				    g.drawLine(left + 5,
					       y + rowHeight / 2 - 1,
					       left + rowHeight / 2,
					       y + rowHeight - 6);
				    g.drawLine(left + rowHeight / 2,
					       y + rowHeight - 6, 
					       left + rowHeight - 4,
					       y + 5);
				}
				drawString(g, box.getText(),
					   left + rowHeight + 2,
					   y + rowAscent,
					   col_width - rowHeight - 2);
			    } else if (row[c] instanceof Component) {
				Component cb = (Component) row[c];
				Dimension size = cb.preferredSize();
				if (size.width > (col_width - 2))
				    size.width = col_width - 2;
				cb.reshape(left + 2, y,
					   size.width, rowHeight);
				cb.setBackground(stringBackground);
				cb.setForeground(stringForeground);
				cb.show();
				// cb.validate();
			    } else 
				drawString(g, row[c].toString(), left + 2,
					   y + rowAscent, col_width - 2);
		    } else {
			if (row[c] != null)
			    if (row[c] instanceof Component) {
				Component cb = (Component) row[c];
				cb.hide();
			    }
		    }
		    x += dimensions[c];
		}
		y += rowHeight;
		g.setColor(orig_color);
	    } else {
		Object[] row = (Object[]) labels.elementAt(r);
		for (int c = 0; c < row.length; c++)
		    if (row[c] instanceof Component)
			((Component) row[c]).hide();
	    }
	} // for
	
    } // synchronized
}

    private Color getStringForeground(Color orig_color,
				      int row, int row_color) {
      Color bg = getBackground();
      Color stringForeground = orig_color;

      if (row == selected_row) {
	if (row_color == COLOR_NONE) {
	  stringForeground = getBackground();
	}
	else {
	  if (bg.equals(Color.white)) {
	    stringForeground =
	      new Color(Color.HSBtoRGB((float) 0.075,
				       (float) (row_color / 100.0),
				       1));
	  }
	  else {
	    stringForeground =
	      new Color(calcRGB(bg.getRed(), row_color),
			calcRGB(bg.getGreen(), row_color),
			calcRGB(bg.getBlue(), row_color));
	  }
	}
      }

      return stringForeground;
    }

    private Color getStringBackground(int row, int row_color) {
      Color bg = getBackground();
      Color stringBackground;

      if (row == selected_row) {
	stringBackground = selectColor;
      }
      else if (row_color != COLOR_NONE) {
	if (bg.equals(Color.white)) {
	  stringBackground = 
	    new Color(Color.HSBtoRGB((float) 0.075,
				     (float) (row_color / 100.0),
				     1));
	}
	else {
	  stringBackground =
	    new Color(calcRGB(bg.getRed(), row_color),
		      calcRGB(bg.getGreen(), row_color),
		      calcRGB(bg.getBlue(), row_color));
	}
      }
      else {
	stringBackground = getBackground();
      }

      return stringBackground;
    }

    private int calcRGB(int color, int index) {
      return (color + (((255 - color) * index) / 100));
    }

    private boolean row_visible(int row) {
	return ((row >= scrollrow) &&
		(row < (scrollrow + visibleRows))) ;
    }
	

    public void updateView() {
	repaint();
    }


    /**
     * @see sunsoft.jws.visual.designer.gui.ColumnList
     */
    public void setDisplayRows(int rows) {
	requested_rows = rows;
    }


    public void setVisibleChars(int chars) {
	requested_chars = chars;
    }


    /**
     * the minimum width is enough to fully display the first column, but not
     * more than 200 pixels. The minimum height is the height to display the
     * number of rows requested with setDisplayRows() (default is 3)
     *
     * @return       Dimension
     */
    public Dimension minimumSize() {
	return preferredSize();
	/*
	cacheDimensions();
	if (dimensions == null)
	    return new Dimension(0, 0);
	else {
	    int w = Math.min(dimensions[0], 200);
	    return new Dimension(w, headerHeight + rowHeight * requested_rows);
	}
	*/
    }
    
    public Dimension preferredSize() {
	cacheDimensions();
	if (dimensions == null)
	    return new Dimension(100, 100);
	else
	    if (requested_chars == 0)
		return new Dimension(totalWidth,
				     headerHeight + rowHeight * requested_rows);
	    else
		return new Dimension(requested_chars * charWidth,
				     headerHeight + rowHeight * requested_rows);
    }

    /**
     * highlight row, and make it visible
     * only one row can be highlighed at a time - previously highlighted
     * rows will be restored to normal
     * to remove the highlighting, set the highlighted row to -1
     *
     * @param row    index of row
     * @return       actual index of highlighted row, -1 if none
     */
    public int highlight(int row) {
	if (selected_row != row) {
	    int old_selected_row = selected_row;
	    if (row < records)
		selected_row = row;
	    else
		selected_row = -1;
	    
	    // make sure we can see the new selection
	    // Tilman 05/21: Not tested if the makeVisible() call works
	    if ((selected_row != -1) && (!row_visible(selected_row))) 
		parent.makeVisible(selected_row);

	    // repaint if anything is visible
	    if (row_visible(old_selected_row) || row_visible(selected_row))
		repaint();
	}
	return selected_row;
    }


    protected int getRowY(int row) {
	if ((dimensions == null) || (visibleRows == -1))
	    return -2;
	if (row_visible(row)) {
	    return headerHeight + rowHeight * (row - scrollrow);
	} else
	    return -1;
    }
    
    
    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    private synchronized void start_drag(int start_x) {
	int x = -scrollx;
	for (int c = 0; c < columns; c++) {
	    x += dimensions[c];
	    if ((start_x >= (x - 5)) && (start_x <= (x + 5))) {
		drag_column = c;
		dragging = true;
		drag_start = start_x;
		break;
	    }
	}
    }


   /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    private synchronized int mouseColumn(int mouseX) {
	int x = -scrollx;
	if (mouseX < x)
	    return -1;
	for (int c = 0; c < columns; c++) {
	    x += dimensions[c];
	    if (mouseX < x)
		return c;
	}
	return -1;
    }


    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    private synchronized void drag_column(int drag_x) {
	int diff = drag_x - drag_start;
	if (dimensions[drag_column] + diff >= 5) {
	    dimensions[drag_column] += diff;
	    totalWidth += diff;
	    drag_start = drag_x;
	    repaint();
	}
    }

    private int prevX = 0;
    private int prevY = 0;

    public boolean mouseDown(Event e, int x, int y) {
	// Workaround for windows bug where the click count is set to 2, even
	// if the two clicks were separated by a lot of distance.
	if (Global.isWindows()) {
	    if (Math.abs(prevX - x) + Math.abs(prevY - y) > 4)
		e.clickCount = 1;
	    
	    prevX = x;
	    prevY = y;
	}
	if (y > headerHeight) {
	    int row = (y - headerHeight - 1) / rowHeight + scrollrow;
	    if (row < records) {
		int column = mouseColumn(x);
		if (column != -1) {
		    Object entry = ((Object[] ) labels.elementAt(row))[column];
		    if ((entry instanceof CLCheckbox) &&
			(mouseColumn(x - rowHeight) != column)) {
			CLCheckbox box = (CLCheckbox) entry;
			box.setState(!box.getState());
			repaint();
			parent.postEvent(new Event(box, Event.ACTION_EVENT,
						   new Boolean(box.getState())));
			return true;
		    }
		}
		if (selectable) {
		    row = highlight(row);
		    if (e.clickCount == 1)
			parent.postEvent(new Event(parent, Event.LIST_SELECT,
						   new Integer(row)));
		    else if (e.clickCount == 2)
			parent.postEvent(new Event(parent, Event.ACTION_EVENT,
						   new Integer(row)));
		}
	    }
	} else if (y < headerHeight) {
	    start_drag(x);
	}
	return true;
    }


    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public boolean mouseDrag(Event e, int x, int y) {
	if (dragging) {
	    drag_column(x);
	    return true;
	}
	return false;
    }


    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public boolean mouseUp(Event e, int x, int y) {
	if (dragging) {
	    dragging = false;
	    parent.updateView();
	    return true;
	}
	return false;
    }

    boolean resizeCursor = false;
    boolean checkedFrame = false;

    private Frame getFrame() {
	if (ourFrame == null) {
	    if (!checkedFrame) {
		checkedFrame = true;
		Component c = parent;
		while (c != null) {
		    if (c instanceof Frame) {
			ourFrame = (Frame) c;
			break;
		    } else
			c = c.getParent();
		}
	    }
	}
	return ourFrame;
    }
    
    
    public boolean mouseMove(Event e, int x, int y) {
	if (getFrame() != null) {
	    boolean resizable = false;
	    
	    if (y < headerHeight) {
		int x1 = -scrollx;
		for (int c = 0; c < columns; c++) {
		    x1 += dimensions[c];
		    if ((x >= (x1 - 5)) && (x <= (x1 + 5))) {
			resizable = true;
			break;
		    }
		}
	    }
	    
	    if (resizable != resizeCursor) {
		if (resizable)
		    ourFrame.setCursor(Frame.E_RESIZE_CURSOR);
		else
		    ourFrame.setCursor(Frame.DEFAULT_CURSOR);
		
		resizeCursor = resizable;
	    }
	    return true;
	}
	else
	    return false;
    }
	    

    /**
     * restore the cursor when exiting 
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public boolean mouseExit(Event e, int x, int y) {
	if (resizeCursor) {
	    ourFrame.setCursor(Frame.DEFAULT_CURSOR);
	    resizeCursor = false;
	}
	return false;
    }


    /**
     * ...
     *
     * @param ...    ...
     * @return       ...
     * @exception    ...
     */
    public synchronized boolean update_row_colors() {
	int r;
	boolean is_changing = false;
	for (r = 0; r < records; r++) {
	    int row_color = ((Integer) row_colors.elementAt(r)).intValue();
	    if (row_color != COLOR_NONE) {
		is_changing = true;
		row_colors.setElementAt(new Integer(row_color - 4), r);
	    }
	}
	if (is_changing) {
	repaint();
	}
	return is_changing;
    }


}
