/*
 * Decompiled with CFR 0.152.
 */
package xtrememp.playlist;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import xtrememp.playlist.PlaylistEvent;
import xtrememp.playlist.PlaylistItem;
import xtrememp.playlist.PlaylistListener;
import xtrememp.playlist.filter.Predicate;
import xtrememp.playlist.filter.TruePredicate;

public class Playlist {
    private static Logger logger = LoggerFactory.getLogger(Playlist.class);
    protected final List<PlaylistItem> cachedPlaylist;
    protected final List<PlaylistItem> filteredPlaylist;
    protected final List<PlaylistItem> shuffledList;
    protected final List<PlaylistListener> listeners;
    protected final Random rnd;
    protected Predicate<PlaylistItem> filterPredicate;
    protected PlayMode playMode = PlayMode.REPEAT_ALL;
    protected PlaylistItem cursor;
    protected int shuffledIndex = -1;
    protected boolean isModified = false;

    public Playlist() {
        this.cachedPlaylist = new ArrayList<PlaylistItem>();
        this.filteredPlaylist = new ArrayList<PlaylistItem>();
        this.shuffledList = new ArrayList<PlaylistItem>();
        this.listeners = new ArrayList<PlaylistListener>();
        this.rnd = new Random();
        this.filterPredicate = TruePredicate.getInstance();
    }

    public PlayMode getPlayMode() {
        return this.playMode;
    }

    public void setPlayMode(PlayMode playMode) {
        this.playMode = playMode;
        if (playMode == PlayMode.SHUFFLE) {
            int size = this.size();
            switch (size) {
                case 1: {
                    this.shuffledList.add(this.filteredPlaylist.get(0));
                    break;
                }
                default: {
                    for (int i = 0; i < size; ++i) {
                        this.shuffledList.add(this.rnd.nextInt(this.shuffledList.size() + 1), this.filteredPlaylist.get(i));
                    }
                    int cursorShuffledIndex = this.shuffledList.indexOf(this.cursor);
                    if (cursorShuffledIndex == -1) break;
                    Collections.swap(this.shuffledList, 0, cursorShuffledIndex);
                }
            }
            this.shuffledIndex = 0;
        } else {
            this.shuffledList.clear();
            this.shuffledIndex = -1;
        }
        this.firePlayModeChangedEvent();
    }

    public boolean addItem(PlaylistItem item) {
        boolean added = this.cachedPlaylist.add(item);
        if (this.filterPredicate.evaluate(item)) {
            this.filteredPlaylist.add(item);
        }
        this.setModified(added);
        if (this.playMode == PlayMode.SHUFFLE) {
            this.addToShuffledList(item);
        }
        this.fireItemAddedEvent(item);
        return added;
    }

    public void addItemAt(int pos, PlaylistItem item) {
        this.cachedPlaylist.add(pos, item);
        if (this.filterPredicate.evaluate(item)) {
            this.filteredPlaylist.add(pos, item);
        }
        this.setModified(item != null);
        if (this.playMode == PlayMode.SHUFFLE) {
            this.addToShuffledList(item);
        }
        this.fireItemAddedEvent(item);
    }

    public boolean addAll(Collection<? extends PlaylistItem> c) {
        boolean added = this.cachedPlaylist.addAll(c);
        for (PlaylistItem playlistItem : c) {
            if (!this.filterPredicate.evaluate(playlistItem)) continue;
            this.filteredPlaylist.add(playlistItem);
        }
        this.setModified(added);
        if (this.playMode == PlayMode.SHUFFLE) {
            for (PlaylistItem playlistItem : c) {
                this.addToShuffledList(playlistItem);
            }
        }
        for (PlaylistItem playlistItem : c) {
            this.fireItemAddedEvent(playlistItem);
        }
        return added;
    }

    public boolean removeItem(PlaylistItem item) {
        boolean removed = this.cachedPlaylist.remove(item);
        this.filteredPlaylist.remove(item);
        this.setModified(removed);
        if (this.playMode == PlayMode.SHUFFLE) {
            this.removeFromShuffledList(item);
        }
        this.fireItemRemovedEvent(item);
        return removed;
    }

    public PlaylistItem removeItemAt(int pos) {
        PlaylistItem item = this.cachedPlaylist.remove(pos);
        this.filteredPlaylist.remove(item);
        this.setModified(item != null);
        if (this.playMode == PlayMode.SHUFFLE) {
            this.removeFromShuffledList(item);
        }
        this.fireItemRemovedEvent(item);
        return item;
    }

    public boolean removeAll(Collection<? extends PlaylistItem> c) {
        boolean removed = this.cachedPlaylist.removeAll(c);
        this.filteredPlaylist.removeAll(c);
        this.setModified(removed);
        if (this.playMode == PlayMode.SHUFFLE) {
            for (PlaylistItem playlistItem : c) {
                this.removeFromShuffledList(playlistItem);
            }
        }
        for (PlaylistItem playlistItem : c) {
            this.fireItemRemovedEvent(playlistItem);
        }
        return removed;
    }

    public void clear() {
        Iterator<PlaylistItem> iterator = this.cachedPlaylist.iterator();
        while (iterator.hasNext()) {
            PlaylistItem item = iterator.next();
            iterator.remove();
            this.fireItemRemovedEvent(item);
        }
        this.filteredPlaylist.clear();
        this.shuffledList.clear();
        this.shuffledIndex = 0;
        this.begin();
    }

    public void sort(Comparator<PlaylistItem> comparator) {
        Collections.sort(this.cachedPlaylist, comparator);
        Collections.sort(this.filteredPlaylist, comparator);
        this.setModified(true);
    }

    public void filter(Predicate<PlaylistItem> filterPredicate) {
        this.filterPredicate = filterPredicate;
        this.filteredPlaylist.clear();
        if (this.playMode == PlayMode.SHUFFLE) {
            this.shuffledList.clear();
            this.shuffledIndex = 0;
        }
        for (PlaylistItem pli : this.cachedPlaylist) {
            if (!filterPredicate.evaluate(pli)) continue;
            this.filteredPlaylist.add(pli);
            if (this.playMode != PlayMode.SHUFFLE) continue;
            this.addToShuffledList(pli);
        }
        this.setModified(true);
    }

    public boolean isFiltered() {
        return this.filterPredicate != TruePredicate.getInstance();
    }

    public void moveItem(int fromPos, int toPos) {
        int newIndex = this.cachedPlaylist.indexOf(this.filteredPlaylist.get(toPos));
        PlaylistItem pli = this.filteredPlaylist.remove(fromPos);
        if (pli != null) {
            this.filteredPlaylist.add(toPos, pli);
            if (this.cachedPlaylist.remove(pli)) {
                this.cachedPlaylist.add(newIndex, pli);
            }
        }
        this.setModified(true);
    }

    public void randomize() {
        Collections.shuffle(this.cachedPlaylist);
        Collections.shuffle(this.filteredPlaylist);
        this.setModified(true);
    }

    public void reverse() {
        Collections.reverse(this.cachedPlaylist);
        Collections.reverse(this.filteredPlaylist);
        this.setModified(true);
    }

    public void begin() {
        this.cursor = null;
        if (!this.filteredPlaylist.isEmpty()) {
            this.cursor = this.playMode == PlayMode.SHUFFLE ? this.getShuffledCursor(true) : this.filteredPlaylist.get(0);
        }
        this.setModified(true);
    }

    public PlaylistItem getItemAt(int pos) {
        return this.filteredPlaylist.get(pos);
    }

    public List<PlaylistItem> listAllItems() {
        return this.cachedPlaylist;
    }

    public List<PlaylistItem> listItems() {
        return this.filteredPlaylist;
    }

    public int size() {
        return this.filteredPlaylist.size();
    }

    public void nextCursor() {
        if (this.playMode == PlayMode.SHUFFLE) {
            this.cursor = this.getShuffledCursor(true);
        } else {
            int cursorPos = this.getCursorPosition();
            if (++cursorPos > this.size() - 1) {
                cursorPos = 0;
            }
            this.cursor = this.getItemAt(cursorPos);
        }
    }

    public void previousCursor() {
        if (this.playMode == PlayMode.SHUFFLE) {
            this.cursor = this.getShuffledCursor(false);
        } else {
            int cursorPos = this.getCursorPosition();
            if (--cursorPos < 0) {
                cursorPos = this.size() - 1;
            }
            this.cursor = this.getItemAt(cursorPos);
        }
    }

    public void setModified(boolean flag) {
        this.isModified = flag;
    }

    public PlaylistItem getCursor() {
        return this.cursor;
    }

    public void setCursor(PlaylistItem newCursor) {
        this.cursor = newCursor;
        if (this.playMode == PlayMode.SHUFFLE && this.cursor != null && this.shuffledList.size() > 1 && this.shuffledList.get(this.shuffledIndex) != this.cursor) {
            this.shuffledIndex = ++this.shuffledIndex > this.shuffledList.size() - 1 ? 0 : this.shuffledIndex;
            Collections.swap(this.shuffledList, this.shuffledIndex, this.shuffledList.indexOf(this.cursor));
        }
    }

    public int getCursorPosition() {
        return this.indexOf(this.cursor);
    }

    public void setCursorPosition(int pos) {
        this.cursor = this.getItemAt(pos);
    }

    private void addToShuffledList(PlaylistItem item) {
        int randomIndex = this.shuffledIndex + this.rnd.nextInt(this.shuffledList.size() - this.shuffledIndex + 1);
        this.shuffledList.add(randomIndex, item);
    }

    private void removeFromShuffledList(PlaylistItem item) {
        int maxIndex;
        this.shuffledList.remove(item);
        this.shuffledIndex = this.shuffledList.isEmpty() ? 0 : (this.shuffledIndex > (maxIndex = this.shuffledList.size() - 1) ? maxIndex : this.shuffledIndex);
    }

    private PlaylistItem getShuffledCursor(boolean next) {
        if (!this.shuffledList.isEmpty()) {
            int n = next ? (++this.shuffledIndex > this.shuffledList.size() - 1 ? 0 : this.shuffledIndex) : (--this.shuffledIndex < 0 ? this.shuffledList.size() - 1 : this.shuffledIndex);
            this.shuffledIndex = n;
            return this.shuffledList.get(this.shuffledIndex);
        }
        return null;
    }

    public int indexOf(PlaylistItem item) {
        return this.filteredPlaylist.indexOf(item);
    }

    public boolean isModified() {
        return this.isModified;
    }

    public boolean isEmpty() {
        return this.filteredPlaylist.isEmpty();
    }

    public void addPlaylistListener(PlaylistListener listener) {
        if (listener == null) {
            return;
        }
        this.listeners.add(listener);
        logger.info("Playlist listener added");
    }

    public void removePlaylistListener(PlaylistListener listener) {
        if (listener == null) {
            return;
        }
        this.listeners.remove(listener);
        logger.info("Playlist listener removed");
    }

    private void fireItemAddedEvent(PlaylistItem item) {
        PlaylistEvent event = new PlaylistEvent(this, item);
        for (PlaylistListener listener : this.listeners) {
            listener.playlistItemAdded(event);
        }
        logger.info("Playlist item added: {}", (Object)item);
    }

    private void fireItemRemovedEvent(PlaylistItem item) {
        PlaylistEvent event = new PlaylistEvent(this, item);
        for (PlaylistListener listener : this.listeners) {
            listener.playlistItemRemoved(event);
        }
        logger.info("Playlist item removed: {}", (Object)item);
    }

    private void firePlayModeChangedEvent() {
        PlaylistEvent event = new PlaylistEvent(this);
        for (PlaylistListener listener : this.listeners) {
            listener.playModeChanged(event);
        }
        logger.info("Play mode changed: {}", (Object)this.playMode);
    }

    public static enum PlayMode {
        REPEAT_NONE,
        REPEAT_ONE,
        REPEAT_ALL,
        SHUFFLE;

    }
}

