/* ----------------------------------------------------------------------------
   The Kiwi Toolkit
   Copyright (C) 1998-99 Mark A. Lindner

   This file is part of Kiwi.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
   The author may be contacted at:
   
   frenzy@ix.netcom.com
   ----------------------------------------------------------------------------

   $Log: LocaleData.java,v $
   Revision 1.3  1999/06/28 08:19:19  markl
   Added another form of getMessage().

   Revision 1.2  1999/04/19 05:31:53  markl
   New I18N support.

   Revision 1.1  1999/04/18 10:26:12  markl
   Initial revision

   ----------------------------------------------------------------------------
*/

package kiwi.util;

import java.io.*;
import java.text.*;
import java.util.*;

/** A specialization of <code>PropertyResourceBundle</code> that adds some
 * convenience methods.
 *
 * @author Mark Lindner
 * @author PING Software Group
 * @version 1.0 (04/99)
 */

public class LocaleData extends PropertyResourceBundle
  {
  /** The default message list delimiter. */
  public static final String DEFAULT_DELIMITER = ",";

  private final Object unitArray[] = new Object[1];

  /** Construct a new <code>LocaleData</code> object from the given input
   * stream.
   *
   * @param instream The stream to read the data from.
   * @exception java.io.IOException If an error occurred while reading from
   * the stream.
   */
  
  public LocaleData(InputStream instream) throws IOException
    {
    super(instream);
    }

  /** Get a message for the specified key.
   *
   * @param key The key.
   * @exception java.util.MissingResourceException If the specified key was not
   * found.
   * @return A message for the specified key.
   */
  
  public String getMessage(String key) throws MissingResourceException
    {
    return(getString(key));
    }

  /** Get a message for the specified key, and format the message, substituting
   * the specified arguments for the message's placeholders. Messages may have
   * placeholders of the form {n}, where n is a non-negative integer. For
   * example, the message <tt>"My name is {0}, and I am {1} years old."</tt>
   * and argument list <code>{ "Joe", new Integer(12) }</code> would be
   * formatted as <tt>My name is Joe, and I am 12 years old.</tt>
   *
   * @param key The key.
   * @param args An array of arguments for the message.
   * @exception java.util.MissingResourceException If the specified key was not
   * found.
   * @return A fromatted message for the specified key.
   */
  
  public String getMessage(String key, Object args[])
    {
    return(MessageFormat.format(getMessage(key), args));
    }

  /** Get a message for the specified key, and format the message, substituting
   * the specified argument for the message's first placeholder. Messages may
   * have* placeholders of the form {n}, where n is a non-negative integer. For
   * example, the message <tt>"My name is {0}"</tt> and argument
   * <code>"Joe"</code> would be formatted as <tt>My name is Joe.</tt>
   *
   * @param key The key.
   * @param args An array of arguments for the message.
   * @exception java.util.MissingResourceException If the specified key was not
   * found.
   * @return A fromatted message for the specified key.
   */
  
  public String getMessage(String key, Object arg)
    {
    // Reuse a single array so we don't waste heap space.
    
    synchronized(unitArray)
      {
      unitArray[0] = arg;
      return(getMessage(key, unitArray));
      }
    }
  
  /** Get a message list for the specified key. Retrieves a message for the
   * specified key, and breaks the message on the default delimiter (",")
   * constructing an array in the process.
   *
   * @param key The key.
   * @exception java.util.MissingResourceException If the specified key was not
   * found.
   * @return An array of messages for the specified key.
   */
  
  public String[] getMessageList(String key) throws MissingResourceException
    {
    return(getMessageList(key, DEFAULT_DELIMITER));
    }

  /** Get a message list for the specified key. Retrieves a message for the
   * specified key, and breaks the message on the specified delimiter
   * constructing an array in the process.
   *
   * @param key The key.
   * @param delimiter The delimiter to use.
   * @exception java.util.MissingResourceException If the specified key was not
   * found.
   * @return An array of messages for the specified key.
   */
  
  public String[] getMessageList(String key, String delimiter)
    throws MissingResourceException
    {
    String msg = getMessage(key);
    
    return(StringUtils.split(msg, delimiter));
    }

  /** Determine if a message is defined for the specified key.
   *
   * @param key The key.
   * @return <b>true</b> if the key exists, and <b>false</b> otherwise.
   */
   
  public boolean isMessageDefined(String key)
    {
    boolean ok = true;
    
    try
      {
      getMessage(key);
      }
    catch(MissingResourceException ex)
      {
      ok = false;
      }
    
    return(ok);
    }
  
  }
