/* ----------------------------------------------------------------------------
   The Kiwi Toolkit
   Copyright (C) 1998-99 Mark A. Lindner

   This file is part of Kiwi.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
   The author may be contacted at:
   
   frenzy@ix.netcom.com
   ----------------------------------------------------------------------------

   $Log: CompressedResourceLoader.java,v $
   Revision 1.2  1999/01/10 03:41:46  markl
   added GPL header & RCS tag

   ----------------------------------------------------------------------------
*/

package kiwi.util;

import java.io.*;
import java.util.zip.*;

/** A utility class containing methods for retrieving compressed application
  * resources; these resources typically reside within a JAR file among the
  * classes that make up an application. The location of a resource is
  * specified as a path relative to the location of its associated object's
  * class within the application's class hierarchy.
  * <p>
  * Resources may be retrieved as byte arrays, as <code>String</code>s, as
  * <code>InputStream</code>s, as <code>AudioClip</code>s, as
  * <code>Image</code>s,, or as <code>Properties</code> objects.
  * <p>
  * Both the GZIP and ZIP compression formats are supported.
  *
  * @see kiwi.util.ResourceLoader
  *
  * @author Mark Lindner
  * @author PING Software Group
  * @version 1.0.1 (10/98)
  */

public class CompressedResourceLoader extends ResourceLoader
  {
  /** The GZIP inflater type. */
  public static final int GZIP = 0;

  /** The ZIP inflater type. */
  public static final int ZIP = 1;

  /** The default inflater type. */
  public static final int DEFAULT_INFLATER = GZIP;

  protected static int inflater = DEFAULT_INFLATER;

  /** Construct a new <code>CompressedResourceLoader</code> with a default
    * input buffer size and inflater type.
    *
    * @param clazz The class object that serves as the resource anchor for this
    * <code>CompressedResourceLoader</code>.
    */

  public CompressedResourceLoader(Class clazz)
    {
    super(clazz);
    }

  /** Create a new compressed resource loader with the default input buffer
    * size.
    *
    * @param clazz The class object that serves as the resource anchor for this
    * <code>CompressedResourceLoader</code>.
    * @param type The inflater type; one of the static constants declared
    * above.
    * @exception java.lang.IllegalArgumentException If the value of
    * <code>type</code> is invalid.
    */

  public CompressedResourceLoader(Class clazz, int type)
    throws IllegalArgumentException
    {
    super(clazz);
    _init(type);
    }

  /** Create a new compressed resource loader. A new resource loader is created
    * with the specified input buffer size and inflater type.
    *
    * @param clazz The class object that serves as the resource anchor for this
    * <code>CompressedResourceLoader</code>.
    * @param type The inflater type; one of the static constants declared
    * above.
    * @param size The input buffer size.
    * @exception java.lang.IllegalArgumentException If the value of
    * <code>type</code> or <code>size</code> is invalid.
    */
  
  public CompressedResourceLoader(Class clazz, int type, int size)
    throws IllegalArgumentException
    {
    super(clazz);
    _init(type);
    }

  /* Internal initialization code. */

  private void _init(int type) throws IllegalArgumentException
    {
    if((type != GZIP) && (type != ZIP))
      throw(new IllegalArgumentException("Invalid inflater type"));

    this.inflater = type;
    }

  /** Get the inflater type.
    *
    * @return The inflater type used by this loader.
    */

  public int getInflaterType()
    {
    return(inflater);
    }

  /** Retrieve a compressed resource as a stream. Resources are located at a
    * path that is relative to a given class. The data read will be
    * automatically decompressed using the specified appropriate inflater type,
    * or the default inflater type if none was specified.
    *
    * @param clazz The class object associated with this resource.
    * @param path A file path specifying the location of the resource; the path
    * is relative to the location of the class file.
    * @return An <code>InputStream</code> from which the (decompressed)
    * resource data may be read.
    * @exception java.io.IOException If the resource was not found.
    */

  public InputStream getResourceAsStream(String path) throws IOException
    {
    InputStream is = super.getResourceAsStream(path);
    InflaterInputStream inf = null;

    switch(inflater)
      {
      case ZIP:
        inf = new ZipInputStream(is);
        break;

      case GZIP:
      default:
        inf = new GZIPInputStream(is, KiwiUtils.blockSize * 4);
        break;
      }

    return(inf);
    }
  }
