/* ----------------------------------------------------------------------------
   The Kiwi Toolkit
   Copyright (C) 1998-99 Mark A. Lindner

   This file is part of Kiwi.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
   The author may be contacted at:
   
   frenzy@ix.netcom.com
   ----------------------------------------------------------------------------

   $Log: RepeaterButton.java,v $
   Revision 1.4  1999/08/24 07:41:21  markl
   More fixes.

   Revision 1.3  1999/08/24 06:06:20  markl
   Fixed event handling code.

   Revision 1.2  1999/01/10 02:56:27  markl
   added GPL header & RCS tag

   ----------------------------------------------------------------------------
*/

package kiwi.ui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.plaf.*;

import kiwi.event.*;

/** A trivial extension of <code>KButton</code> that repeatedly fires action
  * events when the mouse button is held down within its bounds. The initial
  * and repeat delays are configurable. All other functionality is identical to
  * that of <code>KButton</code>.
  *
  * @see kiwi.ui.KButton
  * @see javax.swing.Timer
  *
  * @version 1.4 (08/99)
  * @author Mark Lindner
  */

public class RepeaterButton extends KButton
  {
  private Timer timer;

  /** The default initial delay. */
  public static final int DEFAULT_INITIAL_DELAY = 1000;

  /** The default repeat delay. */
  public static final int DEFAULT_REPEAT_DELAY = 300;

  /** Construct a new <code>RepeaterButton</code> with the specified icon.
   *
   * @param icon The icon for the button.
   */
     
  public RepeaterButton(Icon icon)
    {
    super(icon);
    _init();
    }

  /** Construct a new <code>RepeaterButton</code> with the specified text.
   *
   * @param text The text for the button.
   */

  public RepeaterButton(String text)
    {
    super(text);
    _init();
    }

  /** Construct a new <code>RepeaterButton</code> with the specified text and
   * icon.
   *
   * @param text The text for the button.
   * @param icon The icon for the button.
   */

  public RepeaterButton(String text, Icon icon)
    {
    super(text, icon);
    _init();
    }

  /* common initialization code */
  
  private void _init()
    {
    timer = new Timer(DEFAULT_REPEAT_DELAY, new ActionListener()
                      {
                      public void actionPerformed(ActionEvent evt)
                        {
                        _fireEvent();
                        }
                      });
    
    timer.setInitialDelay(DEFAULT_INITIAL_DELAY);
    timer.setRepeats(true);
    }
  
  /** Set the initial delay on this button.
   *
   * @param msec The number of milliseconds to wait after the initial mouse
   * press to begin generating action events.
   */

  public void setInitialDelay(int msec)
    {
    timer.setInitialDelay(msec);
    }

  /** Get the initial delay.
   *
   * @return The current initial delay, in milliseconds.
   */

  public int getInitialDelay()
    {
    return(timer.getInitialDelay());
    }

  /** Set the repeat delay.
   *
   * @param msec The number of milliseconds to wait between the firing of
   * successive action events.
   */

  public void setRepeatDelay(int msec)
    {
    timer.setDelay(msec);
    }

  /** Get the repeat delay.
   *
   * @return The current repeat delay, in milliseconds.
   */

  public int getRepeatDelay()
    {
    return timer.getDelay();
    }

  /** Enable or disable the button.
   *
   * @param flag A flag specifying whether the button should be enabled or
   * disabled.
   */

  public void setEnabled(boolean flag)
    {
    if(!flag && timer.isRunning())
      _stop();
    
    super.setEnabled(flag);
    }
  
  /** Process mouse events. */

  protected void processMouseEvent(MouseEvent evt)
    {
    if(!isEnabled())
      return;
    
    int id = evt.getID();

    switch(id)
      {
      case evt.MOUSE_CLICKED:
        super.processMouseEvent(evt);
        break;

      case evt.MOUSE_PRESSED:
        _start();
        break;

      case evt.MOUSE_RELEASED:
        _stop();
        break;
      }
    }

  /* start the loop */

  private void _start()
    {
    model.setArmed(true);
    model.setPressed(true);
    repaint();
    timer.start();
    }
  
  /* stop the loop */

  private void _stop()
    {
    timer.stop();

    model.setArmed(false);
    model.setPressed(false);
    repaint();
    }

  /* fire an ActionEvent */
  
  private void _fireEvent()
    {
    fireActionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED,
                                        null));
    }
  }
