
package kiwi.ui;

import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.plaf.metal.*;

import kiwi.ui.model.*;
import kiwi.util.*;

/** An extension of <code>JTable</code> that fixes a number of blatant bugs
 * and limitations in that component.
 *
 * @author Mark Lindner
 * @author PING Software Group
 * @version 0.1 (07/99)
 */

public class KTable extends JTable
  {
  private TableSorter sorter;
  private boolean sortable = false, editable = true;
  private TableModel realModel = null;
  private Icon i_sort;
  private _HeaderRenderer headerRenderer;
  private boolean columnReordering = false;
  
  /** Construct a new <code>KTable</code>.
   */
  
  public KTable()
    {
    i_sort = KiwiUtils.getResourceManager().getIcon("sort-down.gif");
    
    setAutoCreateColumnsFromModel(true);
    
    setShowHorizontalLines(false);
    setShowVerticalLines(false);
    setShowGrid(false);
    setTableHeader(new _TableHeader(getColumnModel()));

    setSelectionModel(new _SelectionModel());
    }

  /** Get the row translation for a given row in the table. If sorting is
   * turned on, the visual row may not be the same as the row in the underlying
   * data model; this method obtains the model row corresponding to the
   * specified visual row, whether sorting is turned on or not.
   *
   * @param row The visual row.
   * @return The corresponding row in the data model.
   */

  public int getRowTranslation(int row)
    {
    return(sortable ? sorter.getRowTranslation(row) : row);
    }
  
  /** Set the data model for this table.
   *
   * @param model The new model.
   */
  
  public void setModel(TableModel model)
    {
    realModel = model;
    if(sortable)
      {
      sorter = new TableSorter(realModel);
      super.setModel(sorter);
      sorter.registerTableHeaderListener(this);
      }
    else
      super.setModel(realModel);

    prepareHeader();
    }

  /** Prepare the header to use teh custom header renderer.
   */

  private void prepareHeader()
    {
    TableColumnModel cmodel = getColumnModel();
    int cols = cmodel.getColumnCount();

    for(int i = 0; i < cols; i++)
      {
      TableColumn tc = cmodel.getColumn(i);
      tc.setHeaderRenderer(headerRenderer = new _HeaderRenderer());
      }
    }

  /** Enable or disable this table. A disabled table cannot be edited or
   * sorted, nor can the selection be changed, nor the columns reordered or
   * resized.
   *
   * @param editable A flag specifying whether this table should be enabled
   * or disabled.
   */

  public void setEnabled(boolean enabled)
    {
    Color fg = (enabled
                ? MetalLookAndFeel.getControlTextColor()
                : MetalLookAndFeel.getInactiveControlTextColor());
    
    setForeground(fg);
    setSelectionForeground(fg);
    super.setEnabled(enabled);
    getTableHeader().setEnabled(enabled);
    
    }

  /** Set the editable state of this table.
   *
   * @param editable A flag specifying whether this table is editable.
   */
  
  public void setEditable(boolean editable)
    {
    this.editable = editable;
    }

  /** Determine if the table is editable.
   *
   * @return <b>true</b> if the table is editable and <b>false</b> otherwise.
   */
  
  public boolean isEditable()
    {
    return(editable);
    }

  /** Set the sortable state of this table.
   *
   * @param sortable A flag specifying whether this table is sortable.
   */
  
  public void setSortable(boolean sortable)
    {
    if(this.sortable == sortable)
      return;
    
    this.sortable = sortable;

    if(sortable)
      {
      if(realModel != null)
        {
        sorter = new TableSorter(realModel);
        super.setModel(sorter);
        sorter.registerTableHeaderListener(this);
        }
      }
    else
      {
      if(realModel != null)
        super.setModel(realModel);
      sorter = null;
      }

    prepareHeader();    
    }

  /** Determine if the table is sortable.
   *
   * @return <b>true</b> if the table is sortable and <b>false</b> otherwise.
   */
  
  public boolean isSortable()
    {
    return(sortable);
    }  

  /** Determine if a cell is editable. Editability of a given cell is
   * ultimately determined by the table model, unless the table has been
   * made non-editable via a call to <code>setEditable()</code>, or if it has
   * been disabled via a call to <code>setEnabled()</code>.
   *
   * @param row The row of the cell.
   * @param col The column of the cell.
   * @return <b>true</b> if the cell at the specified coordinates is editable,
   * or <b>false</b> if it is not editable or if the table has been made
   * non-editable.
   * @see #setEditable
   */

  public boolean isCellEditable(int row, int col)
    {
    if(!isEnabled())
      return(false);
    
    return(editable ? getModel().isCellEditable(row, col) : false);
    }

  /** Scroll the table to ensure that a given row is visible.
   *
   * @param row The row that must be visible.
   */
  
  public void ensureRowIsVisible(int row)
    {
    Rectangle r = getCellRect(row, 0, false);
    r.y = ((rowHeight + rowMargin) * row) + (getSize().height / 2);
    scrollRectToVisible(r);
    }
  
  /* A custom list selection model that honors disabled state by disallowing
   * changes to the current selection(s).
   */

  private class _SelectionModel extends DefaultListSelectionModel
    {
    public void clearSelection()
      {
      if(isEnabled())
        super.clearSelection();
      }
    
    public void addSelectionInterval(int a, int b)
      {
      if(isEnabled())
        super.addSelectionInterval(a, b);
      }
    
    public void insertIndexInterval(int a, int b, boolean before)
      {
      if(isEnabled())
        super.insertIndexInterval(a, b, before);
      }
    
    public void removeIndexInterval(int a, int b)
      {
      if(isEnabled())
        super.removeIndexInterval(a, b);
      }

    public void setSelectionInterval(int a, int b)
      {
      if(isEnabled())
        super.setSelectionInterval(a, b);
      }

    public void setAnchorSelectionIndex(int a)
      {
      if(isEnabled())
        super.setAnchorSelectionIndex(a);
      }

    public void setLeadSelectionIndex(int a)
      {
      if(isEnabled())
        super.setLeadSelectionIndex(a);
      }
    }

  /* A custom header renderer that displays a sort icon in each column if the
   * table is sortable.
   */
  
  private class _HeaderRenderer extends HeaderCellRenderer
    {
    public Component getTableCellRendererComponent(JTable table,
                                                   Object value,
                                                   boolean isSelected,
                                                   boolean hasFocus,
                                                   int row,
                                                   int column)
      {
      setIcon(isSortable() ? i_sort : null);
      
      return(super.getTableCellRendererComponent(table, value, isSelected,
                                                   hasFocus, row, column));
      } 
    }

  /* A custom table header that honors disabled state by disallowing column
   * resizing and reordering.
   */

  private class _TableHeader extends JTableHeader
    {

    public _TableHeader(TableColumnModel model)
      {
      super(model);
      }
    
    public boolean getResizingAllowed()
      {
      return(isEnabled() ? super.getResizingAllowed() : false);
      }

    public boolean getReorderingAllowed()
      {
      return(isEnabled() ? super.getReorderingAllowed() : false);
      }    
    }
  }
