/* (C) 1999-2000 Samuel Audet <guardia@cam.org>

Profitable use of this source code based on its execution or sale
excluding the cost of the media, shipping, manwork or supporting
hardware is not allowed unless granted by the author himself.  Any
modifications or inclusion of this code in other non-profitable programs
must contain this message and the original author's name. Programs based
on any of this source code must therefore contain the original or
modified source code files.  Use of this source code in a commercial
program will require permission from the author.  No more than 50% of
the original size of the source code from Disk Indexer can be used in a
non-commercial program, unless granted by the author.

*/

import java.io.*;
import java.awt.*;
import java.awt.image.*;
import java.util.*;
import java.util.zip.*;
import javax.swing.tree.*;

import encoder.jpeg.*;

public class HDataNode implements Cloneable
{
// stuff that is save in database
   String fileName = null, desc = null;
   ID3Tag id3tag = null;
   byte type = -1;

   public static final byte d = 100, // directory
                            f = 102, // file
                            v = 118, // volume, disk
                            c = 99,  // container
                            z = 122; // zip

   long length = -1, dateModified = -1;
   short width = -1, height = -1;

   int childPointer = -1, blobPointer = -1;
   boolean[] scanOptions = null;

// blob data
   Image thumbnail = null;

// length of all static data
   public static final short infoSize = 36;

// stuff not save in entry nodes, but in the page header
   int primaryPointer = -1, myPointer = -1, parentPointer = -1, index = -1;

   TreePath treePath = null;
   String path = null;

   protected static final Component component = new Component() {};
   protected static final MediaTracker tracker = new MediaTracker(component);
   protected static final Toolkit toolkit = Toolkit.getDefaultToolkit();

public HDataNode()
{

}

public HDataNode(File aFile)
{
   copyFrom(aFile);
}


public Object clone()
{
   try
   {
      HDataNode aClone = (HDataNode) super.clone();

      aClone.fileName = fileName;
      aClone.desc = desc;
      if(id3tag != null)
         aClone.id3tag = (ID3Tag) id3tag.clone();
      aClone.type = type;
      aClone.length = length;
      aClone.dateModified = dateModified;
      aClone.width = width;
      aClone.height = height;
      aClone.childPointer = childPointer;
      aClone.blobPointer = blobPointer;

      aClone.primaryPointer = myPointer;
      aClone.myPointer = myPointer;
      aClone.parentPointer = parentPointer;
      aClone.index = index;

      aClone.thumbnail = thumbnail; // clone?

      aClone.treePath = treePath;
      aClone.path = path;

      return aClone;
   }
   catch(CloneNotSupportedException e)
   {
      return null;
   }
}

public boolean equals(Object aNode)
{
   if(aNode == this)
      return true;

   if(aNode == null)
      return false;

   if(!(aNode instanceof HDataNode))
      return false;

   // good enough??
   if( primaryPointer == ((HDataNode) aNode).primaryPointer &&
       type == ((HDataNode) aNode).type &&
       toString().equals(aNode.toString()) &&
       dateModified == ((HDataNode) aNode).dateModified)
      return true;
   else
      return false;
}

public void clear()
{
   fileName = null;
   desc = null;
   id3tag = null;
   type = -1;
   length = -1;
   dateModified = -1;
   width = -1;
   height = -1;
   childPointer = -1;
   blobPointer = -1;

   primaryPointer = -1;
   myPointer = -1;
   parentPointer = -1;
   index = -1;

   flushThumbnail();

   treePath = null;
   path = null;
}

private void flushThumbnail()
{
   if(thumbnail != null)
      thumbnail.flush();
   thumbnail = null;
}


// put file attributes into this datanode
public void copyFrom(File aFile)
{
   copyFrom(aFile,new boolean[] {false,false,false});
}


// put file attributes into this datanode
public void copyFrom(File aFile, boolean[] scanOptions)
{
   clear();

   dateModified = aFile.lastModified();
   length = aFile.length();
   if(aFile.isDirectory())
      type = d;
   else
      type = f;

   fileName = aFile.getName();

   if(scanOptions[1]) // scan ID3tags
   {
      try
      {
         id3tag = new ID3Tag();
         if(!id3tag.read(aFile.getPath()))
            id3tag = null;
      }
      catch(IOException e) { id3tag = null; }
   }
   else
      id3tag = null;

   // ImageIcon seems to leak on most releases of Swing 1.1 and Java 2, so
   // let's only use it in the GUI and here let's use Image with the ToolKit only...
   if(scanOptions[2]) // scan GIF/JPG
   {
      try
      {
         Image image = toolkit.getImage(aFile.getPath());
         synchronized(tracker)
         {
            tracker.addImage(image,0);
            tracker.waitForID(0);
            tracker.removeImage(image, 0);
         }
         // System.out.println(image + " " + aFile.getPath() + " " + image.getHeight(null) + " " + image.getWidth(null));

         flushThumbnail();
         if(image.getHeight(null) != -1 && image.getWidth(null) != -1)
         {
            thumbnail = image.getScaledInstance(-1, 90, Image.SCALE_SMOOTH);
            synchronized(tracker)
          {
               tracker.addImage(thumbnail,0);
             tracker.waitForID(0);
               tracker.removeImage(thumbnail, 0);
          }

            width = (short) image.getWidth(null);
            height = (short) image.getHeight(null);
            //System.out.println("Made: " + thumbnail + " " + thumbnail.getWidth(null) + " " + thumbnail.getHeight(null));

         }

         image.flush();
         image = null;
      }
//      catch(IOException e) { flushThumbnail(); }
      catch(InterruptedException e) { flushThumbnail(); }
   }
   else
      flushThumbnail();
}


// put zip entry attributes into this datanode
public void copyFrom(ZipEntry aZipEntry)
{
   clear();

   dateModified = aZipEntry.getTime();
   length = aZipEntry.getSize();
   fileName = aZipEntry.getName();
   if(aZipEntry.isDirectory())
   {
      type = d;
      fileName = fileName.substring(fileName.lastIndexOf('/',fileName.length()-2)+1,fileName.length()-1);
   }
   else
   {
      type = f;
      fileName = fileName.substring(fileName.lastIndexOf('/')+1);
   }
}



public boolean readFrom(RandomAccessFile datafile) throws IOException
{
   short strLength;

   strLength = datafile.readShort();
   if(strLength <= 0)
      fileName = null;
   else
   {
      byte[] fileNameRaw = new byte[strLength];
      datafile.read(fileNameRaw);
      fileName = new String(fileNameRaw);
   }

   strLength = datafile.readShort();
   if(strLength <= 0)
      desc = null;
   else
   {
      byte[] descRaw = new byte[strLength];
      datafile.read(descRaw);
      desc = new String(descRaw);
   }

   short dataLength = datafile.readShort();
   if(dataLength <= 0)
      id3tag = null;
   else if(dataLength == id3tag.tagsize)
      id3tag = new ID3Tag(datafile);
   else
      datafile.skipBytes(dataLength);

   type = datafile.readByte();
   length = datafile.readLong();
   dateModified = datafile.readLong();
   width = datafile.readShort();
   height = datafile.readShort();
   childPointer = datafile.readInt();
   blobPointer = datafile.readInt();
   byte temp = datafile.readByte();
   if(temp == 0)
      scanOptions = null;
   else
   {
      if(scanOptions == null)
         scanOptions = new boolean[3];
      scanOptions[0] = (temp & 1) == 1;
      scanOptions[1] = (temp & 2) == 2;
      scanOptions[2] = (temp & 4) == 4;
   }

   flushThumbnail();

   return true;
}


public short size()
{
   short strLength = 0;
   if(fileName != null)
      strLength += (short) fileName.getBytes().length;
   if(desc != null)
      strLength += (short) desc.getBytes().length;
   if(id3tag != null)
      strLength += (short) id3tag.tagsize;

   return (short) (strLength + infoSize);
}


public String toString()
{
   /*
   if(type == d)
      return fileName + " " + length + " <DIR> " + new Date(dateModified);
   else
      return fileName + " " + length + " " + new Date(dateModified);
      */

   if(type == v)
       return desc;
   else
       return fileName;
}

// write node to database
public boolean writeTo(RandomAccessFile dataFile) throws IOException
{
   if(fileName == null)
      dataFile.writeShort(0);
   else
   {
      byte[] fileNameRaw = fileName.getBytes();
      dataFile.writeShort(fileNameRaw.length);
      dataFile.write(fileNameRaw);
   }

   if(desc == null)
      dataFile.writeShort(0);
   else
   {
      byte[] descRaw = desc.getBytes();
      dataFile.writeShort(descRaw.length);
      dataFile.write(descRaw);
   }

   if(id3tag == null)
      dataFile.writeShort(0);
   else
   {
      dataFile.writeShort(id3tag.tagsize);
      id3tag.write(dataFile);
   }

   dataFile.writeByte(type);
   dataFile.writeLong(length);
   dataFile.writeLong(dateModified);
   dataFile.writeShort(width);
   dataFile.writeShort(height);
   dataFile.writeInt(childPointer);
   dataFile.writeInt(blobPointer);

   byte temp = 0;
   if(scanOptions != null)
   {
      if(scanOptions[0]) temp |= 1;
      if(scanOptions[1]) temp |= 2;
      if(scanOptions[2]) temp |= 4;
   }
   dataFile.writeByte(temp);

   return true;
}

// takes the raw Image in this object, and returns JPEG encoded data
public byte[] getBlobData()
{
   if(thumbnail == null)
      return null;

   //System.out.println("Trying: " + thumbnail + " " + thumbnail.getWidth(null) + " " + thumbnail.getHeight(null));

   // make an educated guess at the possible maximum final size of the JPG
   ByteArrayOutputStream dataStream = new ByteArrayOutputStream(3*thumbnail.getHeight(null)*thumbnail.getWidth(null)/4);
   new JpegEncoder(thumbnail,75,dataStream).Compress();

   //System.out.println("guessed: " + 3*thumbnail.getHeight(null)*thumbnail.getWidth(null)/4 + " gave: " + dataStream.size());

   return dataStream.toByteArray();
}

// sets the Image object in this node using the given JPEG encoded data
public boolean setBlobData(byte[] data)
{
   flushThumbnail();

   thumbnail = toolkit.createImage(data);

   try
   {
      synchronized(tracker)
      {
         tracker.addImage(thumbnail,0);
         tracker.waitForID(0);
         tracker.removeImage(thumbnail, 0);
      }
   }
   catch(InterruptedException e) { }

   //System.out.println(thumbnail + " " + fileName + " " + data.length);

   if(thumbnail.getHeight(null) == -1 || thumbnail.getWidth(null) == -1)
   {
      System.out.println("NOT!!");
      flushThumbnail();
      return false;
   }

   return true;
}

public static void skip(RandomAccessFile dataFile) throws IOException
{
   dataFile.skipBytes(dataFile.readShort());
   dataFile.skipBytes(dataFile.readShort());
   dataFile.skipBytes(dataFile.readShort());
   dataFile.skipBytes(HDataNode.infoSize - 6);
}

protected void finalize() throws Throwable
{
   fileName = null;
   desc = null;
   flushThumbnail();

   super.finalize();
}

}
