/*
Copyright (c) 2011 Dietger van Antwerpen (dietger@xs4all.nl)

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef _CIE_H
#define _CIE_H

#include "math/vec4.h"

#define SPECTRAL_CIE_BINS 95

__constant float CIE_xbar[SPECTRAL_CIE_BINS] = { 
		0.000129900000f, 0.000232100000f,
		0.000414900000f, 0.000741600000f, 0.001368000000f, 0.002236000000f,
		0.004243000000f, 0.007650000000f, 0.014310000000f, 0.023190000000f,
		0.043510000000f, 0.077630000000f, 0.134380000000f, 0.214770000000f,
		0.283900000000f, 0.328500000000f, 0.348280000000f, 0.348060000000f,
		0.336200000000f, 0.318700000000f, 0.290800000000f, 0.251100000000f,
		0.195360000000f, 0.142100000000f, 0.095640000000f, 0.057950010000f,
		0.032010000000f, 0.014700000000f, 0.004900000000f, 0.002400000000f,
		0.009300000000f, 0.029100000000f, 0.063270000000f, 0.109600000000f,
		0.165500000000f, 0.225749900000f, 0.290400000000f, 0.359700000000f,
		0.433449900000f, 0.512050100000f, 0.594500000000f, 0.678400000000f,
		0.762100000000f, 0.842500000000f, 0.916300000000f, 0.978600000000f,
		1.026300000000f, 1.056700000000f, 1.062200000000f, 1.045600000000f,
		1.002600000000f, 0.938400000000f, 0.854449900000f, 0.751400000000f,
		0.642400000000f, 0.541900000000f, 0.447900000000f, 0.360800000000f,
		0.283500000000f, 0.218700000000f, 0.164900000000f, 0.121200000000f,
		0.087400000000f, 0.063600000000f, 0.046770000000f, 0.032900000000f,
		0.022700000000f, 0.015840000000f, 0.011359160000f, 0.008110916000f,
		0.005790346000f, 0.004106457000f, 0.002899327000f, 0.002049190000f,
		0.001439971000f, 0.000999949300f, 0.000690078600f, 0.000476021300f,
		0.000332301100f, 0.000234826100f, 0.000166150500f, 0.000117413000f,
		0.000083075270f, 0.000058706520f, 0.000041509940f, 0.000029353260f,
		0.000020673830f, 0.000014559770f, 0.000010253980f, 0.000007221456f,
		0.000005085868f, 0.000003581652f, 0.000002522525f, 0.000001776509f,
		0.000001251141f };
__constant float CIE_ybar[SPECTRAL_CIE_BINS] = { 
		0.000003917000f, 0.000006965000f,
		0.000012390000f, 0.000022020000f, 0.000039000000f, 0.000064000000f,
		0.000120000000f, 0.000217000000f, 0.000396000000f, 0.000640000000f,
		0.001210000000f, 0.002180000000f, 0.004000000000f, 0.007300000000f,
		0.011600000000f, 0.016840000000f, 0.023000000000f, 0.029800000000f,
		0.038000000000f, 0.048000000000f, 0.060000000000f, 0.073900000000f,
		0.090980000000f, 0.112600000000f, 0.139020000000f, 0.169300000000f,
		0.208020000000f, 0.258600000000f, 0.323000000000f, 0.407300000000f,
		0.503000000000f, 0.608200000000f, 0.710000000000f, 0.793200000000f,
		0.862000000000f, 0.914850100000f, 0.954000000000f, 0.980300000000f,
		0.994950100000f, 1.000000000000f, 0.995000000000f, 0.978600000000f,
		0.952000000000f, 0.915400000000f, 0.870000000000f, 0.816300000000f,
		0.757000000000f, 0.694900000000f, 0.631000000000f, 0.566800000000f,
		0.503000000000f, 0.441200000000f, 0.381000000000f, 0.321000000000f,
		0.265000000000f, 0.217000000000f, 0.175000000000f, 0.138200000000f,
		0.107000000000f, 0.081600000000f, 0.061000000000f, 0.044580000000f,
		0.032000000000f, 0.023200000000f, 0.017000000000f, 0.011920000000f,
		0.008210000000f, 0.005723000000f, 0.004102000000f, 0.002929000000f,
		0.002091000000f, 0.001484000000f, 0.001047000000f, 0.000740000000f,
		0.000520000000f, 0.000361100000f, 0.000249200000f, 0.000171900000f,
		0.000120000000f, 0.000084800000f, 0.000060000000f, 0.000042400000f,
		0.000030000000f, 0.000021200000f, 0.000014990000f, 0.000010600000f,
		0.000007465700f, 0.000005257800f, 0.000003702900f, 0.000002607800f,
		0.000001836600f, 0.000001293400f, 0.000000910930f, 0.000000641530f,
		0.000000451810f };
__constant float CIE_zbar[SPECTRAL_CIE_BINS] = { 
		0.000606100000f, 0.001086000000f,
		0.001946000000f, 0.003486000000f, 0.006450001000f, 0.010549990000f,
		0.020050010000f, 0.036210000000f, 0.067850010000f, 0.110200000000f,
		0.207400000000f, 0.371300000000f, 0.645600000000f, 1.039050100000f,
		1.385600000000f, 1.622960000000f, 1.747060000000f, 1.782600000000f,
		1.772110000000f, 1.744100000000f, 1.669200000000f, 1.528100000000f,
		1.287640000000f, 1.041900000000f, 0.812950100000f, 0.616200000000f,
		0.465180000000f, 0.353300000000f, 0.272000000000f, 0.212300000000f,
		0.158200000000f, 0.111700000000f, 0.078249990000f, 0.057250010000f,
		0.042160000000f, 0.029840000000f, 0.020300000000f, 0.013400000000f,
		0.008749999000f, 0.005749999000f, 0.003900000000f, 0.002749999000f,
		0.002100000000f, 0.001800000000f, 0.001650001000f, 0.001400000000f,
		0.001100000000f, 0.001000000000f, 0.000800000000f, 0.000600000000f,
		0.000340000000f, 0.000240000000f, 0.000190000000f, 0.000100000000f,
		0.000049999990f, 0.000030000000f, 0.000020000000f, 0.000010000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f, 0.000000000000f, 0.000000000000f, 0.000000000000f,
		0.000000000000f };	
	
inline Vec3* Spectralf4toXYZ( Vec3 *out, const float4 *SDF, const float4 *spectrum )
{
	const float SPECTRAL_CIE_WAVELENGTH_MIN = 360.f;
	const float SPECTRAL_CIE_WAVELENGTH_MAX = 830.f;
	
			
	// compute piecewise constant spectral CIE curve intervals
	int4 id = convert_int4((((*spectrum) - SPECTRAL_CIE_WAVELENGTH_MIN) / (SPECTRAL_CIE_WAVELENGTH_MAX - SPECTRAL_CIE_WAVELENGTH_MIN)) * SPECTRAL_CIE_BINS);
	id = min( id, (int4)(SPECTRAL_CIE_BINS-1,SPECTRAL_CIE_BINS-1,SPECTRAL_CIE_BINS-1,SPECTRAL_CIE_BINS-1));
	id = max( id, (int4)(0,0,0,0));
	
	// initialize XYZ color
	v3init( out, 0.f, 0.f, 0.f );
	
	// compute X
	const float4 CIE_x = (float4)(CIE_xbar[id.x], CIE_xbar[id.y], CIE_xbar[id.z], CIE_xbar[id.w]);
	out->x += v4dot( SDF , &CIE_x );
	
	// compute Y
	const float4 CIE_y = (float4)(CIE_ybar[id.x], CIE_ybar[id.y], CIE_ybar[id.z], CIE_ybar[id.w]);
	out->y += v4dot( SDF , &CIE_y );
	
	// compute Z
	const float4 CIE_z = (float4)(CIE_zbar[id.x], CIE_zbar[id.y], CIE_zbar[id.z], CIE_zbar[id.w]);
	out->z += v4dot( SDF , &CIE_z );
	
	return out;
}

inline Vec3* Spectralf1toXYZ( Vec3 *out, const float *SDF, const float *spectrum )
{
	const float SPECTRAL_CIE_WAVELENGTH_MIN = 360.f;
	const float SPECTRAL_CIE_WAVELENGTH_MAX = 830.f;
				
	// compute piecewise constant spectral CIE curve intervals
	int id = (int)((((*spectrum) - SPECTRAL_CIE_WAVELENGTH_MIN) / (SPECTRAL_CIE_WAVELENGTH_MAX - SPECTRAL_CIE_WAVELENGTH_MIN)) * SPECTRAL_CIE_BINS);
	id = min( id, (int)(SPECTRAL_CIE_BINS-1));
	id = max( id, (int)(0));
	
	// initialize XYZ color
	v3init( out, 0.f, 0.f, 0.f );
	
	// compute X
	out->x += *SDF * CIE_xbar[id];
	// compute Y
	out->y += *SDF * CIE_ybar[id];
	// compute Z
	out->z += *SDF * CIE_zbar[id];
	
	return out;
}

inline Vec3* XYZtoRGB( Vec3 *out, const Vec3* XYZ)
{
	// clamp negatives
	out->x = max(0.f,   2.5623f  * XYZ->x + (-1.1661f) * XYZ->y + (-0.3962f) * XYZ->z);
	out->y = max(0.f, (-1.0215f) * XYZ->x +   1.9778f  * XYZ->y +   0.0437f  * XYZ->z);
	out->z = max(0.f,   0.0752f  * XYZ->x + (-0.2562f) * XYZ->y +   1.1810f  * XYZ->z);
	return out;
}

inline Vec3* Spectralf4toRGB( Vec3 *out, const float4 *SDF, const float4 *spectrum )
{
	#ifndef SPECTRAL
		v3initf4( out, SDF );
	#else
		Vec3 XYZ;
		// convert spectrum to XYZ
		Spectralf4toXYZ( &XYZ, SDF, spectrum );
		// convert XYZ to RGB
		XYZtoRGB( out, &XYZ );
	#endif
	return out;
}
	
inline Vec3* Spectralf1toRGB( Vec3 *out, const float *SDF, const float *spectrum )
{
	#ifndef SPECTRAL
		v3init( out, *SDF, *SDF, *SDF );
	#else
		Vec3 XYZ;
		// convert spectrum to XYZ
		Spectralf1toXYZ( &XYZ, SDF, spectrum );
		// convert XYZ to RGB
		XYZtoRGB( out, &XYZ );
	#endif
	return out;
}
	
#endif