/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.util;

/**
 * The Class Platform.
 *
 * @version 0.3
 * @since jNPad v0.1
 */
@SuppressWarnings("nls")
public final class Platform {
  /** The Constant OS_NAME. */
  private static final String OS_NAME              = System.getProperty("os.name").toLowerCase();
  /** The Constant OS_VERSION. */
  public static final String  OS_VERSION           = System.getProperty("os.version").toLowerCase();
  /** The Constant OS_ARCH. */
  public static final String  OS_ARCH              = System.getProperty("os.arch");
  /** The Constant ARCH_DATA_MODEL. */
  public static final String  ARCH_DATA_MODEL      = System.getProperty("sun.arch.data.model");
  /** The Constant SUN_DESKTOP. */
  public static final String  SUN_DESKTOP          = System.getProperty("sun.desktop");

  /** The Constant isWindows. */
  public static final boolean isWindows            = OS_NAME.startsWith("windows");
  /** The Constant isWin2kOrNewer. */
  public static final boolean isWin2kOrNewer       = isWindows && isOsVersionAtLeast("5.0");
  /** The Constant isWinVistaOrNewer. */
  public static final boolean isWinVistaOrNewer    = isWindows && isOsVersionAtLeast("6.0");
  /** The Constant isWin7OrNewer. */
  public static final boolean isWin7OrNewer        = isWindows && isOsVersionAtLeast("6.1");
  /** The Constant isWindowsNT. */
  public static final boolean isWindowsNT          = OS_NAME.startsWith("windows nt");
  /** The Constant isWindows2000. */
  public static final boolean isWindows2000        = OS_NAME.startsWith("windows 2000");
  /** The Constant isWindows2003. */
  public static final boolean isWindows2003        = OS_NAME.startsWith("windows 2003");
  /** The Constant isWindowsXP. */
  public static final boolean isWindowsXP          = OS_NAME.startsWith("windows xp");
  /** The Constant isWindowsVista. */
  public static final boolean isWindowsVista       = OS_NAME.startsWith("windows vista");
  /** The Constant isWindows7. */
  public static final boolean isWindows7           = OS_NAME.startsWith("windows 7");
  /** The Constant isWindows9x. */
  public static final boolean isWindows9x          = OS_NAME.startsWith("windows 9") || OS_NAME.startsWith("windows me");
  /** The Constant isOS2. */
  public static final boolean isOS2                = OS_NAME.startsWith("os/2") || OS_NAME.startsWith("os2");
  /** The Constant isMac. */
  public static final boolean isMac                = OS_NAME.startsWith("mac");
  /** The Constant isFreeBSD. */
  public static final boolean isFreeBSD            = OS_NAME.startsWith("freebsd");
  /** The Constant isLinux. */
  public static final boolean isLinux              = OS_NAME.startsWith("linux");
  /** The Constant isSolaris. */
  public static final boolean isSolaris            = OS_NAME.startsWith("sunos");
  /** The Constant isUnix. */
  public static final boolean isUnix               = !isWindows && !isOS2;

  /** The Constant isKDE. */
  public static final boolean isKDE                = SUN_DESKTOP != null && SUN_DESKTOP.toLowerCase().contains("kde");
  /** The Constant isGnome. */
  public static final boolean isGnome              = SUN_DESKTOP != null && SUN_DESKTOP.toLowerCase().contains("gnome");

  /** The Constant isXWindow. */
  public static final boolean isXWindow            = isUnix && !isMac;

  /** The Constant areSymLinksSupported. */
  public static final boolean areSymLinksSupported = isUnix || isWinVistaOrNewer;

  /** The Constant is32Bit. */
  public static final boolean is32Bit              = ARCH_DATA_MODEL == null || ARCH_DATA_MODEL.equals("32");
  /** The Constant is64Bit. */
  public static final boolean is64Bit              = !is32Bit;
  /** The Constant isAMD64. */
  public static final boolean isAMD64              = "amd64".equals(OS_ARCH);
  /** The Constant isMacIntel64. */
  public static final boolean isMacIntel64         = isMac && "x86_64".equals(OS_ARCH);

  /**
   * no instances.
   */
  private Platform() {
    super();
  }

  /**
   * Checks if is os version at least.
   *
   * @param version the version
   * @return true, if is os version at least
   * @since 0.3
   */
  public static boolean isOsVersionAtLeast(String version) {
    return Utilities.compareVersionNumbers(OS_VERSION, version) >= 0;
  }

  /**
   * Gets the JRE version.
   *
   * @return the JRE version
   */
  public static String getJREVersion() {
    return System.getProperty("java.version");
  }

  /**
   * Checks if is JRE4.
   *
   * @return true, if is JRE4
   */
  public static boolean isJRE4() {
    return getJREVersion().startsWith("1.4");
  }

  /**
   * Checks if is JRE5.
   *
   * @return true, if is JRE5
   */
  public static boolean isJRE5() {
    return getJREVersion().startsWith("1.5");
  }

  /**
   * Checks if is JRE6.
   *
   * @return true, if is JRE6
   */
  public static boolean isJRE6() {
    return getJREVersion().startsWith("1.6");
  }

  /**
   * Checks if is JRE7.
   *
   * @return true, if is JRE7
   */
  public static boolean isJRE7() {
    return getJREVersion().startsWith("1.7");
  }

  /**
   * Checks if is JRE8.
   *
   * @return true, if is JRE8
   */
  public static boolean isJRE8() {
    return getJREVersion().startsWith("1.8");
  }

  /**
   * Checks if is JRE4 above.
   *
   * @return true, if is JRE4 above
   */
  public static boolean isJRE4Above() {
    return (getJREVersion().compareTo("1.4") >= 0);
  }

  /**
   * Checks if is JRE5 above.
   *
   * @return true, if is JRE5 above
   */
  public static boolean isJRE5Above() {
    return (getJREVersion().compareTo("1.5") >= 0);
  }

  /**
   * Checks if is JRE6 above.
   *
   * @return true, if is JRE6 above
   */
  public static boolean isJRE6Above() {
    return (getJREVersion().compareTo("1.6") >= 0);
  }

  /**
   * Checks if is JRE7 above.
   *
   * @return true, if is JRE7 above
   */
  public static boolean isJRE7Above() {
    return (getJREVersion().compareTo("1.7") >= 0);
  }

  /**
   * Checks if is JRE8 above.
   *
   * @return true, if is JRE8 above
   */
  public static boolean isJRE8Above() {
    return (getJREVersion().compareTo("1.8") >= 0);
  }

  /**
   * Checks if is JRE supported.
   *
   * @return true, if is JRE supported
   */
  public static boolean isJRESupported() {
    return Platform.isJRE6Above();
  }

}
