/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.highlighter;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.Shape;

import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultHighlighter;
import javax.swing.text.JTextComponent;
import javax.swing.text.Position;
import javax.swing.text.View;

import jnpad.GUIUtilities;
import jnpad.config.Config;

/**
 * The Class SearchHighlightPainter.
 *
 * @version 0.3
 * @since   jNPad v0.1
 * @see javax.swing.text.DefaultHighlighter.DefaultHighlightPainter
 * @see IHighlightPainter
 */
public class SearchHighlightPainter extends DefaultHighlighter.DefaultHighlightPainter implements IHighlightPainter {
  private Color _color;
  private Color _borderColor;

  /**
   * Instantiates a new search highlight painter.
   */
  public SearchHighlightPainter() {
    super(Config.SEARCH_HIGHLIGHTER_COLOR.getValue()); // lpm
    configure(CFG_COLOR);
  }

  /**
   * Configure.
   *
   * @param cfg int
   */
  @Override
  public void configure(final int cfg) {
    if ( (cfg & CFG_COLOR) != 0) {
      setColor(Config.SEARCH_HIGHLIGHTER_COLOR.getValue());
      setBorderColor(Config.SEARCH_HIGHLIGHTER_BORDER_COLOR.getValue());
    }
  }

  /**
   * Gets the color.
   *
   * @return the color
   * @see javax.swing.text.DefaultHighlighter.DefaultHighlightPainter#getColor()
   */
  @Override
  public Color getColor() {
    return _color;
  }

  /**
   * Sets the color.
   *
   * @param color the new color
   */
  public void setColor(Color color) {
    if (color != null) {
      _color = color;
    }
  }

  /**
   * Gets the border color.
   *
   * @return the border color
   */
  public Color getBorderColor() {
    return _borderColor;
  }

  /**
   * Sets the border color.
   *
   * @param color the new border color
   */
  public void setBorderColor(Color color) {
    if (color != null) {
      _borderColor = color;
    }
  }

  /**
   * Paint layer.
   *
   * @param g the g
   * @param offs0 the offs0
   * @param offs1 the offs1
   * @param bounds the bounds
   * @param c the c
   * @param view the view
   * @return the shape
   * @see javax.swing.text.DefaultHighlighter.DefaultHighlightPainter#paintLayer(java.awt.Graphics, int, int, java.awt.Shape, javax.swing.text.JTextComponent, javax.swing.text.View)
   */
  @Override
  public Shape paintLayer(Graphics g, int offs0, int offs1, Shape bounds, JTextComponent c, View view) {
    Rectangle r = getDrawingArea(offs0, offs1, bounds, view);

    if (r == null) {
      return null;
    }

    Color color = getColor();
    g.setColor(color == null ? c.getSelectionColor() : color);
    g.fillRect(r.x, r.y, r.width, r.height);

    Color borderColor = getBorderColor();
    if (borderColor != null) { // borde
      GUIUtilities.drawBorder(g, borderColor, r.x, r.y, r.width, r.height);
    }

    return r;
  }


  /**
   * Gets the drawing area.
   *
   * @param offs0 the offs0
   * @param offs1 the offs1
   * @param bounds the bounds
   * @param view the view
   * @return the drawing area
   */
  private Rectangle getDrawingArea(int offs0, int offs1, Shape bounds, View view) {
    // Contained in view, can just use bounds.
    if (offs0 == view.getStartOffset() && offs1 == view.getEndOffset()) {
      Rectangle alloc;

      if (bounds instanceof Rectangle) {
        alloc = (Rectangle) bounds;
      }
      else {
        alloc = bounds.getBounds();
      }

      return alloc;
    }
    // Should only render part of View.
    try {
      // --- determine locations ---
      Shape shape = view.modelToView(offs0, Position.Bias.Forward, offs1, Position.Bias.Backward, bounds);

      return (shape instanceof Rectangle) ? (Rectangle) shape : shape.getBounds();
    }
    catch (BadLocationException e) {
      // can't render
    }

    // Can't render
    return null;
  }

}
