/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text;

import java.awt.Color;
import java.awt.Font;
import java.io.Serializable;

import jnpad.config.Configurable;
import jnpad.text.syntax.ASN1Scheme;
import jnpad.text.syntax.CPPScheme;
import jnpad.text.syntax.CScheme;
import jnpad.text.syntax.ContentTypes;
import jnpad.text.syntax.JavaScheme;
import jnpad.text.syntax.PlainScheme;
import jnpad.text.syntax.PropertiesScheme;

/**
 * The Class Scheme.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public abstract class Scheme implements Configurable, Serializable {
  /** UID */
  private static final long serialVersionUID = -8902741649205093510L;
  
  /** is mini view. */
  protected boolean mini;
  
  /**
   * Instantiates a new scheme.
   *
   * @param mini the mini
   */
  public Scheme(boolean mini) {
    this.mini = mini;
  }
  
  /**
   * Gets the scheme.
   *
   * @param contentType the content type
   * @param mini the mini
   * @return the scheme
   */
  public static Scheme getScheme(String contentType, boolean mini) {
    if (ContentTypes.PROPERTIES.equals(contentType))
      return new PropertiesScheme(mini);
    if (ContentTypes.JAVA.equals(contentType)) 
      return new JavaScheme(mini);
    if (ContentTypes.C.equals(contentType))
      return new CScheme(mini);
    if (ContentTypes.CPP.equals(contentType))
      return new CPPScheme(mini);
    if (ContentTypes.ASN1.equals(contentType))
      return new ASN1Scheme(mini);
    return new PlainScheme(mini);    
    
  }

  /**
   * Gets the selected text color.
   *
   * @return the selected text color
   */
  public abstract Color getSelectedTextColor();

  /**
   * Gets the text color.
   *
   * @return the text color
   */
  public abstract Color getTextColor();

  /**
   * Sets the text color.
   *
   * @param c the new text color
   */
  public abstract void setTextColor(Color c);
  
  /**
   * Gets the text font.
   *
   * @return the text font
   */
  public abstract Font getTextFont();

  /**
   * Sets the text font.
   *
   * @param f the new text font
   */
  public abstract void setTextFont(Font f);
  
  /**
   * Gets the display name.
   *
   * @return the display name
   */
  public String getDisplayName() {
    return getDisplayName(getContentType());
  }

  /**
   * Gets the display name.
   *
   * @param contentType the content type
   * @return the display name
   */
  public static String getDisplayName(String contentType) {
    if(ContentTypes.JAVA.equals(contentType))
      return "Java"; //$NON-NLS-1$
    if(ContentTypes.PROPERTIES.equals(contentType))
      return TextBundle.getString("Scheme.properties"); //$NON-NLS-1$
    if(ContentTypes.C.equals(contentType))
      return "C"; //$NON-NLS-1$
    if(ContentTypes.CPP.equals(contentType))
      return "C++"; //$NON-NLS-1$
    if(ContentTypes.ASN1.equals(contentType))
      return "ASN.1"; //$NON-NLS-1$
    return TextBundle.getString("Scheme.text"); //$NON-NLS-1$
  }
  
  /**
   * Gets the content type.
   *
   * @return the content type
   */
  public abstract String getContentType();
  
  /**
   * Gets the start comment.
   *
   * @return the start comment
   */
  public abstract String[] getStartComment();

  /**
   * Gets the end comment.
   *
   * @return the end comment
   */
  public abstract String[] getEndComment();
  
  /**
   * Gets the start multiline comment.
   *
   * @return the start multiline comment
   */
  public abstract String[] getStartMultilineComment();

  /**
   * Gets the end multiline comment.
   *
   * @return the end multiline comment
   */
  public abstract String[] getEndMultilineComment();
  
  /**
   * Checks for c like syntax.
   *
   * @param contentType the content type
   * @return true, if successful
   */
  public static boolean hasCLikeSyntax(String contentType) {
    return ContentTypes.JAVA.equals(contentType) ||
           ContentTypes.C.equals(contentType) ||
           ContentTypes.CPP.equals(contentType);
  }
  
  /**
   * Checks for c like syntax.
   *
   * @return true, if successful
   */
  public boolean hasCLikeSyntax() {
    return hasCLikeSyntax(getContentType());
  }
}
