/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.action;

import java.awt.event.ItemEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jnpad.JNPadFrame;
import jnpad.action.JNPadActions.Group;
import jnpad.text.Buffer;
import jnpad.text.IEditAction;
import jnpad.ui.MnemonicHelper;
import jnpad.ui.status.StatusDisplayable;
import jnpad.util.Utilities;

/**
 * The Class SelectEncodingAction.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public final class SelectEncodingAction extends JNPadAction implements IEditAction {
  private String                          encoding;

  static final String                     NAME             = "select-encoding";                 //$NON-NLS-1$

  private static Map<String, JNPadAction> map              = new HashMap<String, JNPadAction>();

  /** UID */
  private static final long               serialVersionUID = 5774689924855432940L;

  /**
   * Instantiates a new select encoding action.
   *
   * @param jNPad the jNPad's frame
   * @param encoding String
   * @param mnemonicHelper MnemonicHelper
   */
  public SelectEncodingAction(JNPadFrame jNPad, String encoding, MnemonicHelper mnemonicHelper) {
    super(jNPad, NAME, Group.CONFIGURE);
    this.encoding = encoding;

    setLabel(encoding);

    final String name = NAME + Utilities.SPACE_STRING + encoding;
    setName(name);

    if (mnemonicHelper != null) {
      char mnemonic = mnemonicHelper.getMnemonic(getLabel());
      if (mnemonic != 0) {
        setMnemonic(mnemonic);
      }
    }

    setDescription(ActionBundle.getString(NAME.concat(".description"), getLabel())); //$NON-NLS-1$

    setStateAction(true);
    map.put(name, this);
  }

  /**
   * Handle state changed.
   *
   * @param e the item event
   * @see jnpad.action.JNPadAction#handleStateChanged(java.awt.event.ItemEvent)
   */
  @Override
  public void handleStateChanged(final ItemEvent e) {
    Buffer buffer = jNPad.getActiveBuffer();
    if (buffer != null && ItemEvent.SELECTED == e.getStateChange()) {
      buffer.setEncoding(encoding);
      jNPad.setStatus(ActionBundle.getString(NAME.concat(".message"), getLabel()), StatusDisplayable.TIMEOUT_DEFAULT); //$NON-NLS-1$
    }
  }

  /**
   * Gets the completions for prefix.
   * 
   * @param prefix the prefix
   * @return the completions for prefix
   */
  static List<String> getCompletionsForPrefix(String prefix) {
    List<String> list = new ArrayList<String>();
    for (String key : map.keySet()) {
      JNPadAction action = map.get(key);
      if (key.startsWith(prefix) && action.isEnabled()) {
        list.add(key);
      }
    }
    Collections.sort(list);
    return list;
  }

  /**
   * Perform action.
   *
   * @param name the name
   * @return true, if successful
   */
  static boolean performAction(String name) {
    JNPadAction action = map.get(name);
    if (action != null && action.isEnabled()) {
      action.setSelected(!action.isSelected());
      return true;
    }
    return false;
  }
  
}
