/**
 * @(#)Configurations.java
 *
 * <code>Configurations</code> class stores every preference 
 * the application provides. It also saves and loads these 
 * preferences to a file named <code>"config.ini"</code>.
 * 
 * @author Mehmet Mithat Cinar
 * @version 1.00 21.12.2009
 */

package bin;
 
import java.awt.Color;
import java.awt.Font;
import java.io.*;

public class Configurations implements Serializable {
	
	/**
	 * <code>serialVersionUID</code><br>
	 * Created by Eclipse.
	 */
	private static final long serialVersionUID = -3766560233184966131L;

	/**
	 * <code>font</code><br>
	 * The font object.
	 */
    private Font font = new Font(Font.SANS_SERIF,Font.ITALIC,30);
    
	/**
	 * <code>fontColor</code><br>
	 * Font color.
	 */
    private Color fontColor = new Color(0.2f,0.2f,0.2f);
    
	/**
	 * <code>backColor</code><br>
	 * Background color.
	 */
    private Color backColor = Color.WHITE;
	
	/**
	 * <code>useBGImage</code><br>
	 * Whether a background image is used or not.
	 */
    private boolean useBGImage = true;
	
	/**
	 * <code>BGImagePath</code><br>
	 * If used, the path of the background image.
	 */
    private String BGImagePath = "images/backgrounds/jLibris.jpg";
	
	/**
	 * <code>bookPath</code><br>
	 * Path of the book.
	 */
    private String bookPath = "";

	/**
	 * <code>pageNum</code><br>
	 * Current page number.
	 */
    private int pageNum = 0;
    
	/**
	 * <code>relax</code><br>
	 * Whether "Eye Relaxation Mode" is on or off.
	 */
    private boolean relax = false;
	
	/**
	 * <code>dummyLangVar</code><br>
	 * This variable is updated only when the 
	 * configurations are written to a file.
	 * Elsewhere the original Language class is used.
	 */
    private int dummyLangVar = 0;
    	
	/**
	 * Default constructor. Does nothing. If a 
	 * <code>Configurations</code> object is created 
	 * using the default constructor, then all the 
	 * variables get their default values.
	 */
    public Configurations() {}
        
	/**
	 * The real constructor. Page number and Relaxation 
	 * Mode can't be defined using the constructor.
	 * @param chosenBookPath Book path.
	 * @param chosenFont Font.
	 * @param chosenFontColor Font color.
	 * @param chosenBackColor Background color.
	 * @param chosenUseBGImage Whether BG image is used or not.
	 * @param chosenBGImagePath BGImage path.
	 * @param currentLanguage The current language.
	 */
    public Configurations(String chosenBookPath, Font chosenFont, Color chosenFontColor, Color chosenBackColor, boolean chosenUseBGImage, String chosenBGImagePath, int chosenPageNum,int currentLanguage) {
    	// Assign the bookPath
		bookPath = chosenBookPath;
		// Assign the font
    	font = chosenFont;
    	// Assign the font color
    	fontColor = chosenFontColor;
    	// Assign the are you really reading this
    	backColor = chosenBackColor;
    	// Assign the useBGImage
    	useBGImage = chosenUseBGImage;
    	// Assign background image path
    	BGImagePath = chosenBGImagePath;
    	// Assign pageNum
    	pageNum = chosenPageNum;
    	// Set the language
    	setLanguage(currentLanguage);
    }
    
	/**
	 * <code>getLanguage</code><br>
	 * Accessor for the language value.
	 * @return The index of the current language.
	 */
    public int getLanguage() {
    	// Get the language
    	return Language.getLanguageIndex();
    }
        
	/**
	 * <code>setLanguage</code><br>
	 * Mutator for the language object.
	 * @param i Index of the language.
	 */
    public void setLanguage(int i) {
    	// Set the language
    	Language.chooseLanguage(i);
    }
    
	/**
	 * <code>getFont</code><br>
	 * Accessor for the font object.
	 * @return The font object.
	 */
    public Font getFont() {
    	// Return font
    	return font;
    }
        
	/**
	 * <code>setFont</code><br>
	 * Mutator for the font object.
	 * @param f New font.
	 */
    public void setFont(Font f) {
    	// Set font to f
    	font = f;
    }
    
	/**
	 * <code>getFontColor</code><br>
	 * Accessor for the fontColor object.
	 * @return The Color object for the font color.
	 */
    public Color getFontColor() {
    	// Return fontColor
    	return fontColor;
    }
        
	/**
	 * <code>setFontColor</code><br>
	 * Mutator for the fontColor object.
	 * @param c New color.
	 */
    public void setFontColor(Color c) {
    	// Set fontColor to c
    	fontColor = c;
    }
    
	/**
	 * <code>getBackColor</code><br>
	 * Accessor for the backColor object.
	 * @return The Color object for the background color.
	 */
    public Color getBackColor() {
    	// Return backColor
    	return backColor;
    }
    
	/**
	 * <code>setBackColor</code><br>
	 * Mutator for the backColor object.
	 * @param c New background color.
	 */
    public void setBackColor(Color c) {
    	// Set backColor to c
    	backColor = c;
    }
    
	/**
	 * <code>isUseBGImage</code><br>
	 * Accessor for the useBGImage variable.
	 * @return Whether a BG image is used.
	 */
    public boolean isUseBGImage() {
    	// Return useBGImage
    	return useBGImage;
    }
        
	/**
	 * <code>setUseBGImage</code><br>
	 * Mutator for the useBGImage variable.
	 * @param b New useBGImage value
	 */
    public void setUseBGImage(boolean b) {
    	// Set useBGImage to b
    	useBGImage = b;
    }
    
	/**
	 * <code>getBGImagePath</code><br>
	 * Accessor for the BGImagePath variable.
	 * @return A String object representing the 
	 * background image's path.
	 */
    public String getBGImagePath() {
    	// Return BGImagePath
    	return BGImagePath;
    }
        
	/**
	 * <code>setBGImagePath</code><br>
	 * Mutator for the BGImagePath variable.
	 * @param p New path to the BG image.
	 */
    public void setBGImagePath(String p) {
    	// Set BGImagePath to p
    	BGImagePath = p;
    }
    
	/**
	 * <code>getBookPath</code><br>
	 * Accessor for the bookPath variable.
	 * @return A String object representing the 
	 * background image's path.
	 */
    public String getBookPath() {
    	// Return bookPath
    	return bookPath;
    }
        
	/**
	 * <code>setBookPath</code><br>
	 * Mutator for the bookPath variable.
	 * @param p New path to the book.
	 */
    public void setBookPath(String p) {
    	// Set bookPath to p
    	bookPath = p;
    }
    
	/**
	 * <code>getPageNum</code><br>
	 * Accessor for the pageNum variable.
	 * @return The page index.
	 */
    public int getPageNum() {
    	// Return pageNum
    	return pageNum;
    }
        
	/**
	 * <code>setPageNum</code><br>
	 * Mutator for the pageNum variable.
	 * @param pn New page number.
	 */
    public void setPageNum(int pn) {
    	// Set pageNum to pn
    	pageNum = pn;
    }
    
	/**
	 * <code>getRelax</code><br>
	 * Accessor for the relax variable.
	 * @return Whether the relaxation mode is on.
	 */
    public boolean getRelax() {
    	// Return relax
    	return relax;
    }
        
	/**
	 * <code>setRelax</code><br>
	 * Mutator for the relax variable.
	 */
    public void setRelax(boolean r) {
    	// Set relax to r
    	relax = r;
    }
    
    /**
	 * <code>saveConfigurations</code><br>
     * Saves the configurations as an object.
     * This works because the class Configurations is <i>Serializable</i>.
     * @return true if successful, false otherwise.
     */
    public boolean saveConfigurations() {
    	// Update the language data for the Configurations class
    	dummyLangVar = getLanguage();
    	// Try
    	try {
    		// Create an ObjectOutputStream object to write to the config.ini file
	    	ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream("config.ini"));
	    	// Write this object to oos
	    	oos.writeObject(this);
	    	// Close the file
	    	oos.close();
	    	return true;
	    // If an error occurs
    	} catch (Exception e) {
    		// Return false
    		return false;
    	}
    }
    
    
    /**
	 * <code>loadConfigurations</code><br>
     * Reads the configurations from the config file.
     * Sets the values of the object to the read ones.
     * If reading is not successful, these values remain the same.
     * To prevent partial loss of data, this method firstly keeps
     * the configurations object being read in a temporary variable.
     * Then, if no exceptions occur meanwhile, it assigns the new values.  
     */
    public void loadConfigurations() {
    	// Try
    	try {
    		// Create an ObjectInputStream object to read from the config.ini
    		ObjectInputStream ois = new ObjectInputStream(new FileInputStream("config.ini"));
    		// Read the object and store it in a temporary Configurations object
    		Configurations tempConf = (Configurations) ois.readObject();
    		// Set font
    		font = tempConf.getFont();
    		// Set font color
    		fontColor = tempConf.getFontColor();
    		// Set back color
    		backColor = tempConf.getBackColor();
    		// Set useBGImage
    		useBGImage = tempConf.isUseBGImage();
    		// Set BGImagePath
    		BGImagePath = tempConf.getBGImagePath();
    		// Set the book path
    		bookPath = tempConf.getBookPath();
    		// Set the page number
    		pageNum = tempConf.getPageNum();
    		// Set relax
    		relax = tempConf.getRelax();
    		// Set language
    		setLanguage(tempConf.dummyLangVar);
    	// If an error occurs
    	} catch(Exception e) {
    		// Don't do anything
    	}
    }    
}
