/* $Id: RowSetEventSupport.java,v 1.1 1999/03/03 06:00:22 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql;

import java.util.ArrayList;
import java.util.Iterator;
import javax.sql.RowSet;
import javax.sql.RowSetEvent;
import javax.sql.RowSetListener;

/**
 * Delegate class for managing row set events. This class is associated with
 * a <CODE>RowSet</CODE> instance and manages its list of listeners.
 * When a <CODE>RowSetEvent</CODE> occurs, this class will notify the
 * observers.
 * <BR>
 * Last modified $Date: 1999/03/03 06:00:22 $
 * @version $Revision: 1.1 $
 * @author George Reese (borg@imaginary.com)
 */
public class RowSetEventSupport {
    // The event listeners.
    private ArrayList listeners = new ArrayList();
    // The row set whose listeners are being managed.
    private RowSet    rowSet    = null;

    /**
     * Constructs a new <CODE>RowSetEventSupport</CODE> object that will
     * manage the listeners for the specified <CODE>RowSet</CODE>.
     * @param rs the <CODE>RowSet</CODE> to be managed
     */
    public RowSetEventSupport(RowSet rs) {
	super();
	rowSet = rs;
    }

    /**
     * Adds a listener to the list of listeners.
     * @param rsl the listener to be added
     */
    public void addListener(RowSetListener rsl) {
	listeners.add(rsl);
    }

    /**
     * Directs this object to notify the listeners that the
     * <CODE>RowSet</CODE> cursor has moved.
     */
    public void cursorMoved() {
	Iterator it = listeners.iterator();
	RowSetEvent event = new RowSetEvent(rowSet);
	
	while( it.hasNext() ) {
	    RowSetListener l = (RowSetListener)it.next();

	    l.cursorMoved(event);
	}
    }
    
    /**
     * Removes a listener from the list of listeners.
     * @param rsl the listener to be removed
     */
    public void removeListener(RowSetListener rsl) {
	listeners.remove(rsl);
    }

    /**
     * Directs this object to notify the listeners that the current row of the
     * <CODE>RowSet</CODE> cursor has been updated, inserted, or deleted.
     */
    public void rowChanged() {
	Iterator it = listeners.iterator();
	RowSetEvent event = new RowSetEvent(rowSet);
	
	while( it.hasNext() ) {
	    RowSetListener l = (RowSetListener)it.next();

	    l.rowChanged(event);
	}
    }
    
    /**
     * Directs this object to notify the listeners that the
     * <CODE>RowSet</CODE> cursor has loaded a new set of data.
     */
    public void rowSetChanged() {
	Iterator it = listeners.iterator();
	RowSetEvent event = new RowSetEvent(rowSet);
	
	while( it.hasNext() ) {
	    RowSetListener l = (RowSetListener)it.next();

	    l.cursorMoved(event);
	}
    }
}
