/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui.table;

import java.awt.Color;
import java.awt.Component;

import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.table.TableCellRenderer;

/**
 * The Class LabelTableCellRenderer.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class LabelTableCellRenderer extends JLabel implements TableCellRenderer {
  /** The supports focus border. */
  protected boolean           supportsFocusBorder;
  
  /** The tooltip enabled. */
  protected boolean           tooltipEnabled;
  
  /** The alternate row color enabled. */
  protected boolean           alternateRowColorEnabled = true;
  
  /** The alternate row color. */
  protected Color             alternateRowColor;

  /** The Constant DEFAULT_NO_FOCUS_BORDER. */
  private static final Border DEFAULT_NO_FOCUS_BORDER  = new EmptyBorder(1, 1, 1, 1);

  /** UID. */
  private static final long   serialVersionUID         = -6438957077711690615L;

  /**
   * Instantiates a new label table cell renderer.
   */
  public LabelTableCellRenderer() {
    this(true);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param tooltipEnabled the tooltip enabled
   */
  public LabelTableCellRenderer(boolean tooltipEnabled) {
    this.tooltipEnabled = tooltipEnabled;
    setOpaque(true);
    setSupportsFocusBorder(true);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   */
  public LabelTableCellRenderer(int halignment) {
    this(true);
    setHorizontalAlignment(halignment);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param valignment the valignment
   */
  public LabelTableCellRenderer(int halignment, int valignment) {
    this(true);
    setHorizontalAlignment(halignment);
    setVerticalAlignment(valignment);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param tooltipEnabled the tooltip enabled
   */
  public LabelTableCellRenderer(int halignment, boolean tooltipEnabled) {
    this(tooltipEnabled);
    setHorizontalAlignment(halignment);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param valignment the valignment
   * @param tooltipEnabled the tooltip enabled
   */
  public LabelTableCellRenderer(int halignment, int valignment, boolean tooltipEnabled) {
    this(tooltipEnabled);
    setHorizontalAlignment(halignment);
    setVerticalAlignment(valignment);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColorEnabled the alternate row color enabled
   */
  public LabelTableCellRenderer(boolean tooltipEnabled, boolean alternateRowColorEnabled) {
    this(tooltipEnabled);
    setAlternateRowColorEnabled(alternateRowColorEnabled);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param alternateRowColorEnabled the alternate row color enabled
   * @param halignment the halignment
   */
  public LabelTableCellRenderer(boolean alternateRowColorEnabled, int halignment) {
    this(halignment);
    setAlternateRowColorEnabled(alternateRowColorEnabled);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColorEnabled the alternate row color enabled
   */
  public LabelTableCellRenderer(int halignment, boolean tooltipEnabled, boolean alternateRowColorEnabled) {
    this(halignment, tooltipEnabled);
    setAlternateRowColorEnabled(alternateRowColorEnabled);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param valignment the valignment
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColorEnabled the alternate row color enabled
   */
  public LabelTableCellRenderer(int halignment, int valignment, boolean tooltipEnabled, boolean alternateRowColorEnabled) {
    this(halignment, valignment, tooltipEnabled);
    setAlternateRowColorEnabled(alternateRowColorEnabled);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param alternateRowColor the alternate row color
   */
  public LabelTableCellRenderer(Color alternateRowColor) {
    this(true);
    setAlternateRowColor(alternateRowColor);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColor the alternate row color
   */
  public LabelTableCellRenderer(boolean tooltipEnabled, Color alternateRowColor) {
    this(tooltipEnabled);
    setAlternateRowColor(alternateRowColor);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param alternateRowColor the alternate row color
   */
  public LabelTableCellRenderer(int halignment, Color alternateRowColor) {
    this(halignment);
    setAlternateRowColor(alternateRowColor);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColor the alternate row color
   */
  public LabelTableCellRenderer(int halignment, boolean tooltipEnabled, Color alternateRowColor) {
    this(halignment, tooltipEnabled);
    setAlternateRowColor(alternateRowColor);
  }

  /**
   * Instantiates a new label table cell renderer.
   *
   * @param halignment the halignment
   * @param valignment the valignment
   * @param tooltipEnabled the tooltip enabled
   * @param alternateRowColor the alternate row color
   */
  public LabelTableCellRenderer(int halignment, int valignment, boolean tooltipEnabled, Color alternateRowColor) {
    this(halignment, valignment, tooltipEnabled);
    setAlternateRowColor(alternateRowColor);
  }

  /**
   * Sets the supports focus border.
   *
   * @param supportsFocusBorder boolean
   */
  public void setSupportsFocusBorder(boolean supportsFocusBorder) {
    this.supportsFocusBorder = supportsFocusBorder;
    if (supportsFocusBorder) {
      setBorder(getNoFocusBorder());
    }
  }

  /**
   * Gets the supports focus border.
   *
   * @return boolean
   */
  public boolean getSupportsFocusBorder() {return supportsFocusBorder;}

  /**
   * Gets the no focus border.
   *
   * @return Border
   */
  protected static Border getNoFocusBorder() {
    Border border = UIManager.getBorder("Table.cellNoFocusBorder"); //$NON-NLS-1$
    if (border != null)
      return border;
    return DEFAULT_NO_FOCUS_BORDER;
  }

  /**
   * Checks if is tooltip enabled.
   *
   * @return boolean
   */
  public boolean isTooltipEnabled() {return tooltipEnabled;}

  /**
   * Sets the tooltip enabled.
   *
   * @param tooltipEnabled boolean
   */
  public void setTooltipEnabled(boolean tooltipEnabled) {
    this.tooltipEnabled = tooltipEnabled;
  }

  /**
   * Checks if is alternate row color enabled.
   *
   * @return boolean
   */
  public boolean isAlternateRowColorEnabled() {return alternateRowColorEnabled;}

  /**
   * Sets the alternate row color enabled.
   *
   * @param alternateRowColorEnabled boolean
   */
  public void setAlternateRowColorEnabled(boolean alternateRowColorEnabled) {
    this.alternateRowColorEnabled = alternateRowColorEnabled;
  }

  /**
   * Gets the alternate row color.
   *
   * @return Color
   */
  public Color getAlternateRowColor() { return alternateRowColor;}

  /**
   * Sets the alternate row color.
   *
   * @param alternateRowColor Color
   */
  public void setAlternateRowColor(Color alternateRowColor) {
    this.alternateRowColor = alternateRowColor;
    this.alternateRowColorEnabled = alternateRowColor != null;
  }

  /**
   * Sets the colors.
   *
   * @param table JTable
   * @param value Object
   * @param isSelected boolean
   * @param hasFocus boolean
   * @param row int
   * @param column int
   */
  protected void setColors(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    TableUtilities.setColors(this, table, isSelected, row, alternateRowColorEnabled, alternateRowColor);
  }

  /**
   * Sets the border.
   *
   * @param table JTable
   * @param value Object
   * @param isSelected boolean
   * @param hasFocus boolean
   * @param row int
   * @param column int
   */
  protected void setBorder(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
    TableUtilities.setBorder(this, table, isSelected, hasFocus, row, column, getNoFocusBorder());
  }

  /**
   * Sets the state.
   *
   * @param table JTable
   * @param value Object
   * @param isSelected boolean
   * @param hasFocus boolean
   * @param row int
   * @param column int
   */
  protected void setState(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
  //empty
  }

  /**
   * Sets the value.
   *
   * @param table JTable
   * @param value Object
   * @param row int
   * @param column int
   */
  protected void setValue(JTable table, Object value, int row, int column) {
    TableUtilities.setValue(this, value, tooltipEnabled);
  }

  /**
   * Sets the font.
   *
   * @param table JTable
   * @param value Object
   * @param row int
   * @param column int
   */
  protected void setFont(JTable table, Object value, int row, int column) {
    setFont(table.getFont());
  }

  /**
   * Gets the table cell renderer component.
   *
   * @param table the table
   * @param value the value
   * @param isSelected is selected
   * @param hasFocus has focus
   * @param row the row
   * @param column the column
   * @return the table cell renderer component
   * @see javax.swing.table.TableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
   */
  @Override
  public Component getTableCellRendererComponent(JTable table,
                                                 Object value,
                                                 boolean isSelected,
                                                 boolean hasFocus,
                                                 int row,
                                                 int column) {

    setColors(table, value, isSelected, hasFocus, row, column);

    setFont(table, value, row, column);

    if (supportsFocusBorder) {
      setBorder(table, value, isSelected, hasFocus, row, column);
    }

    setState(table, value, isSelected, hasFocus, row, column);

    setValue(table, value, row, column);

    return this;
  }

}
