/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.ui;

import java.awt.*;
import java.awt.event.MouseEvent;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JMenuItem;
import javax.swing.UIManager;

import jnpad.GUIUtilities;
import jnpad.action.JNPadAction;
import jnpad.ui.status.StatusDisplayable;
import jnpad.util.Platform;

/**
 * 
 * @version 0.3
 * @since   jNPad v0.1
 */
public class JNPadMenuItem extends JMenuItem {
  private String            _description;
  private boolean           _descriptionSet;
  private StatusDisplayable _displayable;

  private String            _keyBinding;

  static Font               acceleratorFont;
  static Color              acceleratorForeground;
  static Color              acceleratorSelectionForeground;
  static Color              acceleratorDisabledForeground;

  static {
    String shortcutFont;
    if (Platform.isMac) {
      shortcutFont = "Lucida Grande"; //$NON-NLS-1$
    }
    else {
      shortcutFont = "Monospaced"; //$NON-NLS-1$
    }
    acceleratorFont = UIManager.getFont("MenuItem.acceleratorFont"); //$NON-NLS-1$
    if (acceleratorFont == null) {
      acceleratorFont = new Font(shortcutFont, Font.PLAIN, 12);
    }
    acceleratorForeground = UIManager.getColor("MenuItem.acceleratorForeground"); //$NON-NLS-1$
    if (acceleratorForeground == null) {
      acceleratorForeground = Color.BLACK;
    }
    acceleratorSelectionForeground = UIManager.getColor("MenuItem.acceleratorSelectionForeground"); //$NON-NLS-1$
    if (acceleratorSelectionForeground == null) {
      acceleratorSelectionForeground = Color.BLACK;
    }
    acceleratorDisabledForeground = UIManager.getColor("MenuItem.disabledForeground"); //$NON-NLS-1$
    if (acceleratorDisabledForeground == null) {
      acceleratorDisabledForeground = Color.GRAY;
    }
  }
  
  /** UID */
  private static final long serialVersionUID = -65101014095861044L;

  /**
   *
   */
  public JNPadMenuItem() {
    //empty
  }

  /**
   *
   * @param text String
   */
  public JNPadMenuItem(String text) {
    super(text);
  }

  /**
   *
   * @param text String
   * @param mnemonic int
   */
  public JNPadMenuItem(String text, int mnemonic) {
    super(text, mnemonic);
  }

  /**
   *
   * @param icon Icon
   */
  public JNPadMenuItem(Icon icon) {
    super(icon);
  }

  /**
   *
   * @param text String
   * @param icon Icon
   */
  public JNPadMenuItem(String text, Icon icon) {
    super(text, icon);
  }

  /**
   *
   * @param a Action
   */
  public JNPadMenuItem(Action a) {
    super(a);
    if (a instanceof JNPadAction) {
      setText( ( (JNPadAction) a).getLabel());
      setDescription( ( (JNPadAction) a).getDescription());
      setKeyBinding( ( (JNPadAction) a).getKeyBinding());
    }
  }

  /**
   * 
   * @param keyBinding String
   */
  public void setKeyBinding(String keyBinding) {
    _keyBinding = keyBinding;
  }

  /**
   * 
   * @return String
   */
  public String getKeyBinding() {
    return _keyBinding;
  }

  /**
   * 
   * @return Dimension
   */
  @Override
  public Dimension getPreferredSize() {
    Dimension d = super.getPreferredSize();
    if (_keyBinding != null) {
      d.width += (getFontMetrics(acceleratorFont).stringWidth(_keyBinding) + 30);
    }
    return d;
  }
  
  /**
   *
   * @param description String
   */
  public void setDescription(String description) {
    _description = description;
  }

  /**
   *
   * @return String
   */
  public String getDescription() {
    return _description;
  }

  /**
   *
   * @param displayable StatusDisplayable
   */
  public void setStatusDisplayable(StatusDisplayable displayable) {
    _displayable = displayable;
  }

  /**
   *
   * @return StatusDisplayable
   */
  public StatusDisplayable getStatusDisplayable() {
    return _displayable;
  }

  /**
   *
   * @param e MouseEvent
   * @return StatusDisplayable
   */
  private StatusDisplayable getStatusDisplayable(MouseEvent e) {
    if (_displayable == null) {
      _displayable = GUIUtilities.getJNPadFrame(e);
    }
    return _displayable;
  }

  /**
   *
   * @param e MouseEvent
   */
  @Override
  protected void processMouseEvent(MouseEvent e) {
    if (_description != null) {
      switch (e.getID()) {
        case MouseEvent.MOUSE_ENTERED:
          getStatusDisplayable(e).setStatus(_description);
          _descriptionSet = true;
          break;
        case MouseEvent.MOUSE_EXITED:
        case MouseEvent.MOUSE_RELEASED:
          if (_descriptionSet) {
            getStatusDisplayable(e).clearStatus();
            _descriptionSet = false;
          }
          break;
        default: //Keep FindBugs happy
          break;
      }
    }
    super.processMouseEvent(e);
  }

  /**
   * 
   * @param g Graphics
   */
  @Override
  protected void paintComponent(Graphics g) {
    GUIUtilities.setRenderingHints(g);
    super.paintComponent(g);

    if (_keyBinding != null) {
      g.setFont(acceleratorFont);
      g.setColor(isEnabled() ? (getModel().isArmed() ? acceleratorSelectionForeground : acceleratorForeground) : acceleratorDisabledForeground);
      FontMetrics fm = g.getFontMetrics();
      Insets insets = getInsets();
      g.drawString(_keyBinding,
                  getWidth() - (fm.stringWidth(_keyBinding) + insets.right + insets.left + 5),
                  getFont().getSize() + (insets.top - (Platform.isMac ? 0 : 1)));
    }
    
  }
  
}
