/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.syntax;

import java.awt.Color;
import java.awt.Font;

import jnpad.config.Config;
import jnpad.util.Utilities;

/**
 * The Class SyntaxScheme.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public abstract class SyntaxScheme extends PlainScheme {
  static Color                commentColor;
  static Color                stringColor;
  static Color                numberColor;
  static Color                keywordColor;
  static Color                operatorColor;
  static Color                bracketColor;

  Font                        commentFont;
  Font                        stringFont;
  Font                        keywordFont;
  Font                        operatorFont;
  Font                        bracketFont;

  private static final char[] BRACKETS         = new char[] { '{', '}', '(', ')', '[', ']' };

  private static final char[] OPERATORS        = new char[] { '=' };

  private static final char[] DELIM_CHAR       = new char[] { '\'' };

  private static final char[] DELIM_STRING     = new char[] { '"' };

  /** UID */
  private static final long   serialVersionUID = 5558171778920408921L;

  /**
   * Instantiates a new syntax scheme.
   *
   * @param mini the mini
   */
  public SyntaxScheme(boolean mini) {
    super(mini);
    doUpdateColors();
    doUpdateFonts();
  }

  /**
   * Do update colors.
   */
  private void doUpdateColors() {
    commentColor  = Config.SYNTAX_COMMENT_COLOR.getValue();
    stringColor   = Config.SYNTAX_STRING_COLOR.getValue();
    numberColor   = Config.SYNTAX_NUMBER_COLOR.getValue();
    keywordColor  = Config.SYNTAX_KEYWORD_COLOR.getValue();
    operatorColor = Config.SYNTAX_OPERATOR_COLOR.getValue();
    bracketColor  = Config.SYNTAX_BRACKET_COLOR.getValue();
  }

  /**
   * Do update fonts.
   */
  private void doUpdateFonts() {
    commentFont  = textFont.deriveFont(Config.SYNTAX_COMMENT_STYLE.getValue());
    stringFont   = textFont.deriveFont(Config.SYNTAX_STRING_STYLE.getValue());
    keywordFont  = textFont.deriveFont(Config.SYNTAX_KEYWORD_STYLE.getValue());
    operatorFont = textFont.deriveFont(Config.SYNTAX_OPERATOR_STYLE.getValue());
    bracketFont  = textFont.deriveFont(Config.SYNTAX_BRACKET_STYLE.getValue());
  }
  
  /**
   * Sets the text font.
   *
   * @param f the new text font
   * @see jnpad.text.syntax.PlainScheme#setTextFont(java.awt.Font)
   */
  @Override
  public void setTextFont(Font f) {
    super.setTextFont(f);
    doUpdateFonts();
  }
  
  /**
   * Gets the font style.
   *
   * @param isBold is bold
   * @param isItalic is italic
   * @return the font style
   */
  static int getFontStyle(boolean isBold, boolean isItalic) {
    int i = 0;
    if (isBold)
      i++;
    if (isItalic)
      i += 2;
    return i;
  }
  
  /**
   * Configure.
   *
   * @param cfg the cfg
   * @see jnpad.text.Scheme#configure(int)
   */
  @Override
  public void configure(final int cfg) {
    super.configure(cfg);
    if ((cfg & CFG_COLOR) != 0) {
      doUpdateColors();
    }
    if ((cfg & CFG_FONT) != 0) {
      doUpdateFonts();
    }
  }

  /**
   * Gets the comment color.
   *
   * @return the comment color
   */
  public Color getCommentColor() {
    return commentColor;
  }

  /**
   * Gets the comment font.
   *
   * @return the comment font
   */
  public Font getCommentFont() {
    return commentFont;
  }

  /**
   * Gets the keyword color.
   *
   * @return the keyword color
   */
  public Color getKeywordColor() {
    return keywordColor;
  }

  /**
   * Gets the keyword font.
   *
   * @return the keyword font
   */
  public Font getKeywordFont() {
    return keywordFont;
  }

  /**
   * Gets the number color.
   *
   * @return the number color
   */
  public Color getNumberColor() {
    return numberColor;
  }
  
  /**
   * Gets the operator color.
   *
   * @return the operator color
   */
  public Color getOperatorColor() {
    return operatorColor;
  }

  /**
   * Gets the operator font.
   *
   * @return the operator font
   */
  public Font getOperatorFont() {
    return operatorFont;
  }

  /**
   * Gets the operators.
   *
   * @return the operators
   */
  public char[] getOperators() {
    //return OPERATORS;                 // Original
    //return OPERATORS.clone();         // Keep FindBugs happy [v0.1]
    return Utilities.copyOf(OPERATORS); // Keep FindBugs happy [v0.3]
  }
  
  /**
   * Gets the string color.
   *
   * @return the string color
   */
  public Color getStringColor() {
    return stringColor;
  }

  /**
   * Gets the string font.
   *
   * @return the string font
   */
  public Font getStringFont() {
    return stringFont;
  }
  
  /**
   * Gets the bracket color.
   *
   * @return the bracket color
   */
  public Color getBracketColor() {
    return bracketColor;
  }

  /**
   * Gets the bracket font.
   *
   * @return the bracket font
   */
  public Font getBracketFont() {
    return bracketFont;
  }
  
  /**
   * Gets the brackets.
   *
   * @return the brackets
   */
  public char[] getBrackets() {
    //return BRACKETS;                 // Original
    //return BRACKETS.clone();         // Keep FindBugs happy [v0.1]
    return Utilities.copyOf(BRACKETS); // Keep FindBugs happy [v0.3]
  }

  /**
   * Gets the char delimeters.
   *
   * @return the char delimeters
   */
  public char[] getCharDelimeters() {
    //return DELIM_CHAR;                 // Original
    //return DELIM_CHAR.clone();         // Keep FindBugs happy [v0.1]
    return Utilities.copyOf(DELIM_CHAR); // Keep FindBugs happy [v0.3]
  }
  
  /**
   * Gets the string delimiters.
   *
   * @return the string delimiters
   */
  public char[] getStringDelimiters() {
    //return DELIM_STRING;                 // Original
    //return DELIM_STRING.clone();         // Keep FindBugs happy [v0.1]
    return Utilities.copyOf(DELIM_STRING); // Keep FindBugs happy [v0.3]
  }

}
