/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.syntax;

import java.awt.Font;
import java.awt.Graphics;

import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.Segment;
import javax.swing.text.Utilities;

import jnpad.text.JNPadDocument;

/**
 * The Class PropertiesView.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class PropertiesView extends SyntaxView {
  PropertiesScheme scheme;
  
  /**
   * Instantiates a new properties view.
   *
   * @param elem the elem
   * @param mini the mini
   */
  public PropertiesView(Element elem, boolean mini) {
    super(elem, mini);
    scheme = !mini ? (PropertiesScheme) ((JNPadDocument) getDocument()).getScheme() :
        (PropertiesScheme) ((JNPadDocument) getDocument()).getMiniScheme();
  }

  /**
   * Do draw unselected text.
   *
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawUnselectedText(java.awt.Graphics, int, int, int, int)
   */
  @Override
  int doDrawUnselectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (isSingleLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setColor(scheme.getCommentColor());
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isOperator(indexedChar)) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        g.setColor(scheme.getOperatorColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);

        wordIndex++;

        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        g.setFont(scheme.getValueFont());
        g.setColor(scheme.getValueColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;
        
        continue;
      }
      
      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      g.setColor(scheme.getTextColor());
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
      
      wordIndex++;
    }

    return x;
  }

  /**
   * Do draw selected text.
   *
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawSelectedText(java.awt.Graphics, int, int, int, int)
   */
  @Override
  int doDrawSelectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    g.setColor(scheme.getSelectedTextColor());

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (isSingleLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
      
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isOperator(indexedChar)) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);

        wordIndex++;

        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        g.setFont(scheme.getValueFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;
        
        continue;
      }
      
      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
      
      wordIndex++;
    }

    return x;
  }

  /**
   * Checks if is single line comment start.
   *
   * @param s the Segment
   * @param index the index
   * @return true, if is single line comment start
   * @see jnpad.text.syntax.SyntaxView#isSingleLineCommentStart(javax.swing.text.Segment, int)
   */
  @Override
  boolean isSingleLineCommentStart(Segment s, int index) {
      return s.charAt(index) == '#';
  }
  
  /**
   * Gets the scheme.
   *
   * @return the scheme
   * @see jnpad.text.syntax.SyntaxView#getScheme()
   */
  @Override
  SyntaxScheme getScheme() {
    return scheme;
  }

}
