/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text;

import java.awt.Component;

import jnpad.text.syntax.ContentTypes;
import jnpad.util.Platform;
import jnpad.util.Utilities;

/**
 * The Class MockBuffer.
 *
 * @version 0.3
 * @since jNPad 0.1
 */
public final class MockBuffer extends Component implements IBuffer {
  /** The path. */
  private String            path;

  /** UID */
  private static final long serialVersionUID = 2368812646959536859L;

  /**
   * Instantiates a new mock buffer.
   *
   * @param path String
   */
  public MockBuffer(String path) {
    setFilePath(path);
  }

  /**
   * Instantiates a new mock buffer.
   *
   * @param buffer the buffer
   */
  public MockBuffer(IBuffer buffer) {
    this(buffer.getFilePath());
  }

  /**
   * Sets the file path.
   *
   * @param path the new file path
   * @see jnpad.text.IBuffer#setFilePath(java.lang.String)
   */
  @Override
  public void setFilePath(String path) {
    this.path = Utilities.defaultString(path);
  }

  /**
   * Gets the file path.
   *
   * @return the file path
   * @see jnpad.text.IBuffer#getFilePath()
   */
  @Override
  public String getFilePath() {
    return path;
  }

  /**
   * Gets the content type.
   *
   * @return the content type
   * @see jnpad.text.IBuffer#getContentType()
   */
  @Override
  public String getContentType() {
    return ContentTypes.PLAIN;
  }

  /**
   * Gets the tool tip text.
   *
   * @return the tool tip text
   * @see jnpad.text.IBuffer#getToolTipText()
   */
  @Override
  public String getToolTipText() {
    return path;
  }

  /**
   * Equals.
   *
   * @param obj the obj
   * @return true, if successful
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object obj) {
    return obj instanceof IBuffer &&
        Utilities.equals(path, ((IBuffer) obj).getFilePath(), !Platform.isWindows);
  }

  /**
   * Hash code.
   *
   * @return the int
   * @see java.lang.Object#hashCode()
   */
  @Override
  public int hashCode() {
    return path.hashCode();
  }

  /**
   * Checks if is read only.
   *
   * @return true, if is read only
   * @see jnpad.text.IBuffer#isReadOnly()
   */
  @Override
  public boolean isReadOnly() {
    return true;
  }

  /**
   * Checks if is line wrap.
   *
   * @return true, if is line wrap
   * @see jnpad.text.IBuffer#isLineWrapped()
   */
  @Override
  public boolean isLineWrapped() {
    return false;
  }
  
  /**
   * Checks if is dirty.
   *
   * @return true, if is dirty
   * @see jnpad.text.IBuffer#isDirty()
   */
  @Override
  public boolean isDirty() {
    return false;
  }

  /**
   * Checks for selection.
   *
   * @return true, if successful
   * @see jnpad.text.IBuffer#hasSelection()
   */
  @Override
  public boolean hasSelection() {
    return false;
  }

  /**
   * Gets the caret position.
   *
   * @return the caret position
   * @see jnpad.text.IBuffer#getCaretPosition()
   */
  @Override
  public int getCaretPosition() {
    return 0;
  }

  /**
   * Reload.
   *
   * @see jnpad.text.IBuffer#reload()
   */
  @Override
  public void reload() {
    //empty
  }

  /**
   * Update controls.
   *
   * @see jnpad.config.Updatable#updateControls()
   */
  @Override
  public void updateControls() {
    updateControls(CTRLS_ALL);
  }

  /**
   * Update controls.
   *
   * @param ctrls the ctrls
   * @see jnpad.config.Updatable#updateControls(int)
   */
  @Override
  public void updateControls(final int ctrls) {
    //empty
  }

  /**
   * Configure.
   *
   * @param cfg the cfg
   * @see jnpad.config.Configurable#configure(int)
   */
  @Override
  public void configure(final int cfg) {
    //empty
  }

  /**
   * To string.
   *
   * @return the string
   * @see java.awt.Component#toString()
   */
  @Override
  public String toString() {
    return path;
  }

}
