/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.action;

import static jnpad.util.Utilities.SPACE_STRING;

import java.awt.event.ItemEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jnpad.GUIUtilities;
import jnpad.JNPadFrame;
import jnpad.GUIUtilities.FontSize;
import jnpad.action.JNPadActions.Group;
import jnpad.config.Config;
import jnpad.config.Configurable;

/**
 * The Class SelectFontSizeAction.
 *
 * @version 0.3
 * @since jNPad 0.3
 */
public final class SelectFontSizeAction extends JNPadAction {
  private FontSize                        fontSize;

  static final String                     NAME             = "select-fontsize";                 //$NON-NLS-1$

  private static Map<String, JNPadAction> map              = new HashMap<String, JNPadAction>();

  /** UID */
  private static final long               serialVersionUID = -4771867648034883226L;

  /**
   * Instantiates a new select font size action.
   *
   * @param jNPad the jNPad's frame
   * @param fontSize the font size
   */
  public SelectFontSizeAction(JNPadFrame jNPad, FontSize fontSize) {
    super(jNPad, NAME, Group.CONFIGURE);
    this.fontSize = fontSize;
    
    String actionLabel, description, name;
    boolean selected;
    
    final int fsize1 = Config.TEXT_FONT.getValue().getSize();
    final int fsize2 = Config.TEXT_FONT.getDefault().getSize();

    switch(fontSize) {
      case TINY:
        actionLabel = ActionBundle.getString(NAME.concat(".tiny.label")); //$NON-NLS-1$
        description = ActionBundle.getString(NAME.concat(".tiny.description"), "tiny"); //$NON-NLS-1$ //$NON-NLS-2$
        name        = NAME + SPACE_STRING + "tiny"; //$NON-NLS-1$
        selected    = fsize1 == fsize2 - 4;
        break;
      case SMALL:
        actionLabel = ActionBundle.getString(NAME.concat(".small.label")); //$NON-NLS-1$
        description = ActionBundle.getString(NAME.concat(".small.description"), "small"); //$NON-NLS-1$ //$NON-NLS-2$
        name        = NAME + SPACE_STRING + "small"; //$NON-NLS-1$
        selected    = fsize1 == fsize2 - 2;
        break;
      case LARGE:
        actionLabel = ActionBundle.getString(NAME.concat(".large.label")); //$NON-NLS-1$
        description = ActionBundle.getString(NAME.concat(".large.description"), "large"); //$NON-NLS-1$ //$NON-NLS-2$
        name        = NAME + SPACE_STRING + "large"; //$NON-NLS-1$
        selected    = fsize1 == fsize2 + 2;
        break;
      case HUGE:
        actionLabel = ActionBundle.getString(NAME.concat(".huge.label")); //$NON-NLS-1$
        description = ActionBundle.getString(NAME.concat(".huge.description"), "huge"); //$NON-NLS-1$ //$NON-NLS-2$
        name        = NAME + SPACE_STRING + "huge"; //$NON-NLS-1$
        selected    = Config.isDefaultMode();
        selected    = fsize1 == fsize2 + 4;
        break;
      case NORMAL:
      default:  
        actionLabel = ActionBundle.getString(NAME.concat(".normal.label")); //$NON-NLS-1$
        description = ActionBundle.getString(NAME.concat(".normal.description"), "normal"); //$NON-NLS-1$ //$NON-NLS-2$
        name        = NAME + SPACE_STRING + "normal"; //$NON-NLS-1$
        selected    = fsize1 == fsize2;
        break;
    }

    setName(name);
    setActionLabel(actionLabel);
    setDescription(description);

    setSelected(selected);
    setStateAction(true);
    
    map.put(name, this);
  }

  /**
   * Handle state changed.
   *
   * @param e the ItemEvent
   * @see jnpad.action.JNPadAction#handleStateChanged(java.awt.event.ItemEvent)
   */
  @Override
  public void handleStateChanged(final ItemEvent e) {
    if (ItemEvent.SELECTED == e.getStateChange()) {
      Config.TEXT_FONT.setValue(GUIUtilities.getFont(fontSize, Config.TEXT_FONT.getDefault()));
      jNPad.configure(Configurable.CFG_FONT);
    }
  }

  /**
   * Gets the completions for prefix.
   * 
   * @param prefix the prefix
   * @return the completions for prefix
   */
  static List<String> getCompletionsForPrefix(String prefix) {
    List<String> list = new ArrayList<String>();
    for (String key : map.keySet()) {
      JNPadAction action = map.get(key);
      if (key.startsWith(prefix) && action.isEnabled()) {
        list.add(key);
      }
    }
    Collections.sort(list);
    return list;
  }

  /**
   * Perform action.
   *
   * @param name the name
   * @return true, if successful
   */
  static boolean performAction(String name) {
    JNPadAction action = map.get(name);
    if (action != null && action.isEnabled()) {
      action.setSelected(!action.isSelected());
      return true;
    }
    return false;
  }
  
}
