/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.action;

import static jnpad.action.JNPadActions.ACTION_NAME_JOIN_LINES;

import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.text.Document;

import jnpad.JNPadFrame;
import jnpad.action.JNPadActions.Group;
import jnpad.config.Accelerators;
import jnpad.text.JNPadTextArea;
import jnpad.text.Buffer;
import jnpad.text.IEditAction;
import jnpad.ui.ReporterUtilities;
import jnpad.ui.status.StatusDisplayable.StatusType;
import jnpad.util.Utilities;

/**
 * The Class JoinLinesAction.
 *
 * @version 0.3
 * @since jNPad 0.1
 */
public final class JoinLinesAction extends JNPadAction implements IEditAction {
  /** Logger */
  private static final Logger LOGGER           = Logger.getLogger(JoinLinesAction.class.getName());

  /** UID */
  private static final long   serialVersionUID = -7779330402635811907L;

  /**
   * Instantiates a new join lines action.
   *
   * @param jNPad the jNPad's frame
   */
  public JoinLinesAction(JNPadFrame jNPad) {
    super(jNPad,
          ACTION_NAME_JOIN_LINES,
          Group.EDIT,
          Accelerators.JOIN_LINES, null);
  }

  /**
   * Perform action.
   *
   * @see jnpad.action.JNPadAction#performAction()
   */
  @Override
  public void performAction() {
    Buffer buffer = jNPad.getActiveBuffer();

    if (buffer != null) {
      try {
        buffer.beginCompoundEdit();

        JNPadTextArea textArea = buffer.getSelectedTextArea();
        Document doc = textArea.getDocument();

        int start, len;
        String text;
        boolean finalSpace;

        if (textArea.hasSelection()) {
          start      = textArea.getSelectionStart();
          text       = textArea.getSelectedText();
          len        = textArea.getSelectionEnd() - start;
          finalSpace = doc.getLength() != textArea.getSelectionEnd();
        }
        else {
          start      = 0;
          text       = textArea.getText();
          len        = doc.getLength();
          finalSpace = false;
        }

        // ------
        if (len > 5000) {
          ReporterUtilities.reportInfo(jNPad, 
                                       ActionBundle.getString(ACTION_NAME_JOIN_LINES.concat(".tooBig")), //$NON-NLS-1$
                                       getLabel());
          return;
        }
        // ------

        StringTokenizer st = new StringTokenizer(text, Utilities.LF_STRING);

        doc.remove(start, len);
        int offset = start;
        while (st.hasMoreTokens()) {
          String insert = st.nextToken().trim();
          if (st.hasMoreTokens() || finalSpace) {
            insert += Utilities.SPACE;
          }
          doc.insertString(offset, insert, null);
          offset += insert.length();
        }
        textArea.setCaretPosition(offset);
      }
      catch (Exception ex) {
        LOGGER.log(Level.WARNING, ex.getMessage(), ex);
        jNPad.setStatus(StatusType.ERROR, "Error - ".concat(getLabel())); //$NON-NLS-1$
      }
      finally {
        buffer.endCompoundEdit();
      }
    }
  }

}
