/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import jnpad.config.Config;
import jnpad.util.Utilities;

/**
 * The Class JNPadParser.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
public class JNPadParser {
  private String              file;
  private List<JNPadInput>    jNPadInputs = new ArrayList<JNPadInput>();

  /** Logger */
  private static final Logger LOGGER      = Logger.getLogger(JNPadParser.class.getName());

  /**
   * Instantiates a new jNPad parser.
   *
   * @param file the file
   */
  public JNPadParser(String file) {
    this.file = file;
  }

  /**
   * Load.
   *
   * @throws IOException Signals that an I/O exception has occurred.
   */
  public void load() throws IOException {
    clear();
    BufferedReader in = null;
    try {
      in = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8")); //$NON-NLS-1$
      String line;
      while ((line = in.readLine()) != null) {
        line = line.trim();
        if (line.length() < 1 || line.startsWith("#")) { // lnea en blanco o comentario de lnea //$NON-NLS-1$
          continue;
        }
        int i = line.indexOf('#');
        if (i != Utilities.INDEX_NOT_FOUND) {
          line = line.substring(0, i);
        }
        try {
          jNPadInputs.add(Config.parseJNPadInput(line));
        }
        catch (IllegalArgumentException ex) {
          LOGGER.warning(ex.getMessage());
        }
      }
    }
    catch (IOException ex) {
      LOGGER.log(Level.WARNING, ex.getMessage(), ex);
    }
    finally {
      try {
        if (in != null) {
          in.close();
        }
      }
      catch (IOException ex) {
        LOGGER.log(Level.WARNING, ex.getMessage(), ex);
      }
    }
  }

  /**
   *
   * @param in JNPadInput
   */
  public void add(JNPadInput in) {
    jNPadInputs.add(in);
  }

  /**
   * Adds the all.
   *
   * @param c the Collection
   */
  public void addAll(Collection<? extends JNPadInput> c) {
    jNPadInputs.addAll(c);
  }

  /**
   * Clear.
   */
  public void clear() {
    jNPadInputs.clear();
  }

  /**
   * Gets the jNPad input list.
   *
   * @return the jNPad input list
   */
  public List<JNPadInput> getJNPadInputList() {
    return jNPadInputs;
  }

  /**
   * Save.
   *
   * @throws IOException Signals that an I/O exception has occurred.
   */
  public void save() throws IOException {
    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file), "UTF-8")); //$NON-NLS-1$

      pw.println("# ==========================================================================="); //$NON-NLS-1$
      pw.println("# JNPad session"); //$NON-NLS-1$
      pw.println("# ==========================================================================="); //$NON-NLS-1$
      for (JNPadInput jNPadInput : jNPadInputs) {
        pw.println(Config.writeJNPadInputProp(jNPadInput));
      }
      pw.println("# ==========================================================================="); //$NON-NLS-1$
    } 
	finally {
      if (pw != null) {
        try {
          pw.close();
        }
        catch (Exception ex) {
          //ignored
        }
      }
    }

  }

}
