/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad;

import java.awt.Frame;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.ButtonGroup;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JRootPane;
import javax.swing.KeyStroke;

import jnpad.action.*;
import jnpad.config.Accelerators;
import jnpad.config.Config;

/**
 * The Class JNPadActionHelper.
 *
 * @version 0.3
 * @since   jNPad v0.1
 */
final class JNPadActionHelper {
  // File
  JNPadAction        newFileAction;
  JNPadAction        openFileAction;
  JNPadAction        openAllRecentFilesAction;
  JNPadAction        emptyRecentFilesListAction;
  JNPadAction        closeFileAction;
  JNPadAction        closeFilesAction;
  JNPadAction        closeAllAction;
  JNPadAction        closeOtherAction;
  JNPadAction        closeAllToLeftAction;
  JNPadAction        closeAllToRightAction;
  JNPadAction        saveFileAction;
  JNPadAction        saveFileAsAction;
  JNPadAction        saveAllFilesAction;
  JNPadAction        fileBrowserAction;
  JNPadAction        filePropertiesAction;
  JNPadAction        printSetAction;
  JNPadAction        printAction;
  JNPadAction        reloadAction;
  JNPadAction        executeActionAction;
  JNPadAction        loadSessionAction;
  JNPadAction        saveSessionAction;
  JNPadAction        reduceAsTrayIconAction;
  JNPadAction        restartAction;
  JNPadAction        exitAction;
  
  // Edit
  JNPadAction        undoAction;
  JNPadAction        redoAction;
  JNPadAction        copyAction;
  JNPadAction        copyLineAction;
  JNPadAction        cutAction;
  JNPadAction        cutLineAction;
  JNPadAction        pasteAction;
  JNPadAction        selectAllAction;
  JNPadAction        selectParagraphAction;
  JNPadAction        selectionBeginParagraphAction;
  JNPadAction        selectionEndParagraphAction;
  JNPadAction        selectLineAction;
  JNPadAction        selectionBeginLineAction;
  JNPadAction        selectionEndLineAction;
  JNPadAction        selectWordAction;
  JNPadAction        deleteAction;
  JNPadAction        deleteParagraphAction;
  JNPadAction        deletionBeginParagraphAction;
  JNPadAction        deletionEndParagraphAction;
  JNPadAction        deleteLineAction;
  JNPadAction        deletionBeginLineAction;
  JNPadAction        deletionEndLineAction;
  JNPadAction        deleteWordAction;
  JNPadAction        tabsToSpacesAction;
  JNPadAction        spacesToTabsAction;
  JNPadAction        beginLinesAction;
  JNPadAction        endLinesAction;
  JNPadAction        duplicateCurrentLineAction;
  JNPadAction        joinLinesAction;
  JNPadAction        sortLinesAscendingAction;
  JNPadAction        sortLinesDescendingAction;
  JNPadAction        removeEndSpacesAction;
  JNPadAction        removeEmptyLinesAction;
  JNPadAction        completeWordAction;
  JNPadAction        completeWordAllAction;
  JNPadAction        wordCountAction;
  JNPadAction        insertFileExtensionAction;
  JNPadAction        insertFileFullNameAction;
  JNPadAction        insertFileNameAction;
  JNPadAction        insertFileFullPathAction;
  JNPadAction        insertFilePathAction;
  JNPadAction        insertDateAction;

  // Search
  JNPadAction        findAction;
  JNPadAction        findNextAction;
  JNPadAction        findPreviousAction;
  JNPadAction        replaceAction;
  JNPadAction        replaceNextAction;
  JNPadAction        incrementalSearchAction;
  JNPadAction        highlightAllOccurrencesAction;
  JNPadAction        clearAllOccurrencesAction;
  JNPadAction        nextOccurrenceAction;
  JNPadAction        previousOccurrenceAction;
  JNPadAction        goToLineAction;
  JNPadAction        nextBookmarkAction;
  JNPadAction        previousBookmarkAction;
  JNPadAction        toggleBookmarkAction;
  JNPadAction        clearAllBookmarksAction;

  // View
  JNPadAction        toolBarVisibleAction;
  JNPadAction        toolbarsFileAction;
  JNPadAction        toolbarsEditRecantationAction;
  JNPadAction        toolbarsEditClipboardAction;
  JNPadAction        toolbarsEditSelectionAction;
  JNPadAction        toolbarsEditDeletionAction;
  JNPadAction        toolbarsSearchAction;
  JNPadAction        toolbarsViewAction;
  JNPadAction        toolbarsFormatAction;
  JNPadAction        toolbarsWindowAction;
  JNPadAction        toolbarsAllVisibleAction;
  JNPadAction        toolbarsNoneVisibleAction;
  JNPadAction        findResultsVisibleAction;
  JNPadAction        statusBarVisibleAction;
  JNPadAction        lineWrapAction;
  JNPadAction        lineNumbersAction;
  JNPadAction        activeLineAction;
  JNPadAction        rightMarginLineAction;
  JNPadAction        markStripAction;
  JNPadAction        markOccurrencesAction;
  JNPadAction        markBracketsAction;
  JNPadAction        focusOnMainViewAction;
  JNPadAction        selectScopeAsViewerAction;
  JNPadAction        selectScopeAsBufferSetAction;
  JNPadAction        selectScopeAsBufferAction;
  JNPadAction        selectScopeAsEditPaneAction;
  JNPadAction        fullScreenAction;

  // Format
  JNPadAction        selectEOLAsDOSAction;
  JNPadAction        selectEOLAsUnixAction;
  JNPadAction        selectEOLAsMacAction;
  JNPadAction        rightIndentAction;
  JNPadAction        leftIndentAction;
  JNPadAction        toUpperCaseAction;
  JNPadAction        toLowerCaseAction;
  JNPadAction        invertUpperLowerAction;
  JNPadAction        capitalizeAction;
  JNPadAction        toTitleAction;
  JNPadAction        setReadOnlyAction;
  JNPadAction        alignStringsByCommaAction;
  JNPadAction        alignStringsByEqualAction;
  JNPadAction        alignStringsByClipboardAction;
  JNPadAction        increaseFontAction;
  JNPadAction        decreaseFontAction;

  // Configure
  JNPadAction        autoCompletionAction;
  JNPadAction        autoIndentAction;
  JNPadAction        indentUsingSpacesAction;
  JNPadAction        loadOptionsFileAction;
  JNPadAction        loadDefaultOptionsFileAction;
  JNPadAction        loadShortcutsFileAction;
  JNPadAction        loadLoggingConfigFileAction;
  JNPadAction        fontsAction;
  JNPadAction        resetFontAction;
  JNPadAction        foregroundAction;
  JNPadAction        brightenForegroundAction;
  JNPadAction        darkenForegroundAction;
  JNPadAction        resetForegroundAction;
  JNPadAction        backgroundAction;
  JNPadAction        brightenBackgroundAction;
  JNPadAction        darkenBackgroundAction;
  JNPadAction        resetBackgroundAction;

  // Window
  JNPadAction        windowsAction;
  JNPadAction        nextBufferAction;
  JNPadAction        previousBufferAction;
  JNPadAction        splitHorizontallyAction;
  JNPadAction        splitVerticallyAction;
  JNPadAction        unsplitCurrentAction;
  JNPadAction        unsplitAction;
  JNPadAction        restoreSplitAction;
  JNPadAction        previousEditPaneAction;
  JNPadAction        nextEditPaneAction;
  JNPadAction        splitViewerHorizontallyAction;
  JNPadAction        splitViewerVerticallyAction;
  JNPadAction        unsplitViewerCurrentAction;
  JNPadAction        unsplitViewerAction;
  JNPadAction        restoreViewerSplitAction;
  JNPadAction        previousBufferSetAction;
  JNPadAction        nextBufferSetAction;

  // Help
  JNPadAction        cliArgsAction;
  JNPadAction        licenceAction;
  JNPadAction        check4UpdatesAction;
  JNPadAction        aboutAction;

  private JNPadFrame jNPad;

  /**
   * Instantiates a new jNPad action helper.
   *
   * @param jNPad the jNPad's frame
   */
  JNPadActionHelper(JNPadFrame jNPad) {
    this.jNPad = jNPad;

    registerActions();
  }

  /**
   * Register actions.
   */
  void registerActions() {
    ActionManager manager = ActionManager.INSTANCE;

	manager.registerAction(newFileAction                = new NewFileAction(jNPad));
	manager.registerAction(openFileAction               = new OpenFileAction(jNPad));
	manager.registerAction(openAllRecentFilesAction     = new OpenAllRecentFilesAction(jNPad));
	manager.registerAction(emptyRecentFilesListAction   = new EmptyRecentFilesListAction(jNPad));
	manager.registerAction(closeFileAction              = new CloseFileAction(jNPad));
	manager.registerAction(closeFilesAction             = new CloseFilesAction(jNPad));
	manager.registerAction(closeAllAction               = new CloseAllAction(jNPad));
	manager.registerAction(closeOtherAction             = new CloseOtherAction(jNPad));
	manager.registerAction(closeAllToLeftAction         = new CloseAllToLeftAction(jNPad));
	manager.registerAction(closeAllToRightAction        = new CloseAllToRightAction(jNPad));
	manager.registerAction(saveFileAction               = new SaveFileAction(jNPad));
	manager.registerAction(saveAllFilesAction           = new SaveAllFilesAction(jNPad));
	manager.registerAction(saveFileAsAction             = new SaveFileAsAction(jNPad));
	manager.registerAction(fileBrowserAction            = new FileBrowserAction(jNPad));
	manager.registerAction(filePropertiesAction         = new FilePropertiesAction(jNPad));
	manager.registerAction(printSetAction               = new PrintSetAction(jNPad));
	manager.registerAction(printAction                  = new PrintAction(jNPad));
	manager.registerAction(reloadAction                 = new ReloadAction(jNPad));
    /*manager.registerAction(*/executeActionAction      = new ExecuteActionAction(jNPad)/*)*/;
	manager.registerAction(loadSessionAction            = new LoadSessionAction(jNPad));
	manager.registerAction(saveSessionAction            = new SaveSessionAction(jNPad));
	manager.registerAction(reduceAsTrayIconAction       = new ReduceAsTrayIconAction(jNPad));
    manager.registerAction(restartAction                = new RestartAction(jNPad));
	manager.registerAction(exitAction                   = new ExitAction(jNPad));

	manager.registerAction(undoAction                   = new UndoAction(jNPad));
	manager.registerAction(redoAction                   = new RedoAction(jNPad));
	manager.registerAction(cutAction                    = new CutAction(jNPad));
	manager.registerAction(cutLineAction                = new CutLineAction(jNPad));
	manager.registerAction(copyAction                   = new CopyAction(jNPad));
	manager.registerAction(copyLineAction               = new CopyLineAction(jNPad));
	manager.registerAction(pasteAction                  = new PasteAction(jNPad));
	manager.registerAction(selectAllAction              = new SelectAllAction(jNPad));
	manager.registerAction(selectParagraphAction        = new SelectParagraphAction(jNPad));
	manager.registerAction(selectionBeginParagraphAction= new SelectionBeginParagraphAction(jNPad));
	manager.registerAction(selectionEndParagraphAction  = new SelectionEndParagraphAction(jNPad));
	manager.registerAction(selectLineAction             = new SelectLineAction(jNPad));
	manager.registerAction(selectionBeginLineAction     = new SelectionBeginLineAction(jNPad));
	manager.registerAction(selectionEndLineAction       = new SelectionEndLineAction(jNPad));
	manager.registerAction(selectWordAction             = new SelectWordAction(jNPad));
	manager.registerAction(deleteAction                 = new DeleteAction(jNPad));
	manager.registerAction(deleteParagraphAction        = new DeleteParagraphAction(jNPad));
	manager.registerAction(deletionBeginParagraphAction = new DeletionBeginParagraphAction(jNPad));
	manager.registerAction(deletionEndParagraphAction   = new DeletionEndParagraphAction(jNPad));
	manager.registerAction(deleteLineAction             = new DeleteLineAction(jNPad));
	manager.registerAction(deletionBeginLineAction      = new DeletionBeginLineAction(jNPad));
	manager.registerAction(deletionEndLineAction        = new DeletionEndLineAction(jNPad));
	manager.registerAction(deleteWordAction             = new DeleteWordAction(jNPad));
	manager.registerAction(tabsToSpacesAction           = new TabsToSpacesAction(jNPad));
	manager.registerAction(spacesToTabsAction           = new SpacesToTabsAction(jNPad));
	manager.registerAction(beginLinesAction             = new BeginLinesAction(jNPad));
	manager.registerAction(endLinesAction               = new EndLinesAction(jNPad));
	manager.registerAction(duplicateCurrentLineAction   = new DuplicateCurrentLineAction(jNPad));
	manager.registerAction(joinLinesAction              = new JoinLinesAction(jNPad));
	manager.registerAction(sortLinesAscendingAction     = new SortLinesAscendingAction(jNPad));
	manager.registerAction(sortLinesDescendingAction    = new SortLinesDescendingAction(jNPad));
	manager.registerAction(removeEndSpacesAction        = new RemoveEndSpacesAction(jNPad));
	manager.registerAction(removeEmptyLinesAction       = new RemoveEmptyLinesAction(jNPad));
	manager.registerAction(completeWordAction           = new CompleteWordAction(jNPad));
	manager.registerAction(completeWordAllAction        = new CompleteWordAllAction(jNPad));
	manager.registerAction(wordCountAction              = new WordCountAction(jNPad));
	manager.registerAction(insertFileExtensionAction    = new InsertFileExtensionAction(jNPad));
	manager.registerAction(insertFileFullNameAction     = new InsertFileFullNameAction(jNPad));
	manager.registerAction(insertFileNameAction         = new InsertFileNameAction(jNPad));
	manager.registerAction(insertFileFullPathAction     = new InsertFileFullPathAction(jNPad));
	manager.registerAction(insertFilePathAction         = new InsertFilePathAction(jNPad));
	manager.registerAction(insertDateAction             = new InsertDateAction(jNPad));

	manager.registerAction(findAction                   = new FindAction(jNPad));
	manager.registerAction(findNextAction               = new FindNextAction(jNPad));
	manager.registerAction(findPreviousAction           = new FindPreviousAction(jNPad));
	manager.registerAction(replaceAction                = new ReplaceAction(jNPad));
	manager.registerAction(replaceNextAction            = new ReplaceNextAction(jNPad));
	manager.registerAction(incrementalSearchAction      = new IncrementalSearchAction(jNPad));
	manager.registerAction(highlightAllOccurrencesAction= new HighlightAllOccurrencesAction(jNPad));
	manager.registerAction(clearAllOccurrencesAction    = new ClearAllOccurrencesAction(jNPad));
	manager.registerAction(nextOccurrenceAction         = new NextOccurrenceAction(jNPad));
	manager.registerAction(previousOccurrenceAction     = new PreviousOccurrenceAction(jNPad));
	manager.registerAction(goToLineAction               = new GoToLineAction(jNPad));
	manager.registerAction(nextBookmarkAction           = new NextBookmarkAction(jNPad));
	manager.registerAction(previousBookmarkAction       = new PreviousBookmarkAction(jNPad));
	manager.registerAction(toggleBookmarkAction         = new ToggleBookmarkAction(jNPad));
	manager.registerAction(clearAllBookmarksAction      = new ClearAllBookmarksAction(jNPad));

	manager.registerAction(toolBarVisibleAction         = new ToolBarVisibleAction(jNPad));
	manager.registerAction(toolbarsFileAction           = new ToolbarsFileAction(jNPad));
	manager.registerAction(toolbarsEditRecantationAction= new ToolbarsEditRecantationAction(jNPad));
	manager.registerAction(toolbarsEditClipboardAction  = new ToolbarsEditClipboardAction(jNPad));
	manager.registerAction(toolbarsEditSelectionAction  = new ToolbarsEditSelectionAction(jNPad));
	manager.registerAction(toolbarsEditDeletionAction   = new ToolbarsEditDeletionAction(jNPad));
	manager.registerAction(toolbarsSearchAction         = new ToolbarsSearchAction(jNPad));
	manager.registerAction(toolbarsViewAction           = new ToolbarsViewAction(jNPad));
	manager.registerAction(toolbarsFormatAction         = new ToolbarsFormatAction(jNPad));
	manager.registerAction(toolbarsWindowAction         = new ToolbarsWindowAction(jNPad));
	manager.registerAction(toolbarsAllVisibleAction     = new ToolbarsAllVisibleAction(jNPad));
	manager.registerAction(toolbarsNoneVisibleAction    = new ToolbarsNoneVisibleAction(jNPad));
	manager.registerAction(findResultsVisibleAction     = new FindResultsVisibleAction(jNPad));
	manager.registerAction(statusBarVisibleAction       = new StatusBarVisibleAction(jNPad));
	manager.registerAction(lineWrapAction               = new LineWrapAction(jNPad));
	manager.registerAction(lineNumbersAction            = new LineNumbersAction(jNPad));
	manager.registerAction(activeLineAction             = new ActiveLineAction(jNPad));
	manager.registerAction(rightMarginLineAction        = new RightMarginLineAction(jNPad));
    manager.registerAction(markStripAction              = new MarkStripAction(jNPad));
	manager.registerAction(markOccurrencesAction        = new MarkOccurrencesAction(jNPad));
    manager.registerAction(markBracketsAction           = new MarkBracketsAction(jNPad));
	manager.registerAction(focusOnMainViewAction        = new FocusOnMainViewAction(jNPad));
	manager.registerAction(selectScopeAsViewerAction    = new SelectScopeAsViewerAction(jNPad));
	manager.registerAction(selectScopeAsBufferSetAction = new SelectScopeAsBufferSetAction(jNPad));
	manager.registerAction(selectScopeAsBufferAction    = new SelectScopeAsBufferAction(jNPad));
	manager.registerAction(selectScopeAsEditPaneAction  = new SelectScopeAsEditPaneAction(jNPad));
	manager.registerAction(fullScreenAction             = new FullScreenAction(jNPad));

    manager.registerAction(selectEOLAsDOSAction         = new SelectEOLAsDOSAction(jNPad));
	manager.registerAction(selectEOLAsUnixAction        = new SelectEOLAsUnixAction(jNPad));
	manager.registerAction(selectEOLAsMacAction         = new SelectEOLAsMacAction(jNPad));
	manager.registerAction(rightIndentAction            = new RightIndentAction(jNPad));
	manager.registerAction(leftIndentAction             = new LeftIndentAction(jNPad));
	manager.registerAction(toUpperCaseAction            = new ToUpperCaseAction(jNPad));
	manager.registerAction(toLowerCaseAction            = new ToLowerCaseAction(jNPad));
    manager.registerAction(invertUpperLowerAction       = new InvertUpperLowerAction(jNPad));
	manager.registerAction(capitalizeAction             = new CapitalizeAction(jNPad));
	manager.registerAction(toTitleAction                = new ToTitleAction(jNPad));
	manager.registerAction(setReadOnlyAction            = new SetReadOnlyAction(jNPad));
	manager.registerAction(alignStringsByCommaAction    = new AlignStringsByCommaAction(jNPad));
	manager.registerAction(alignStringsByEqualAction    = new AlignStringsByEqualAction(jNPad));
	manager.registerAction(alignStringsByClipboardAction= new AlignStringsByClipboardAction(jNPad));
	manager.registerAction(increaseFontAction           = new IncreaseFontAction(jNPad));
	manager.registerAction(decreaseFontAction           = new DecreaseFontAction(jNPad));

	manager.registerAction(autoCompletionAction         = new AutoCompletionAction(jNPad));
	manager.registerAction(autoIndentAction             = new AutoIndentAction(jNPad));
	manager.registerAction(indentUsingSpacesAction      = new IndentUsingSpacesAction(jNPad));
	manager.registerAction(loadOptionsFileAction        = new LoadOptionsFileAction(jNPad));
	manager.registerAction(loadDefaultOptionsFileAction = new LoadDefaultOptionsFileAction(jNPad));
	manager.registerAction(loadShortcutsFileAction      = new LoadShortcutsFileAction(jNPad));
	manager.registerAction(loadLoggingConfigFileAction  = new LoadLoggingConfigFileAction(jNPad));
	manager.registerAction(fontsAction                  = new FontsAction(jNPad));
	manager.registerAction(resetFontAction              = new ResetFontAction(jNPad));
	manager.registerAction(foregroundAction             = new ForegroundAction(jNPad));
    manager.registerAction(brightenForegroundAction     = new BrightenForegroundAction(jNPad));
    manager.registerAction(darkenForegroundAction       = new DarkenForegroundAction(jNPad));
    manager.registerAction(resetForegroundAction        = new ResetForegroundAction(jNPad));
	manager.registerAction(backgroundAction             = new BackgroundAction(jNPad));
    manager.registerAction(brightenBackgroundAction     = new BrightenBackgroundAction(jNPad));
    manager.registerAction(darkenBackgroundAction       = new DarkenBackgroundAction(jNPad));
    manager.registerAction(resetBackgroundAction        = new ResetBackgroundAction(jNPad));

	manager.registerAction(windowsAction                = new WindowAction(jNPad));
	manager.registerAction(nextBufferAction             = new NextBufferAction(jNPad));
	manager.registerAction(previousBufferAction         = new PreviousBufferAction(jNPad));
	manager.registerAction(splitHorizontallyAction      = new SplitHorizontallyAction(jNPad));
	manager.registerAction(splitVerticallyAction        = new SplitVerticallyAction(jNPad));
	manager.registerAction(unsplitCurrentAction         = new UnsplitCurrentAction(jNPad));
	manager.registerAction(unsplitAction                = new UnsplitAction(jNPad));
	manager.registerAction(restoreSplitAction           = new RestoreSplitAction(jNPad));
	manager.registerAction(previousEditPaneAction       = new PreviousEditPaneAction(jNPad));
	manager.registerAction(nextEditPaneAction           = new NextEditPaneAction(jNPad));
	manager.registerAction(splitViewerHorizontallyAction= new SplitViewerHorizontallyAction(jNPad));
	manager.registerAction(splitViewerVerticallyAction  = new SplitViewerVerticallyAction(jNPad));
	manager.registerAction(unsplitViewerCurrentAction   = new UnsplitViewerCurrentAction(jNPad));
	manager.registerAction(unsplitViewerAction          = new UnsplitViewerAction(jNPad));
	manager.registerAction(restoreViewerSplitAction     = new RestoreViewerSplitAction(jNPad));
	manager.registerAction(previousBufferSetAction      = new PreviousBufferSetAction(jNPad));
	manager.registerAction(nextBufferSetAction          = new NextBufferSetAction(jNPad));
	
	manager.registerAction(cliArgsAction                = new CLIArgsAction(jNPad));
    manager.registerAction(licenceAction                = new LicenseAction(jNPad));
    manager.registerAction(check4UpdatesAction          = new Check4UpdatesAction(jNPad));
	manager.registerAction(aboutAction                  = new AboutAction(jNPad));
    
    if (Accelerators.isUsingCompositeShortcuts()) {
      JNPadKeyboardHandler keyboardHandler = jNPad.getKeyboardHandler();
      if (keyboardHandler != null) {
		registerKeyBinding(newFileAction                , keyboardHandler);
		registerKeyBinding(openFileAction               , keyboardHandler);
		registerKeyBinding(openAllRecentFilesAction     , keyboardHandler);
		registerKeyBinding(emptyRecentFilesListAction   , keyboardHandler);
		registerKeyBinding(closeFileAction              , keyboardHandler);
		registerKeyBinding(closeFilesAction             , keyboardHandler);
		registerKeyBinding(closeAllAction               , keyboardHandler);
		registerKeyBinding(closeOtherAction             , keyboardHandler);
		registerKeyBinding(closeAllToLeftAction         , keyboardHandler);
		registerKeyBinding(closeAllToRightAction        , keyboardHandler);
		registerKeyBinding(saveFileAction               , keyboardHandler);
		registerKeyBinding(saveAllFilesAction           , keyboardHandler);
		registerKeyBinding(saveFileAsAction             , keyboardHandler);
		registerKeyBinding(fileBrowserAction            , keyboardHandler);
		registerKeyBinding(filePropertiesAction         , keyboardHandler);
		registerKeyBinding(printSetAction               , keyboardHandler);
		registerKeyBinding(printAction                  , keyboardHandler);
		registerKeyBinding(reloadAction                 , keyboardHandler);
        registerKeyBinding(executeActionAction          , keyboardHandler);
		registerKeyBinding(loadSessionAction            , keyboardHandler);
		registerKeyBinding(saveSessionAction            , keyboardHandler);
		registerKeyBinding(reduceAsTrayIconAction       , keyboardHandler);
        registerKeyBinding(restartAction                , keyboardHandler);
		registerKeyBinding(exitAction                   , keyboardHandler);

		registerKeyBinding(undoAction                   , keyboardHandler);
		registerKeyBinding(redoAction                   , keyboardHandler);
		registerKeyBinding(cutAction                    , keyboardHandler);
		registerKeyBinding(cutLineAction                , keyboardHandler);
		registerKeyBinding(copyAction                   , keyboardHandler);
		registerKeyBinding(copyLineAction               , keyboardHandler);
		registerKeyBinding(pasteAction                  , keyboardHandler);
		registerKeyBinding(selectAllAction              , keyboardHandler);
		registerKeyBinding(selectParagraphAction        , keyboardHandler);
		registerKeyBinding(selectionBeginParagraphAction, keyboardHandler);
		registerKeyBinding(selectionEndParagraphAction  , keyboardHandler);
		registerKeyBinding(selectLineAction             , keyboardHandler);
		registerKeyBinding(selectionBeginLineAction     , keyboardHandler);
		registerKeyBinding(selectionEndLineAction       , keyboardHandler);
		registerKeyBinding(selectWordAction             , keyboardHandler);
		registerKeyBinding(deleteAction                 , keyboardHandler);
		registerKeyBinding(deleteParagraphAction        , keyboardHandler);
		registerKeyBinding(deletionBeginParagraphAction , keyboardHandler);
		registerKeyBinding(deletionEndParagraphAction   , keyboardHandler);
		registerKeyBinding(deleteLineAction             , keyboardHandler);
		registerKeyBinding(deletionBeginLineAction      , keyboardHandler);
		registerKeyBinding(deletionEndLineAction        , keyboardHandler);
		registerKeyBinding(deleteWordAction             , keyboardHandler);
		registerKeyBinding(tabsToSpacesAction           , keyboardHandler);
		registerKeyBinding(spacesToTabsAction           , keyboardHandler);
		registerKeyBinding(beginLinesAction             , keyboardHandler);
		registerKeyBinding(endLinesAction               , keyboardHandler);
		registerKeyBinding(duplicateCurrentLineAction   , keyboardHandler);
		registerKeyBinding(joinLinesAction              , keyboardHandler);
		registerKeyBinding(sortLinesAscendingAction     , keyboardHandler);
		registerKeyBinding(sortLinesDescendingAction    , keyboardHandler);
		registerKeyBinding(removeEndSpacesAction        , keyboardHandler);
		registerKeyBinding(removeEmptyLinesAction       , keyboardHandler);
		registerKeyBinding(completeWordAction           , keyboardHandler);
		registerKeyBinding(completeWordAllAction        , keyboardHandler);
		registerKeyBinding(wordCountAction              , keyboardHandler);
		registerKeyBinding(insertFileExtensionAction    , keyboardHandler);
		registerKeyBinding(insertFileFullNameAction     , keyboardHandler);
		registerKeyBinding(insertFileNameAction         , keyboardHandler);
		registerKeyBinding(insertFileFullPathAction     , keyboardHandler);
		registerKeyBinding(insertFilePathAction         , keyboardHandler);
		registerKeyBinding(insertDateAction             , keyboardHandler);

		registerKeyBinding(findAction                   , keyboardHandler);
		registerKeyBinding(findNextAction               , keyboardHandler);
		registerKeyBinding(findPreviousAction           , keyboardHandler);
		registerKeyBinding(replaceAction                , keyboardHandler);
		registerKeyBinding(replaceNextAction            , keyboardHandler);
		registerKeyBinding(incrementalSearchAction      , keyboardHandler);
		registerKeyBinding(highlightAllOccurrencesAction, keyboardHandler);
		registerKeyBinding(clearAllOccurrencesAction    , keyboardHandler);
		registerKeyBinding(nextOccurrenceAction         , keyboardHandler);
		registerKeyBinding(previousOccurrenceAction     , keyboardHandler);
		registerKeyBinding(goToLineAction               , keyboardHandler);
		registerKeyBinding(nextBookmarkAction           , keyboardHandler);
		registerKeyBinding(previousBookmarkAction       , keyboardHandler);
		registerKeyBinding(toggleBookmarkAction         , keyboardHandler);
		registerKeyBinding(clearAllBookmarksAction      , keyboardHandler);

		registerKeyBinding(toolBarVisibleAction         , keyboardHandler);
		registerKeyBinding(toolbarsFileAction           , keyboardHandler);
		registerKeyBinding(toolbarsEditRecantationAction, keyboardHandler);
		registerKeyBinding(toolbarsEditClipboardAction  , keyboardHandler);
		registerKeyBinding(toolbarsEditSelectionAction  , keyboardHandler);
		registerKeyBinding(toolbarsEditDeletionAction   , keyboardHandler);
		registerKeyBinding(toolbarsSearchAction         , keyboardHandler);
		registerKeyBinding(toolbarsViewAction           , keyboardHandler);
		registerKeyBinding(toolbarsFormatAction         , keyboardHandler);
		registerKeyBinding(toolbarsWindowAction         , keyboardHandler);
		registerKeyBinding(toolbarsAllVisibleAction     , keyboardHandler);
		registerKeyBinding(toolbarsNoneVisibleAction    , keyboardHandler);
		registerKeyBinding(findResultsVisibleAction     , keyboardHandler);
		registerKeyBinding(statusBarVisibleAction       , keyboardHandler);
		registerKeyBinding(lineWrapAction               , keyboardHandler);
		registerKeyBinding(lineNumbersAction            , keyboardHandler);
		registerKeyBinding(activeLineAction             , keyboardHandler);
		registerKeyBinding(rightMarginLineAction        , keyboardHandler);
        registerKeyBinding(markStripAction              , keyboardHandler);
		registerKeyBinding(markOccurrencesAction        , keyboardHandler);
        registerKeyBinding(markBracketsAction           , keyboardHandler);
		registerKeyBinding(focusOnMainViewAction        , keyboardHandler);
        registerKeyBinding(selectScopeAsViewerAction    , keyboardHandler);
        registerKeyBinding(selectScopeAsBufferSetAction , keyboardHandler);
        registerKeyBinding(selectScopeAsBufferAction    , keyboardHandler);
        registerKeyBinding(selectScopeAsEditPaneAction  , keyboardHandler);
		registerKeyBinding(fullScreenAction             , keyboardHandler);

        registerKeyBinding(selectEOLAsDOSAction         , keyboardHandler);
        registerKeyBinding(selectEOLAsUnixAction        , keyboardHandler);
        registerKeyBinding(selectEOLAsMacAction         , keyboardHandler);
        registerKeyBinding(rightIndentAction            , keyboardHandler);
		registerKeyBinding(leftIndentAction             , keyboardHandler);
		registerKeyBinding(toUpperCaseAction            , keyboardHandler);
		registerKeyBinding(toLowerCaseAction            , keyboardHandler);
        registerKeyBinding(invertUpperLowerAction       , keyboardHandler);
		registerKeyBinding(capitalizeAction             , keyboardHandler);
		registerKeyBinding(toTitleAction                , keyboardHandler);
		registerKeyBinding(setReadOnlyAction            , keyboardHandler);
		registerKeyBinding(alignStringsByCommaAction    , keyboardHandler);
		registerKeyBinding(alignStringsByEqualAction    , keyboardHandler);
		registerKeyBinding(alignStringsByClipboardAction, keyboardHandler);
		registerKeyBinding(increaseFontAction           , keyboardHandler);
		registerKeyBinding(decreaseFontAction           , keyboardHandler);

		registerKeyBinding(autoCompletionAction         , keyboardHandler);
		registerKeyBinding(autoIndentAction             , keyboardHandler);
		registerKeyBinding(indentUsingSpacesAction      , keyboardHandler);
		registerKeyBinding(loadOptionsFileAction        , keyboardHandler);
		registerKeyBinding(loadDefaultOptionsFileAction , keyboardHandler);
		registerKeyBinding(loadShortcutsFileAction      , keyboardHandler);
		registerKeyBinding(loadLoggingConfigFileAction  , keyboardHandler);
		registerKeyBinding(fontsAction                  , keyboardHandler);
		registerKeyBinding(resetFontAction              , keyboardHandler);
		registerKeyBinding(foregroundAction             , keyboardHandler);
        registerKeyBinding(brightenForegroundAction     , keyboardHandler);
        registerKeyBinding(darkenForegroundAction       , keyboardHandler);
        registerKeyBinding(resetForegroundAction        , keyboardHandler);
		registerKeyBinding(backgroundAction             , keyboardHandler);
        registerKeyBinding(brightenBackgroundAction     , keyboardHandler);
        registerKeyBinding(darkenBackgroundAction       , keyboardHandler);
        registerKeyBinding(resetBackgroundAction        , keyboardHandler);

		registerKeyBinding(windowsAction                , keyboardHandler);
		registerKeyBinding(nextBufferAction             , keyboardHandler);
		registerKeyBinding(previousBufferAction         , keyboardHandler);
		registerKeyBinding(splitHorizontallyAction      , keyboardHandler);
		registerKeyBinding(splitVerticallyAction        , keyboardHandler);
		registerKeyBinding(unsplitCurrentAction         , keyboardHandler);
		registerKeyBinding(unsplitAction                , keyboardHandler);
		registerKeyBinding(restoreSplitAction           , keyboardHandler);
		registerKeyBinding(previousEditPaneAction       , keyboardHandler);
		registerKeyBinding(nextEditPaneAction           , keyboardHandler);
        registerKeyBinding(splitViewerHorizontallyAction, keyboardHandler);
        registerKeyBinding(splitViewerVerticallyAction  , keyboardHandler);
        registerKeyBinding(unsplitViewerCurrentAction   , keyboardHandler);
        registerKeyBinding(unsplitViewerAction          , keyboardHandler);
        registerKeyBinding(restoreViewerSplitAction     , keyboardHandler);
        registerKeyBinding(previousBufferSetAction      , keyboardHandler);
        registerKeyBinding(nextBufferSetAction          , keyboardHandler);

		registerKeyBinding(cliArgsAction                , keyboardHandler);
        registerKeyBinding(licenceAction                , keyboardHandler);
        registerKeyBinding(check4UpdatesAction          , keyboardHandler);
		registerKeyBinding(aboutAction                  , keyboardHandler);
      }
    }
    
    //registerKeyboardActions();
  }

  /**
   * Register keyboard actions.
   */
  void registerKeyboardActions() {
    final JRootPane rootPane = jNPad.getRootPane();
    
    // --- previous-colorscheme ---
    Action a01 = new AbstractAction("previous-colorscheme") { //$NON-NLS-1$
      private static final long serialVersionUID = 5149945588526013374L;
      @Override public void actionPerformed(final ActionEvent e) {
        colorscheme(false);
      }
    };
    // ---

    // --- next-colorscheme ---
    Action a02 = new AbstractAction("next-colorscheme") { //$NON-NLS-1$
      private static final long serialVersionUID = -1812759330004016042L;
      @Override public void actionPerformed(final ActionEvent e) {
        colorscheme(true);
      }
    };
    // ---
    
    // --- set-extended-state-maximized ---
    Action a03 = new AbstractAction("set-extended-state-maximized") { //$NON-NLS-1$
      private static final long serialVersionUID = 3339326258509531066L;
      @Override public void actionPerformed(final ActionEvent e) {
        int state = getExtendedState();
        if (state == Frame.NORMAL)
          jNPad.setExtendedState(Frame.MAXIMIZED_BOTH);
        else if (state == Frame.MAXIMIZED_BOTH)
          jNPad.setExtendedState(Frame.NORMAL);
      }
    };
    // ---

    // --- set-extended-state-iconified ---
    Action a04 = new AbstractAction("set-extended-state-iconified") { //$NON-NLS-1$
      private static final long serialVersionUID = 2105097047371458816L;
      @Override public void actionPerformed(final ActionEvent e) {
        jNPad.setExtendedState(Frame.ICONIFIED);
      }
    };
    // ---

    // --- set-position-south-west ---
    Action a05 = new AbstractAction("set-position-south-west") { //$NON-NLS-1$
      private static final long serialVersionUID = -2720390875277001200L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 0, 1);
        }
      }
    };
    // ---

    // --- set-position-south ---
    Action a06 = new AbstractAction("set-position-south") { //$NON-NLS-1$
      private static final long serialVersionUID = 5951855174077152744L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 0.5, 1);
        }
      }
    };
    // ---
    
    // --- set-position-south-east ---
    Action a07 = new AbstractAction("set-position-south-east") { //$NON-NLS-1$
      private static final long serialVersionUID = 4708066409402866008L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 1, 1);
        }
      }
    };
    // ---

    // --- set-position-west ---
    Action a08 = new AbstractAction("set-position-west") { //$NON-NLS-1$
      private static final long serialVersionUID = -4665184641352202545L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 0, 0.5);
        }
      }
    };
    // ---
    
    // --- set-position-center ---
    Action a09 = new AbstractAction("set-position-center") { //$NON-NLS-1$
      private static final long serialVersionUID = 4635440816333110584L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 0.5, 0.5);
        }
      }
    };
    // ---
    
    // --- set-position-east ---
    Action a10 = new AbstractAction("set-position-east") { //$NON-NLS-1$
      private static final long serialVersionUID = -4968518183432122423L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 1, 0.5);
        }
      }
    };
    // ---
    
    // --- set-position-north-west ---
    Action a11 = new AbstractAction("set-position-north-west") { //$NON-NLS-1$
      private static final long serialVersionUID = 1353674687006997512L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 0, 0);
        }
      }
    };
    // ---

    // --- set-position-north ---
    Action a12 = new AbstractAction("set-position-north") { //$NON-NLS-1$
      private static final long serialVersionUID = -4996579103782001213L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 0.5, 0);
        }
      }
    };
    // ---
    
    // --- set-position-north-east ---
    Action a13 = new AbstractAction("set-position-north-east") { //$NON-NLS-1$
      private static final long serialVersionUID = 1409903998443228900L;
      @Override public void actionPerformed(final ActionEvent e) {
        if (getExtendedState() == Frame.NORMAL) {
          GUIUtilities.setPositionFrameOnScreen(jNPad, 1, 0);
        }
      }
    };
    // ---
    
    if (Accelerators.isUsingCompositeShortcuts()) {
      JNPadKeyboardHandler keyboardHandler = jNPad.getKeyboardHandler();
      if (keyboardHandler != null) {
        registerKeyBinding(a01, keyboardHandler);
        registerKeyBinding(a02, keyboardHandler);
        registerKeyBinding(a03, keyboardHandler);
        registerKeyBinding(a04, keyboardHandler);
        registerKeyBinding(a05, keyboardHandler);
        registerKeyBinding(a06, keyboardHandler);
        registerKeyBinding(a07, keyboardHandler);
        registerKeyBinding(a08, keyboardHandler);
        registerKeyBinding(a09, keyboardHandler);
        registerKeyBinding(a10, keyboardHandler);
        registerKeyBinding(a11, keyboardHandler);
        registerKeyBinding(a12, keyboardHandler);
        registerKeyBinding(a13, keyboardHandler);
      }
    }
    else {
      ActionMap actionMap = rootPane.getActionMap();
      InputMap inputMap = rootPane.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
      if (Config.isUsingCustomShortcuts()) {
        registerKeyboardAction(actionMap, inputMap, a01, null);
        registerKeyboardAction(actionMap, inputMap, a02, null);
        registerKeyboardAction(actionMap, inputMap, a03, null);
        registerKeyboardAction(actionMap, inputMap, a04, null);
        registerKeyboardAction(actionMap, inputMap, a05, null);
        registerKeyboardAction(actionMap, inputMap, a06, null);
        registerKeyboardAction(actionMap, inputMap, a07, null);
        registerKeyboardAction(actionMap, inputMap, a08, null);
        registerKeyboardAction(actionMap, inputMap, a09, null);
        registerKeyboardAction(actionMap, inputMap, a10, null);
        registerKeyboardAction(actionMap, inputMap, a11, null);
        registerKeyboardAction(actionMap, inputMap, a12, null);
        registerKeyboardAction(actionMap, inputMap, a13, null);
      }
      else {
        registerKeyboardAction(actionMap, inputMap, a01, Accelerators.PREVIOUS_COLORSCHEME);
        registerKeyboardAction(actionMap, inputMap, a02, Accelerators.NEXT_COLORSCHEME);
        registerKeyboardAction(actionMap, inputMap, a03, Accelerators.SET_EXTENDED_STATE_MAXIMIZED);
        registerKeyboardAction(actionMap, inputMap, a04, Accelerators.SET_EXTENDED_STATE_ICONIFIED);
        registerKeyboardAction(actionMap, inputMap, a05, Accelerators.SET_POSITION_SOUTH_WEST);
        registerKeyboardAction(actionMap, inputMap, a06, Accelerators.SET_POSITION_SOUTH);
        registerKeyboardAction(actionMap, inputMap, a07, Accelerators.SET_POSITION_SOUTH_EAST);
        registerKeyboardAction(actionMap, inputMap, a08, Accelerators.SET_POSITION_WEST);
        registerKeyboardAction(actionMap, inputMap, a09, Accelerators.SET_POSITION_CENTER);
        registerKeyboardAction(actionMap, inputMap, a10, Accelerators.SET_POSITION_EAST);
        registerKeyboardAction(actionMap, inputMap, a11, Accelerators.SET_POSITION_NORTH_WEST);
        registerKeyboardAction(actionMap, inputMap, a12, Accelerators.SET_POSITION_NORTH);
        registerKeyboardAction(actionMap, inputMap, a13, Accelerators.SET_POSITION_NORTH_EAST);
      }
    }
  }
  
  /**
   * Register keyboard action.
   *
   * @param actionMap the action map
   * @param inputMap the input map
   * @param action the action
   * @param defaultAccelerator the default accelerator
   */
  private static void registerKeyboardAction(ActionMap actionMap, InputMap inputMap, Action action, KeyStroke defaultAccelerator) {
    String name = (String) action.getValue(Action.NAME);
    KeyStroke ks = Accelerators.getPropAccelerator(name.concat(".shortcut"), defaultAccelerator); //$NON-NLS-1$
    if (ks != null) {
      actionMap.put(name, action);
      inputMap.put(ks, name);
    }
  }

  /**
   * Gets the extended state.
   *
   * @return the extended state
   */
  private int getExtendedState() {
    return jNPad.getExtendedState();
  }

  /**
   * Colorscheme.
   *
   * @param next the next
   */
  private void colorscheme(boolean next) {
    try {
      final ButtonGroup bgColorScheme = jNPad.menus.bgColorScheme;

      int count = bgColorScheme.getButtonCount();
      int selectedIndex = GUIUtilities.getSelectedButtonIndex(bgColorScheme);

      GUIUtilities.setSelectedButton(bgColorScheme,
          next ? ((selectedIndex < count - 1) ? ++selectedIndex : 0) :
              ((selectedIndex > 0) ? --selectedIndex : count - 1));
    }
    catch (Exception ex) {
      //ignored
    }
  }
  
  /**
   * Register key binding.
   *
   * @param action the action
   * @param keyboardHandler the keyboard handler
   */
  private static void registerKeyBinding(Action action, JNPadKeyboardHandler keyboardHandler) {
    if (keyboardHandler != null && Accelerators.isUsingCompositeShortcuts()) {
      String name = (String) action.getValue(Action.NAME);
      String shortcut1 = Accelerators.getPropKeyBinding(name.concat(".shortcut")); //$NON-NLS-1$
      if (shortcut1 != null) {
        keyboardHandler.addKeyBinding(shortcut1, action);
      }
      String shortcut2 = Accelerators.getPropKeyBinding(name.concat(".shortcut2")); //$NON-NLS-1$
      if (shortcut2 != null) {
        keyboardHandler.addKeyBinding(shortcut2, action);
      }
    }
  }
  
}
