(**************************************************************************)
(*                                                                        *)
(*  Setup for web server                                                  *)
(*  Copyright (C) 2021   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE RealmEditor;

        (************************************************************)
        (*                                                          *)
        (*                  PM Setup for web server                 *)
        (*             The list of realms for a domain              *)
        (*                                                          *)
        (*    Started:        10 September 2021                     *)
        (*    Last edited:    17 September 2021                     *)
        (*    Status:         OK                                    *)
        (*                                                          *)
        (************************************************************)


IMPORT SYSTEM, OS2, CommonSettings, DID, Strings, INIData, RINIData, Remote;

FROM Names IMPORT
    (* type *)  DomainName;

FROM Languages IMPORT
    (* type *)  LangHandle,
    (* proc *)  StrToBuffer, StrToBufferAB;

FROM LowLevel IMPORT
    (* proc *)  EVAL, Copy;

(************************************************************************)

CONST
    Nul = CHR(0);

TYPE
    RealmType = ARRAY [0..63] OF CHAR;
    LanguageString = ARRAY [0..31] OF CHAR;

VAR
    OurHWND: OS2.HWND;
    DomName: DomainName;
    RealmName: RealmType;
    RealmKey: ARRAY [0..69] OF CHAR;
    PageFont: CommonSettings.FontName;
    INIname, SetupINIname: ARRAY [0..15] OF CHAR;
    ChangeInProgress, Changed: BOOLEAN;

(************************************************************************)
(*                       SETTING LABELS AND FONTS                       *)
(************************************************************************)

PROCEDURE SetLanguage (lang: LangHandle);

    (* Relabels this page in the new language. *)

    VAR stringval: ARRAY [0..511] OF CHAR;

    BEGIN
        StrToBufferAB (lang, "RealmEdit.Title", DomName, RealmName, stringval);
        OS2.WinSetWindowText (OurHWND, stringval);
        StrToBuffer (lang, "RealmEdit.URL", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.RbaseURLlabel, stringval);
        StrToBuffer (lang, "RealmEdit.rootdir", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.Rrootdirlabel, stringval);
        StrToBuffer (lang, "RealmEdit.username", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.Rusernamelabel, stringval);
        StrToBuffer (lang, "RealmEdit.password", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.Rpasswordlabel, stringval);
        StrToBuffer (lang, "Buttons.OK", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.RealmOK, stringval);
        StrToBuffer (lang, "Buttons.Cancel", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.RealmCancel, stringval);
    END SetLanguage;

(************************************************************************)

PROCEDURE SetPageFonts;

    (* Changes the font of the notebook pages to the font recorded in   *)
    (* the INI file as belonging to this dialogue.                      *)

    VAR NewFontName: CommonSettings.FontName;

    BEGIN
        CommonSettings.CurrentFont (CommonSettings.DomainDlg, NewFontName);
        IF NOT Strings.Equal (NewFontName, PageFont) THEN
            PageFont := NewFontName;
        END (*IF*);
    END SetPageFonts;

(************************************************************************)
(*                         LOADING DIALOGUE DATA                        *)
(************************************************************************)

PROCEDURE LoadDialogueData (hwnd: OS2.HWND);

    VAR state: RINIData.StringReadState;
        name: ARRAY [0..3] OF ARRAY [0..255] OF CHAR;
        j: CARDINAL;

    BEGIN
        EVAL (RINIData.OpenINIFile (INIname));

        (* To help the readability of the code, load the data first     *)
        (* into an array.                                               *)

        RINIData.GetStringList (DomName, RealmKey, state);
        FOR j := 0 TO 3 DO
            RINIData.NextString (state, name[j]);
        END (*FOR*);
        RINIData.CloseStringList (state);

        RINIData.CloseINIFile;

        (* Now store from the array into the dialogue fields.  *)

        OS2.WinSetDlgItemText (hwnd, DID.RbaseURL, name[0]);
        OS2.WinSetDlgItemText (hwnd, DID.Rrootdir, name[1]);
        OS2.WinSetDlgItemText (hwnd, DID.Rusername, name[2]);
        OS2.WinSetDlgItemText (hwnd, DID.Rpassword, name[3]);

    END LoadDialogueData;

(************************************************************************)
(*                           STORING THE DATA                           *)
(************************************************************************)

PROCEDURE StoreData (hwnd: OS2.HWND);

    VAR name: ARRAY [0..3] OF ARRAY [0..255] OF CHAR;
        SOS: ARRAY [0..1028] OF CHAR;
        j, pos, L: CARDINAL;

    BEGIN
        (* Start by storing from the dialogue into an array of strings.  *)

        OS2.WinQueryDlgItemText (hwnd, DID.RbaseURL, 256, name[0]);
        OS2.WinQueryDlgItemText (hwnd, DID.Rrootdir, 256, name[1]);
        OS2.WinQueryDlgItemText (hwnd, DID.Rusername, 256, name[2]);
        OS2.WinQueryDlgItemText (hwnd, DID.Rpassword, 256, name[3]);

        (* Tidying up: base URL should start with '/', and root     *)
        (* directory should end with '\'.  For now, I'm deciding    *)
        (* that the base URL should end with '/'.                   *)

        IF name[0][0] <> '/' THEN
            Strings.Insert ('/', 0, name[0]);
        END (*IF*);
        L := Strings.Length (name[0]);
        IF (name[0][L-1] <> '/') AND (name[0][L-1] <> '\') THEN
            Strings.Append ('/', name[0]);
        END (*IF*);

        L := Strings.Length (name[1]);
        IF (L = 0) OR ((name[1][L-1] <> '/') AND (name[1][L-1] <> '\')) THEN
            Strings.Append ('\', name[1]);
        END (*IF*);

        (* Now load those strings into a string of strings. *)

        pos := 0;
        FOR j := 0 TO 3 DO
            L := Strings.Length (name[j]);
            Copy (SYSTEM.ADR(name[j]), SYSTEM.ADR(SOS[pos]), L);
            INC (pos, L);
            SOS[pos] := Nul;
            INC (pos);
        END (*FOR*);
        SOS[pos] := Nul;
        INC (pos);

        (* Finally, store that string of strings. *)

        EVAL (RINIData.OpenINIFile (INIname));
        RINIData.INIPutBinary (DomName, RealmKey, SOS, pos);
        RINIData.CloseINIFile;

    END StoreData;

(************************************************************************)
(*                WINDOW PROCEDURE FOR SUBCLASSED CASE                  *)
(************************************************************************)

PROCEDURE ["SysCall"] SubWindowProc (hwnd     : OS2.HWND;
                                     msg      : OS2.ULONG;
                                     mp1, mp2 : OS2.MPARAM): OS2.MRESULT;

    (* Window procedure to intercept some of the things that happen in  *)
    (* the dialogue.  We want this here mainly so that we can           *)
    (* detect a new font dropped on the dialogue.  If the message       *)
    (* is something we don't want to deal with here, we pass it         *)
    (* to the parent window procedure.                                  *)

    VAR OldWndProc: OS2.PFNWP;
        owner: OS2.HWND;
        length, AttrFound: CARDINAL;
        NewFontName: CommonSettings.FontName;

    BEGIN
        OldWndProc := SYSTEM.CAST (OS2.PFNWP, OS2.WinQueryWindowPtr (hwnd, OS2.QWL_USER));
        owner := OS2.WinQueryWindow(hwnd,OS2.QW_OWNER);

        (* Because of the interaction between subclassing and DragText, *)
        (* some messages will go lost if we use the obvious strategy of *)
        (* sending them through to OldWndProc.  To get around this, we  *)
        (* have to send those messages directly to the target window.   *)

        IF (msg = OS2.WM_BUTTON2DOWN) OR (msg = OS2.DM_DRAGOVER)
                   OR (msg = OS2.DM_DRAGLEAVE) OR (msg = OS2.DM_DROP) THEN

            RETURN OS2.WinSendMsg (owner, msg, mp1, mp2);

        (* Check for font or colour change. *)

        ELSIF msg = OS2.WM_PRESPARAMCHANGED THEN

            IF ChangeInProgress THEN
                RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
            ELSE
                ChangeInProgress := TRUE;
                length := OS2.WinQueryPresParam (hwnd, OS2.PP_FONTNAMESIZE, 0,
                                             AttrFound, CommonSettings.FontNameSize, NewFontName,
                                              0(*OS2.QPF_NOINHERIT*));
                IF length < CommonSettings.FontNameSize THEN
                    NewFontName[length] := Nul;
                END (*IF*);
                ChangeInProgress := FALSE;
                RETURN NIL;
            END (*IF*);

        END (*IF*);

        RETURN OldWndProc (hwnd, msg, mp1, mp2);

    END SubWindowProc;

(************************************************************************)
(*                   WINDOW PROCEDURE FOR THIS DIALOGUE                 *)
(************************************************************************)

PROCEDURE ["SysCall"] DialogueProc(hwnd     : OS2.HWND
                     ;msg      : OS2.ULONG
                     ;mp1, mp2 : OS2.MPARAM): OS2.MRESULT;

    VAR Title: ARRAY [0..255] OF CHAR;
        listwindow: OS2.HWND;
        ButtonID: CARDINAL;
        index: INTEGER;

    BEGIN
        index := OS2.LONGFROMMR(
                   OS2.WinSendDlgItemMsg (hwnd, DID.RealmList, OS2.LM_QUERYSELECTION, NIL, NIL));
        CASE msg OF
           |  OS2.WM_INITDLG:
                   Title := "RealmEditor";
                   INIData.SetInitialWindowPosition (hwnd, SetupINIname,
                                                                     Title);
                   OS2.WinSetWindowPtr (OurHWND, OS2.QWL_USER,
                               SYSTEM.CAST(SYSTEM.ADDRESS,
                                    OS2.WinSubclassWindow (OurHWND,
                                                           SubWindowProc)));
                   OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.EditRealm), FALSE);
                   OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.DeleteRealm), FALSE);

           |  CommonSettings.FONTCHANGED:

                   IF ChangeInProgress THEN
                       RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
                   ELSE
                       ChangeInProgress := TRUE;
                       SetPageFonts;
                       ChangeInProgress := FALSE;
                       RETURN NIL;
                   END (*IF*);

           |  OS2.WM_COMMAND:

                   listwindow := OS2.WinWindowFromID(hwnd,DID.RealmList);
                   ButtonID := OS2.SHORT1FROMMP(mp1);

                   IF ButtonID = DID.RealmOK THEN
                       StoreData (hwnd);
                       OS2.WinPostMsg (hwnd, OS2.WM_CLOSE, mp1, mp2)
                   ELSIF ButtonID = DID.RealmCancel THEN
                       OS2.WinPostMsg (hwnd, OS2.WM_CLOSE, mp1, mp2)
                   END (*IF*);

                   RETURN NIL;

           |  OS2.WM_CONTROL:

                   RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);

           |  OS2.WM_CLOSE:
                   CommonSettings.EnableFontChanges(FALSE);
                   RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);

        ELSE    (* default *)
           RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
        END (*CASE*);
        RETURN NIL;
    END DialogueProc;

(**********************************************************************)
(*                            MAIN PROGRAM                            *)
(**********************************************************************)

PROCEDURE Edit (owner: OS2.HWND;  domain, realm: ARRAY OF CHAR);

    (* Opens and runs the edit dialogue for this realm. *)

    VAR Lang: LangHandle;
        LangName: LanguageString;

    BEGIN
        OS2.DosError (OS2.FERR_DISABLEHARDERR);
        CommonSettings.GetINIFilename (INIname);
        SetupINIname := "Setup.INI";
        SetupINIname[6] := INIname[LENGTH(INIname)-3];
        Strings.Assign (domain, DomName);
        Strings.Assign (realm, RealmName);
        Strings.Assign ("realm-", RealmKey);
        Strings.Append (realm, RealmKey);
        Changed := FALSE;
        OurHWND := OS2.WinLoadDlg(OS2.HWND_DESKTOP, owner,
                       DialogueProc,    (* dialogue procedure *)
                       0,                   (* use resources in EXE *)
                       DID.RealmProperties,                (* dialogue ID *)
                       NIL);               (* creation parameters *)
        Remote.SetInitialWindowPosition (OurHWND, "RealmEditor");
        CommonSettings.CurrentLanguage (Lang, LangName);
        SetLanguage (Lang);
        LoadDialogueData (OurHWND);

        OS2.WinShowWindow (owner, FALSE);
        OS2.WinProcessDlg(OurHWND);
        OS2.WinShowWindow (owner, TRUE);

        Remote.StoreWindowPosition (OurHWND, "RealmEditor", TRUE);
        OS2.WinDestroyWindow (OurHWND);
    END Edit;

(************************************************************************)

BEGIN
    ChangeInProgress := FALSE;
END RealmEditor.

