(**************************************************************************)
(*                                                                        *)
(*  Web server session manager                                            *)
(*  Copyright (C) 2021   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE HTTPAuth;

        (********************************************************)
        (*                                                      *)
        (*                 HTTP Authentication                  *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            18 September 2021               *)
        (*  Last edited:        22 September 2021               *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (*  I've moved authentication code into this module     *)
        (*  because the Domains module was becoming confusing   *)
        (*                                                      *)
        (********************************************************)

IMPORT Strings, INIData, Base64;

FROM SYSTEM IMPORT CARD8, LOC;

FROM Arith64 IMPORT
    (* const*)  Zero64,
    (* type *)  CARD64;

FROM NetStream IMPORT
    (* type *)  NStream,
    (* proc *)  PutBytes, PutEOL;

FROM MD5 IMPORT
    (* type *)  MD5_CTX,
    (* proc *)  MD5Init, MD5Update, MD5Final, MD5DigestToString;

FROM SHA2 IMPORT
    (* type *)  SHA2_CTX,
    (* proc *)  SHA256Init, SHA2Update, SHA2Final, SHA256DigestToString;

FROM SHA512 IMPORT
    (* type *)  SHA512_CTX,
    (* proc *)  SHA512_256Init, SHA512Update, SHA512Final, SHA512_256DigestToString;

FROM Misc IMPORT
    (* proc *)  MatchLeading;

FROM MiscFuncs IMPORT
    (* proc *)  StringMatch, HeadMatch, ToLower, AppendCard;

FROM MyClock IMPORT
    (* proc *)  PackedDateTimeToGMT, CurrentDateAndTime;

FROM RandCard IMPORT
    (* proc *)  RandCardinal;

FROM FileOps IMPORT
    (* type *)  DirectoryEntry,
    (* proc *)  FirstDirEntry, DirSearchDone;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  LogTransaction;

FROM Names IMPORT
    (* type *)  FilenameString;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

(************************************************************************)

CONST
    Nul = CHR(0);

TYPE
    (* Authorisation for protected realms. *)

    AuthData = POINTER TO AuthRecord;
    AuthRecord =RECORD
                    next: AuthData;
                    realm: RealmType;
                    baseURL: URLtype;
                    root: FilenameString;
                    username, password: ARRAY [0..63] OF CHAR;
                END (*RECORD*);

VAR colon: ARRAY [0..0] OF CHAR;

(************************************************************************)
(*                          OUTPUT ONE LINE                             *)
(************************************************************************)

PROCEDURE PutAndLogLine (NS: NStream;  LogID: TransactionLogID;
                                text: ARRAY OF CHAR;  always: BOOLEAN);

    (* Sends one line, plus CRLF terminator, with a copy to the     *)
    (* transaction log.  If always is FALSE then we only log this   *)
    (* if DetailedLogging is TRUE.                                  *)

    VAR logtext: ARRAY [0..1023] OF CHAR;

    BEGIN
        IF always THEN
            logtext := "> ";
            Strings.Append (text, logtext);
            LogTransaction (LogID, logtext);
        END (*IF*);
        IF text[0] <> Nul THEN
            EVAL (PutBytes (NS, text, LENGTH(text)));
        END (*IF*);
        PutEOL (NS);
    END PutAndLogLine;

(************************************************************************)
(*                          BASIC CHALLENGE                             *)
(************************************************************************)

PROCEDURE BasicChallenge (NS: NStream;  p: AuthData;  LogID: TransactionLogID);

    (* Sends appropriate WWW-Authenticate lines.  *)

    VAR message: ARRAY [0..127] OF CHAR;

    BEGIN
        message := 'WWW-Authenticate: Basic realm="';
        Strings.Append (p^.realm, message);  Strings.Append ('"', message);
        PutAndLogLine (NS, LogID, message, TRUE);
    END BasicChallenge;

(************************************************************************)
(*                        BASIC AUTHENTICATION                          *)
(************************************************************************)

PROCEDURE BasicAuthorised (p: AuthData;  param: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE if param holds the correct response to a basic  *)
    (* authentication request.                                      *)

    VAR user, pass: ARRAY [0..127] OF CHAR;
        N, pos: CARDINAL;  found: BOOLEAN;

    BEGIN
        (* Decode from Base64 to get user:pass  *)

        N := Base64.Decode (param, user);
        user[N] := Nul;
        Strings.FindNext (':', user, 0, found, pos);
        IF NOT found THEN
            RETURN FALSE;
        END (*IF*);
        Strings.Assign (user, pass);
        user[pos] := Nul;
        Strings.Delete (pass, 0, pos+1);
        RETURN StringMatch (user, p^.username)
                  AND Strings.Equal (pass, p^.password);
    END BasicAuthorised;

(************************************************************************)
(*                          DIGEST CHALLENGE                            *)
(************************************************************************)

PROCEDURE MakeNonce (VAR (*OUT*) nonce, opaque: ARRAY OF CHAR);

    (* Creates two random strings. *)

    VAR ctx: MD5_CTX;
        text: ARRAY [0..63] OF CHAR;
        digest: ARRAY [0..15] OF CARD8;

    BEGIN
        (* Let the nonce be a hashed string that contains a timestamp. *)

        ctx := MD5Init();
        CurrentDateAndTime (text);
        MD5Update (ctx, text, LENGTH(text));
        text := "Neddy Seagoon";
        MD5Update (ctx, text, LENGTH(text));
        MD5Final (ctx, digest);
        MD5DigestToString (digest, nonce);

        (* opaque can be just about anything random. *)

        text := "ZizzerZazzerZuz";
        AppendCard (RandCardinal(), text);
        ctx := MD5Init();
        MD5Update (ctx, text, LENGTH(text));
        MD5Final (ctx, digest);
        MD5DigestToString (digest, opaque);

        (************************************
        (* Temporary arrangment for testing *)
        Strings.Assign ("a3a", nonce);
        Strings.Assign ("b3b", opaque);
        ************************************)

    END MakeNonce;

(************************************************************************)

PROCEDURE DigestChallenge (NS: NStream;  p: AuthData;  method: ARRAY OF CHAR;
                            algorithm: ARRAY OF CHAR;  LogID: TransactionLogID);

    (* Sends appropriate WWW-Authenticate lines.  *)

    VAR message, nonce, opaque, qop: ARRAY [0..127] OF CHAR;

    BEGIN
        MakeNonce (nonce, opaque);
        IF Strings.Equal (method, "GET") OR Strings.Equal (method, "HEAD") THEN
            qop := "auth,auth-int";
        ELSE
            qop := "auth";
        END (*IF*);

        message := 'WWW-Authenticate: Digest';
        PutAndLogLine (NS, LogID, message, TRUE);

        message := '     realm="';
        Strings.Append (p^.realm, message);  Strings.Append ('",', message);
        PutAndLogLine (NS, LogID, message, TRUE);

        message := '     qop="';
        Strings.Append (qop, message);
        Strings.Append ('"', message);
        PutAndLogLine (NS, LogID, message, TRUE);

        message := '     algorithm=';
        Strings.Append (algorithm, message);
        Strings.Append (',', message);
        PutAndLogLine (NS, LogID, message, TRUE);

        message := '     nonce="';
        Strings.Append (nonce, message);  Strings.Append ('",', message);
        PutAndLogLine (NS, LogID, message, TRUE);

        message := '     opaque="';
        Strings.Append (opaque, message);  Strings.Append ('"', message);
        PutAndLogLine (NS, LogID, message, TRUE);

    END DigestChallenge;

(************************************************************************)
(*                         HASHING ALGORITHMS                           *)
(************************************************************************)

TYPE
    AlgType = (md5, sha256, sha512_256);

    CtxType =   RECORD
                    kind: AlgType;
                    ctxmd5: MD5_CTX;
                    ctxsha2: SHA2_CTX;
                    ctxsha512: SHA512_CTX;
                END (*RECORD*);

(************************************************************************)

PROCEDURE IdentifyAlg (algname: ARRAY OF CHAR;
                            VAR (*OUT*) sess: BOOLEAN): AlgType;

    (* Translates algorithm name. *)

    VAR result: AlgType;  pos: CARDINAL;

    BEGIN
        Strings.FindNext ("-sess", algname, 0, sess, pos);
        IF sess THEN
            algname[pos] := Nul;
        END (*IF*);
        IF Strings.Equal (algname, "md5") THEN
            result := md5;
        ELSIF Strings.Equal (algname, "sha-256") THEN
            result := sha256;
        ELSIF Strings.Equal (algname, "sha-512-256") THEN
            result := sha512_256;
        ELSE
            (* Default to md5 *)
            result := md5;
        END (*IF*);
        RETURN result;
    END IdentifyAlg;

(************************************************************************)

PROCEDURE AlgDigestLength (alg: AlgType): CARDINAL;

    (* Returns the number of digest bytes for this algorithm. *)

    BEGIN
        IF alg = md5 THEN
            RETURN 16;
        ELSIF (alg = sha256) OR (alg = sha512_256) THEN
            RETURN 32;
        ELSE
            RETURN 0;
        END (*IF*);
    END AlgDigestLength;

(************************************************************************)

PROCEDURE AlgInit (alg: AlgType): CtxType;

    (* Initialises a digest calculation. *)

    VAR result: CtxType;

    BEGIN
        result.kind := alg;
        IF alg = md5 THEN
            result.ctxmd5 := MD5Init();
        ELSIF alg = sha256 THEN
            result.ctxsha2 := SHA256Init();
        ELSIF alg = sha512_256 THEN
            result.ctxsha512 := SHA512_256Init();
        END (*IF*);
        RETURN result;
    END AlgInit;

(************************************************************************)

PROCEDURE AlgUpdate (ctx: CtxType;  VAR (*IN*) data: ARRAY OF LOC;
                                    amount: CARDINAL);

    (* Adds more data to a digest calculation. *)

    BEGIN
        IF ctx.kind = md5 THEN
            MD5Update (ctx.ctxmd5, data, amount);
        ELSIF ctx.kind = sha256 THEN
            SHA2Update (ctx.ctxsha2, data, amount);
        ELSIF ctx.kind = sha512_256 THEN
            SHA512Update (ctx.ctxsha512, data, amount);
        END (*IF*);
    END AlgUpdate;

(************************************************************************)

PROCEDURE AlgFinal (ctx: CtxType;  VAR (*OUT*) result: ARRAY OF CARD8);

    (* Adds more data to a digest calculation. *)

    BEGIN
        IF ctx.kind = md5 THEN
            MD5Final (ctx.ctxmd5, result);
        ELSIF ctx.kind = sha256 THEN
            SHA2Final (ctx.ctxsha2, result);
        ELSIF ctx.kind = sha512_256 THEN
            SHA512Final (ctx.ctxsha512, result);
        END (*IF*);
    END AlgFinal;

(************************************************************************)

PROCEDURE AlgDigestToString (alg: AlgType;  hash: ARRAY OF CARD8;
                                    VAR (*OUT*) result: ARRAY OF CHAR);

    (* Turns a digest into a hexadecimal string.  *)

    BEGIN
        IF alg = md5 THEN
            MD5DigestToString (hash, result);
        ELSIF alg = sha256 THEN
            SHA256DigestToString (hash, result);
        ELSIF alg = sha512_256 THEN
            SHA512_256DigestToString (hash, result);
        END (*IF*);
    END AlgDigestToString;

(************************************************************************)

PROCEDURE EmptyBodyHash (alg: AlgType;  VAR (*OUT*) result: ARRAY OF CHAR);

    (* Returns the (hexadecimal) has of an empty string.  *)

    BEGIN
        IF alg = md5 THEN
            Strings.Assign ("d41d8cd98f00b204e9800998ecf8427e", result);
        ELSIF alg = sha256 THEN
            Strings.Assign (
              "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855",
                                result);
        ELSIF alg = sha512_256 THEN
            Strings.Assign (
              "c672b8d1ef56ed28ab87c3622c5114069bdd3ad7b8f9737498d0c01ecef0967a",
                                result);
        END (*IF*);
    END EmptyBodyHash;

(************************************************************************)
(*                       DIGEST AUTHENTICATION                          *)
(************************************************************************)

PROCEDURE Hash2 (alg: AlgType;  VAR (*IN*) a, b: ARRAY OF CHAR;
                                VAR (*OUT*) result: ARRAY OF CARD8);

    (* Hash of two strings separated by colons. *)

    VAR ctx: CtxType;

    BEGIN
        ctx := AlgInit(alg);
        AlgUpdate (ctx, a, LENGTH(a));
        AlgUpdate (ctx, colon, 1);
        AlgUpdate (ctx, b, LENGTH(b));
        AlgFinal (ctx, result);
    END Hash2;

(************************************************************************)

PROCEDURE Hash3 (alg: AlgType;  VAR (*IN*) a, b, c: ARRAY OF CHAR;
                                VAR (*OUT*) result: ARRAY OF CARD8);

    (* Hash of three strings separated by colons. *)

    VAR ctx: CtxType;

    BEGIN
        ctx := AlgInit(alg);
        AlgUpdate (ctx, a, LENGTH(a));
        AlgUpdate (ctx, colon, 1);
        AlgUpdate (ctx, b, LENGTH(b));
        AlgUpdate (ctx, colon, 1);
        AlgUpdate (ctx, c, LENGTH(c));
        AlgFinal (ctx, result);
    END Hash3;

(************************************************************************)

PROCEDURE Hash3a (alg: AlgType;
                  VAR (*IN*) a: ARRAY OF LOC;  alength: CARDINAL;
                  VAR (*IN*) b: ARRAY OF LOC;  blength: CARDINAL;
                  VAR (*IN*) c: ARRAY OF LOC;  clength: CARDINAL;
                                VAR (*OUT*) result: ARRAY OF CARD8);

    (* Like Hash3, except for the paramater types. *)

    VAR ctx: CtxType;

    BEGIN
        ctx := AlgInit(alg);
        AlgUpdate (ctx, a, alength);
        AlgUpdate (ctx, colon, 1);
        AlgUpdate (ctx, b, blength);
        AlgUpdate (ctx, colon, 1);
        AlgUpdate (ctx, c, clength);
        AlgFinal (ctx, result);
    END Hash3a;

(************************************************************************)

PROCEDURE DigestAuthorised (p: AuthData;
                                method, param: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE if param holds the correct response to a digest *)
    (* authentication request.                                      *)

    VAR pos: CARDINAL;
        nonce, cnonce, opaque, response, qop,
                    nc, algname, uri: ARRAY [0..127] OF CHAR;

    (********************************************************************)

    (*
    PROCEDURE Dump (label, value: ARRAY OF CHAR);
        VAR message: ARRAY [0..127] OF CHAR;
        BEGIN
            Strings.Assign (label, message);
            Strings.Append (" = ", message);
            Strings.Append (value, message);
            LogTransaction (ID, message);
        END Dump;

    (********************************************************************)

    PROCEDURE DumpParams;

        BEGIN
            Dump ("username", p^.username);
            Dump ("realm", p^.realm);
            Dump ("nonce", nonce);
            Dump ("uri", uri);
            Dump ("algorithm", alg);
            Dump ("opaque", opaque);
            Dump ("qop", qop);
            Dump ("nc", nc);
            Dump ("cnonce", cnonce);
        END DumpParams;
    *)
    (********************************************************************)

    PROCEDURE ParseOneItem(): BOOLEAN;

        (* Picks up one kwd=value item from param.  *)
        (* Returns FALSE on unacceptable input.     *)

        VAR kwd, val: ARRAY [0..127] OF CHAR;
            k: CARDINAL;

        BEGIN
            k := 0;
            WHILE param[pos] <> '=' DO
                kwd[k] := param[pos];  INC(k);  INC(pos);
            END (*WHILE*);
            kwd[k] := Nul;
            INC (pos);
            k := 0;
            WHILE (param[pos] <> ',') AND (param[pos] <> Nul) DO
                val[k] := param[pos];  INC(k);  INC(pos);
            END (*WHILE*);
            val[k] := Nul;
            IF val[0] = '"' THEN
                Strings.Delete (val, 0, 1);
                k := LENGTH(val);
                IF k = 0 THEN RETURN FALSE END (*IF*);
                DEC(k);
                IF val[k] = '"' THEN val[k] := Nul;
                ELSE RETURN FALSE;
                END (*IF*);
            END (*IF*);

            (* We now have value and keyword.  Deal with each   *)
            (* keyword appropriately.                           *)

            IF Strings.Equal (kwd, "username") THEN
                IF NOT Strings.Equal (val, p^.username) THEN
                    RETURN FALSE;
                END (*IF*);
            ELSIF Strings.Equal (kwd, "realm") THEN
                IF NOT Strings.Equal (val, p^.realm) THEN
                    RETURN FALSE;
                END (*IF*);
            ELSIF Strings.Equal (kwd, "uri") THEN
                IF NOT HeadMatch(val, p^.baseURL) THEN
                    RETURN FALSE;
                END (*IF*);
                Strings.Assign (val, uri);
            ELSIF Strings.Equal (kwd, "algorithm") THEN
                ToLower (val);
                Strings.Assign (val, algname);
            ELSIF Strings.Equal (kwd, "nc") THEN
                Strings.Assign (val, nc);
            ELSIF Strings.Equal (kwd, "nonce") THEN
                Strings.Assign (val, nonce);
            ELSIF Strings.Equal (kwd, "cnonce") THEN
                Strings.Assign (val, cnonce);
            ELSIF Strings.Equal (kwd, "opaque") THEN
                Strings.Assign (val, opaque);
            ELSIF Strings.Equal (kwd, "response") THEN
                Strings.Assign (val, response);
            ELSIF Strings.Equal (kwd, "qop") THEN
                Strings.Assign (val, qop);
            ELSIF Strings.Equal (kwd, "nc") THEN
                Strings.Assign (val, nc);
            END (*IF*);

            RETURN TRUE;

        END ParseOneItem;

    (********************************************************************)

    VAR ctx: CtxType;
        HA1, HA2, RespHash: ARRAY [0..31] OF CARD8;
        (* HEntity is hash of entity body, not supported here. *)
        Response, HEntity, HA1Hex, HA2Hex: ARRAY [0..63] OF CHAR;
        digestlength: CARDINAL;
        alg: AlgType;
        sess: BOOLEAN;

    BEGIN
        nonce := "";  cnonce := "";  opaque := "";  uri := "";
        qop := "";  nc := "";  response := "";  algname := "md5";
        HEntity := "";

        pos := 0;
        LOOP
            WHILE param[pos] = ' ' DO INC(pos) END(*WHILE*);
            IF param[pos] = Nul THEN EXIT(*LOOP*) END(*IF*);
            IF NOT ParseOneItem() THEN RETURN FALSE END (*IF*);
            IF param[pos] = ',' THEN INC(pos) END(*IF*);
        END (*LOOP*);

        alg := IdentifyAlg (algname, sess);
        digestlength := AlgDigestLength(alg);

        EmptyBodyHash (alg, HEntity);

        (* First step: calculate the HA1 hash.  *)

        Hash3 (alg, p^.username, p^.realm, p^.password, HA1);
        AlgDigestToString (alg, HA1, HA1Hex);
        IF sess THEN
            Hash3a (alg, HA1Hex, 2*digestlength, nonce, LENGTH(nonce),
                                    cnonce, LENGTH(cnonce), HA1);
        END (*IF*);
        AlgDigestToString (alg, HA1, HA1Hex);

        (* Next, calculate the HA2 hash.  *)

        IF Strings.Equal (qop, "auth-int") THEN
            Hash3a (alg, method, LENGTH(method), uri, LENGTH(uri),
                                    HEntity, digestlength, HA2);
        ELSE
            Hash2 (alg, method, uri, HA2);
        END (*IF*);
        AlgDigestToString (alg, HA2, HA2Hex);

        (* Calculate what the response should be. *)

        ctx := AlgInit(alg);
        AlgUpdate (ctx, HA1Hex, 2*digestlength);
        AlgUpdate (ctx, colon, 1);
        AlgUpdate (ctx, nonce, LENGTH (nonce));
        AlgUpdate (ctx,colon, 1);
        IF qop[0] <> Nul THEN
            AlgUpdate (ctx, nc, LENGTH (nc));
            AlgUpdate (ctx, colon, 1);
            AlgUpdate (ctx, cnonce, LENGTH (cnonce));
            AlgUpdate (ctx, colon, 1);
            AlgUpdate (ctx, qop, LENGTH (qop));
            AlgUpdate (ctx, colon, 1);
        END (*IF*);
        AlgUpdate (ctx, HA2Hex, 2*digestlength);
        AlgFinal (ctx, RespHash);
        AlgDigestToString (alg, RespHash, Response);

        RETURN Strings.Equal (response, Response);

    END DigestAuthorised;

(************************************************************************)
(*                          TEST FUNCTIONS                              *)
(************************************************************************)

PROCEDURE TestCalcHA1 (alg, user, realm, pass, nonce, cnonce: ARRAY OF CHAR;
                        VAR (*OUT*) HA1Hex: ARRAY OF CHAR);

    VAR ctx: MD5_CTX;
        HA1: ARRAY [0..15] OF CARD8;

    BEGIN
        ctx := MD5Init();
        MD5Update (ctx, user, LENGTH(user));
        MD5Update (ctx, colon, 1);
        MD5Update (ctx, realm, LENGTH(realm));
        MD5Update (ctx, colon, 1);
        MD5Update (ctx, pass, LENGTH(pass));
        MD5Final (ctx, HA1);
        IF Strings.Equal (alg, "md5-sess") THEN
            ctx := MD5Init();
            MD5Update (ctx, HA1, 16);
            MD5Update (ctx, colon, 1);
            MD5Update (ctx, nonce, LENGTH (nonce));
            MD5Update (ctx, colon, 1);
            MD5Update (ctx, cnonce, LENGTH(cnonce));
            MD5Final (ctx, HA1);
        END (*IF*);
        MD5DigestToString (HA1, HA1Hex);
    END TestCalcHA1;

(************************************************************************)

PROCEDURE TestCalcResponse (HA1Hex, nonce, nc, cnonce, qop,
                              method, uri: ARRAY OF CHAR;
                               VAR (*OUT*) HA2Hex, Response: ARRAY OF CHAR);

    VAR ctx: MD5_CTX;
        HEntity, HA2, RespHash: ARRAY [0..15] OF CARD8;

    BEGIN
        HEntity[0] := 0;
        ctx := MD5Init();
        MD5Update (ctx, method, LENGTH (method));
        MD5Update (ctx, colon, 1);
        MD5Update (ctx, uri, LENGTH (uri));
        IF Strings.Equal (qop, "auth-int") THEN
            MD5Update (ctx, colon, 1);
            MD5Update (ctx, HEntity, 16);
        END (*IF*);
        MD5Final (ctx, HA2);
        MD5DigestToString (HA2, HA2Hex);

        (* Calculate what the response should be. *)

        ctx := MD5Init();
        MD5Update (ctx, HA1Hex, 32);
        MD5Update (ctx, colon, 1);
        MD5Update (ctx, nonce, LENGTH (nonce));
        MD5Update (ctx,colon, 1);
        IF qop[0] <> Nul THEN
            MD5Update (ctx, nc, LENGTH (nc));
            MD5Update (ctx, colon, 1);
            MD5Update (ctx, cnonce, LENGTH (cnonce));
            MD5Update (ctx, colon, 1);
            MD5Update (ctx, qop, LENGTH (qop));
            MD5Update (ctx, colon, 1);
        END (*IF*);
        MD5Update (ctx, HA2Hex, 32);
        MD5Final (ctx, RespHash);
        MD5DigestToString (RespHash, Response);
    END TestCalcResponse;

(************************************************************************)
(*                     AUTHENTICATION CHALLENGES                        *)
(************************************************************************)

PROCEDURE Challenge (NS: NStream;  p: AuthData;  method: ARRAY OF CHAR;
                                                LogID: TransactionLogID);

    (* Sends WWW-Authenticate headers to NS.  *)

    BEGIN
        DigestChallenge (NS, p, method, "md5-sess", LogID);
        BasicChallenge (NS, p, LogID);
        DigestChallenge (NS, p, method, "sha-256", LogID);
    END Challenge;

(************************************************************************)
(*                        AUTHORISATION CHECKS                          *)
(************************************************************************)

PROCEDURE CheckRealm (p: AuthData;  VAR (*INOUT*) ourURL: ARRAY OF CHAR;
                            VAR (*OUT*) realm: RealmType): AuthData;

    (* Returns a pointer to the AuthData realm that matches our URL,    *)
    (* or NIL if there is no such realm.                                *)

    BEGIN
        WHILE (p <> NIL) AND NOT MatchLeading(ourURL, p^.baseURL) DO
            p := p^.next;
        END (*WHILE*);
        IF p = NIL THEN
            realm := "";
        ELSE
            realm := p^.realm;
            IF ourURL[0] = Nul THEN
                Strings.Assign ("/", ourURL);
            END (*IF*);
        END (*IF*);

        RETURN p;

    END CheckRealm;

(************************************************************************)

PROCEDURE Authorised (p: AuthData; method, param: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE if authorisation is not needed, or if correct   *)
    (* values have been supplied.                                   *)

    BEGIN
        (* If the password is the null string, access is allowed    *)
        (* without any password.                                    *)

        IF p^.password[0] = Nul THEN
            RETURN TRUE;
        END (*IF*);

        (* Otherwise, param is required. *)

        IF param[0] = Nul THEN
            RETURN FALSE;
        END (*IF*);

        IF MatchLeading (param, "Basic") THEN
            RETURN BasicAuthorised (p, param);
        ELSIF MatchLeading (param, "Digest") THEN
            RETURN DigestAuthorised (p, method, param);
        ELSE
            RETURN FALSE;
        END (*IF*);
    END Authorised;

(************************************************************************)
(*                     FINDING A FILE WITHIN A REALM                    *)
(************************************************************************)

PROCEDURE LocateInRealm (p: AuthData;  VAR (*IN*) URL: ARRAY OF CHAR;
                        VAR (*OUT*) filename: ARRAY OF CHAR;
                          VAR (*OUT*) lastmodified: ARRAY OF CHAR;
                              VAR (*OUT*) size: CARD64;
                                VAR (*OUT*) SHTML: BOOLEAN): BOOLEAN;

    (* Translates a URL into a file name, also returns its size.    *)

    (* Remark: this is almost the same procedure as Domains.LocateFile, *)
    (* but I haven't yet found a clean way to simplify the code.        *)

    VAR DirEnt: DirectoryEntry;
        filefound: BOOLEAN;

    BEGIN
        size := CARD64{0,0};

        IF URL[LENGTH(URL)-1] = '/' THEN
            Strings.Append ("index.html", URL);
        END (*IF*);

        (* See whether the name ends with .shtml  *)

        IF LENGTH(URL) < 6 THEN
            SHTML := FALSE;
        ELSE
            Strings.Assign (URL, filename);
            Strings.Delete (filename, 0, LENGTH(filename)-6);
            SHTML := StringMatch (filename, ".shtml");
        END (*IF*);

        (* Convert the filename to a fully specified path string. *)

        Strings.Assign (p^.root, filename);
        Strings.Append (URL, filename);
        filefound := FirstDirEntry (filename, FALSE, FALSE, FALSE, DirEnt);
        IF filefound THEN
            (* A file of this name exists. *)
            size := DirEnt.size;
            PackedDateTimeToGMT (DirEnt.datePkd, DirEnt.timePkd,
                                                    lastmodified);

        ELSE
            (* There might still be a directory of this name,   *)
            (* but in the present version I don't handle that.  *)
        END (*IF*);
        DirSearchDone (DirEnt);

        RETURN filefound;

    END LocateInRealm;

(************************************************************************)
(*               LOADING AND UNLOADING AUTHENTICATION DATA              *)
(************************************************************************)

PROCEDURE LoadRealmInfo (DomainName: ARRAY OF CHAR;
                                p: AuthData;  hini: INIData.HINI);

    (* Loads data for one realm.                    *)
    (* We assume that the INI file is already open. *)

    VAR RealmKey: ARRAY [0..69] OF CHAR;
        state: INIData.StringReadState;
        name: ARRAY [0..3] OF ARRAY [0..255] OF CHAR;
        j: CARDINAL;

    BEGIN
        Strings.Assign ("realm-", RealmKey);
        Strings.Append (p^.realm, RealmKey);

        (* To help the readability of the code, load the data first     *)
        (* into an array.                                               *)

        INIData.GetStringList (hini, DomainName, RealmKey, state);
        FOR j := 0 TO 3 DO
            INIData.NextString (state, name[j]);
        END (*FOR*);
        INIData.CloseStringList (state);

        (* Now copy from the array into the AuthData fields.  *)

        Strings.Assign (name[0], p^.baseURL);
        Strings.Assign (name[1], p^.root);
        Strings.Assign (name[2], p^.username);
        Strings.Assign (name[3], p^.password);

    END LoadRealmInfo;

(************************************************************************)

PROCEDURE LoadRealms (hini: INIData.HINI;
                                DomainName: ARRAY OF CHAR): AuthData;

    (* Load the specification of all the realms for this domain.    *)
    (* We assume that the INI file is already open.                 *)

    VAR name: RealmType;
        head, tail, p: AuthData;
        state: INIData.StringReadState;
        key: ARRAY [0..7] OF CHAR;

    BEGIN
        head := NIL;  tail := NIL;

        (* Load the list of realms from the INI file. *)

        key := "Realms";
        INIData.GetStringList (hini, DomainName, key, state);
        REPEAT
            INIData.NextString (state, name);
            IF name[0] <> Nul THEN
                NEW (p);  p^.next := NIL;
                p^.realm := name;
                IF tail = NIL THEN
                    head := p;
                ELSE
                    tail^.next := p;
                END (*IF*);
                tail := p;
                LoadRealmInfo (DomainName, p, hini);
            END (*IF*);
        UNTIL name[0] = Nul;
        INIData.CloseStringList (state);
        RETURN head;

    END LoadRealms;

(************************************************************************)

PROCEDURE DiscardRealmData (VAR (*INOUT*) A: AuthData);

    (* Discards a list of realms. *)

    VAR next: AuthData;

    BEGIN
        WHILE A <> NIL DO
            next := A^.next;
            DISPOSE (A);
            A := next;
        END (*WHILE*);
    END DiscardRealmData;

(************************************************************************)

BEGIN
    colon[0] := ':';
END HTTPAuth.

