/*
 * window.c - window procedure for 'tablet' program
 */

#define INCL_PM
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tablet.h"
#include "resource.h"

/*
 * window handles
 */

static  HWND    hwndFrame  = NULLHANDLE ;
static  HWND    hwndClient = NULLHANDLE ;
static  HWND    hwndMenu   = NULLHANDLE ;
static  HWND    hwndPopup  = NULLHANDLE ;

/*
 * presentaion space associated to client window
 */

static  HDC     hdcClient = NULLHANDLE ;
static  HPS     hpsClient = NULLHANDLE ;

/*
 * presentation parameters
 */

static  UCHAR   presFnt[128] = { 0 } ;
static  ULONG   presClr[2] = { 0x00000000, 0x00ffffff } ;

/*
 * current table to show
 */

static  TABLPTR     theTable = NULL ;

/*
 * current character (last selected)
 */

static  UCHAR       theChar[8] = { 0 } ;

/*
 * query size of current font
 */

void    winQueryCharSize(PSIZEL sz)
{
    POINTL  apt[TXTBOX_COUNT] ;
    
    GpiQueryTextBox(hpsClient, 2, "", TXTBOX_COUNT, apt) ;
    sz->cx = apt[TXTBOX_TOPRIGHT].x - apt[TXTBOX_TOPLEFT].x   ;
    sz->cy = apt[TXTBOX_TOPLEFT].y - apt[TXTBOX_BOTTOMLEFT].y ;
}

/*
 * set title bar, program name, table name and target to send
 */

static  void    setTitle(HWND hwnd)
{
    UCHAR   buff[256] ;
    
    sprintf(buff, "%s %s", ProgramName, theTable->name) ;
    WinSetWindowText(hwndFrame, buff) ;
}

/*
 * changeTablet - selected tablet
 */

static  void    changeTablet(HWND hwnd, TABLPTR tab)
{
    if (tab == NULL || tab == theTable) {
        return ;
    }
    dispCalcSize(theTable = tab) ;
    setTitle(hwnd) ;
    WinInvalidateRect(hwnd, NULL, FALSE) ;
}

/*
 * setMenu - enable/disable menu item
 */

static  void    setMenu(SHORT mid, BOOL enable)
{
    WinEnableMenuItem(hwndMenu,  mid, enable) ;
    WinEnableMenuItem(hwndPopup, mid, enable) ;
}

/*
 * popMenu - popup menu
 */

static  USHORT  sjis2jis(USHORT c1, USHORT c2)
{
    c1 -= (c1 >= 0xa0) ? 0xc1 : 0x81 ;
    if (c2 >= 0x9f) {
        c1 = c1 + c1 + 0x22 ;
        c2 -= 0x7e ;
    } else {
        c1 = c1 + c1 + 0x21 ;
        c2 -= (c2 <= 0x7e) ? 0x1f : 0x20 ;
    }
    return ((c1 << 8) & 0xff00) + (c2 & 0x00ff) ;
}

static  void    popMenu(HWND hwnd, SHORT x, SHORT y)
{
    LONG    sel ;
    ULONG   opt ;
    USHORT  c1, c2, jis, sjis, euc ;
    UCHAR   buff[128] ;
    
    if (hwndPopup == NULLHANDLE) {
        return ;
    }
    if (strlen(theChar) == 0) {
        sel = IDM_TABLE ;
        WinEnableMenuItem(hwndPopup, IDM_CHAR, FALSE) ;
	WinEnableMenuItem(hwndPopup, IDM_MAGN, FALSE) ;
	WinEnableMenuItem(hwndPopup, IDM_CODE, FALSE) ;
    } else {
        sel = IDM_CHAR ;
        WinEnableMenuItem(hwndPopup, IDM_CHAR, TRUE) ;
	WinEnableMenuItem(hwndPopup, IDM_MAGN, TRUE) ;
	WinEnableMenuItem(hwndPopup, IDM_CODE, TRUE) ;

        c1 = (USHORT) theChar[0] ;
	c2 = (USHORT) theChar[1] ;
        sjis = ((c1 << 8) & 0xff00) + (c2 & 0x00ff) ;
	jis  = sjis2jis(c1, c2) ;
	euc  = jis | 0x8080 ;
 
	sprintf(buff, "SJIS %04x", sjis) ;
	WinSendMsg(hwndPopup, MM_SETITEMTEXT,
	            MPFROMSHORT(IDM_SJIS), MPFROMP(buff)) ;
	sprintf(buff, "JIS  %04x", jis) ;
	WinSendMsg(hwndPopup, MM_SETITEMTEXT,
	            MPFROMSHORT(IDM_JIS), MPFROMP(buff)) ;
	sprintf(buff, "EUC  %04x", euc) ;
	WinSendMsg(hwndPopup,  MM_SETITEMTEXT,
	            MPFROMSHORT(IDM_EUC), MPFROMP(buff)) ;
    }
    
    opt = PU_POSITIONONITEM| PU_HCONSTRAIN | PU_VCONSTRAIN |
            PU_MOUSEBUTTON2DOWN |
            PU_KEYBOARD | PU_MOUSEBUTTON1 | PU_MOUSEBUTTON2 ;
    
    WinPopupMenu(hwndFrame, hwndFrame, hwndPopup, (LONG) x, (LONG) y, sel, opt) ;
}

/*
 * window procedure for client
 */

MRESULT EXPENTRY clientWinProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    HPS     hps ;
    SWP     swp ;
    SIZEL   siz ;
    RECTL   rct ;
    HBITMAP hbm ;
    SHORT   cx, cy ;
    BOOL    enable ;
    SHORT   target ;
    ULONG   len    ;
    ULONG   clr    ;
    UCHAR   buff[256] ;
    
    switch (msg) {

    case WM_CREATE :

        /*
	 * mark window handles
	 */

	hwndClient = hwnd ;
	hwndFrame = WinQueryWindow(hwnd, QW_PARENT) ;
	hwndMenu  = WinWindowFromID(hwndFrame, FID_MENU) ;
        hwndPopup = WinLoadMenu(hwnd, NULLHANDLE, ID_POPUP) ;
        
        /*
	 * associate presentaion space
	 */

        siz.cx = siz.cy = 0 ;
        hdcClient = WinOpenWindowDC(hwnd) ;
        hpsClient = GpiCreatePS(WinQueryAnchorBlock(hwnd), hdcClient, &siz,
                    PU_PELS | GPIT_MICRO | GPIF_DEFAULT | GPIA_ASSOC) ;
        if (hdcClient == NULLHANDLE || hpsClient == NULLHANDLE) {
            return (MRESULT) TRUE ;     /* failed */
        }
	colorInit(hpsClient) ;
	
	/*
	 * start managing display area
	 */

	if (targetInit(hwnd) == FALSE || dispInit(hwnd) == FALSE) {
	    return (MRESULT) TRUE ;     /* failed */
	}

        setTitle(hwnd) ;                /* show current table name  */
	dispCalcSize(theTable) ;        /* setup for the table      */

        return (MRESULT) FALSE ;        /* succeed      */

    case WM_DESTROY :

	/*
	 * save current profile
	 */
	 
        WinQueryWindowPos(hwndFrame, &swp) ;
        profileSave(WinQueryAnchorBlock(hwnd), &swp, presFnt, presClr) ;

        /*
	 * finish display managements
	 */

        targetDone(hwnd) ;
        dispDone(hwnd)   ;
	
        /*
	 * dispose presentation space
	 */

        if (hpsClient != NULLHANDLE) {
            GpiDestroyPS(hpsClient) ;
        }
	return (MRESULT) 0 ;
	
    case WM_SIZE :
        cx = SHORT1FROMMP(mp2) ;
	cy = SHORT2FROMMP(mp2) ;
        winQueryCharSize(&siz) ;
	targetPlace(cx, cy, (SHORT) siz.cy + 4) ;
	dispChangeSize(cx, cy, (cy - (SHORT) siz.cy - 4)) ;
        return (MRESULT) 0 ;

    case WM_HSCROLL :
        if (dispScrollHorz(SHORT2FROMMP(mp2), SHORT1FROMMP(mp2))) {
	    WinInvalidateRect(hwnd, NULL, FALSE) ;
	}
        return (MRESULT) 0 ;

    case WM_VSCROLL :
        if (dispScrollVert(SHORT2FROMMP(mp2), SHORT1FROMMP(mp2))) {
	    WinInvalidateRect(hwnd, NULL, FALSE) ;
	}
        return (MRESULT) 0 ;

    case WM_PAINT :
        hps = WinBeginPaint(hwnd, hpsClient, &rct) ;
        dispPaintTablet(hps, theTable, &rct) ;
	WinEndPaint(hps) ;
	return (MRESULT) 0 ;
	
    case WM_MOUSEMOVE :
        if (targetTrackTrace(hwnd, SHORT1FROMMP(mp1), SHORT2FROMMP(mp1))) {
	    return (MRESULT) 0 ;
	}
	break ;

    case WM_BUTTON1DOWN :
        if (targetTrackDone(hwnd)) {
	    setMenu(IDM_EDIT, FALSE) ;
	    return (MRESULT) 0 ;
	}
	if (dispHitChar(hpsClient, theTable,
	                SHORT1FROMMP(mp1), SHORT2FROMMP(mp1), theChar)) {
	    targetSendChar(theChar) ;
	    return (MRESULT) 0 ;
	}
	break ;

    case WM_BUTTON2DOWN :
        if (targetTrackDone(hwnd)) {
	    setMenu(IDM_EDIT, FALSE) ;
	    return (MRESULT) 0 ;
	}
	dispHitChar(NULLHANDLE, theTable,
	                SHORT1FROMMP(mp1), SHORT2FROMMP(mp1), theChar) ;
        popMenu(hwnd, SHORT1FROMMP(mp1), SHORT2FROMMP(mp1)) ;
        return (MRESULT) 0 ;

    case WM_COMMAND :

        switch (SHORT1FROMMP(mp1)) {

	case IDM_SELECT :
	    changeTablet(hwnd, tableSelect(hwnd, theTable)) ;
	    return (MRESULT) 0 ;
	case IDM_PREV   :
	    changeTablet(hwnd, tablePrev(theTable)) ;
    return (MRESULT) 0 ;
	case IDM_NEXT   :
	    changeTablet(hwnd, tableNext(theTable)) ;
	    return (MRESULT) 0 ;

	case IDM_INTERN :
	    targetChange(SHORT1FROMMP(mp1), NULLHANDLE, NULLHANDLE) ;
	    setMenu(IDM_EDIT, TRUE) ;
            return (MRESULT) 0 ;

	case IDM_CLIPBD :
	    targetChange(SHORT1FROMMP(mp1), NULLHANDLE, NULLHANDLE) ;
	    setMenu(IDM_EDIT, FALSE) ;
            return (MRESULT) 0 ;

	case IDM_WINDOW :
	    targetTrackStart(hwnd) ;
	    return (MRESULT) 0 ;

	case IDM_COPY  :
	case IDM_CUT   :
	case IDM_CLEAR :
	    targetEdit(SHORT1FROMMP(mp1)) ;
	    return (MRESULT) 0 ;

        case IDM_CHAR :
	    if (strlen(theChar) > 0) {
	        targetSendChar(theChar) ;
	    }
	    break ;
	    
	case IDM_MAGN :
	    if (strlen(theChar) > 0) {
	        magnifyChar(hwnd, theChar, presFnt, presClr) ;
	    }
	    break ;
	}
        return (MRESULT) 0 ;

    case WM_INITMENU :

        switch (SHORT1FROMMP(mp1)) {

        case IDM_TABLE  :
	    enable = (theTable->prev != NULL || theTable->next != NULL) ;
            setMenu(IDM_SELECT, enable) ;
            enable = (theTable->prev != NULL) ;
	    setMenu(IDM_PREV, enable) ;
            enable = (theTable->next != NULL) ;
	    setMenu(IDM_NEXT, enable) ;
	    break ;

	case IDM_SENDTO :
	    target = targetQueryType() ;
	    setMenu(IDM_INTERN, target != IDM_INTERN) ;
	    setMenu(IDM_CLIPBD, target != IDM_CLIPBD) ;
	    setMenu(IDM_WINDOW, TRUE) ;     /* always ON */
	    break ;

	case IDM_EDIT   :
	    enable = (targetQuerySelection() > 0) ;
	    setMenu(IDM_COPY,  enable) ;
	    setMenu(IDM_CUT,   enable) ;
	    setMenu(IDM_CLEAR, enable) ;
	    break ;
	}
        return (MRESULT) 0 ;

    case WM_PRESPARAMCHANGED :

	switch (LONGFROMMP(mp1)) {

	case PP_FONTNAMESIZE :
	    len = WinQueryPresParam(hwnd, PP_FONTNAMESIZE,
	            0, NULL, sizeof(buff), (PVOID) buff, 0) ;
            if (len > 0 && fontSetFont(hpsClient, buff)) {
	        strcpy(presFnt, buff)  ;
		WinQueryWindowPos(hwnd, &swp) ;
		winQueryCharSize(&siz) ;
		targetPlace(swp.cx, swp.cy, (SHORT) siz.cy + 4) ;
		dispChangeSize(swp.cx, swp.cy, (swp.cy - (SHORT) siz.cy - 4)) ;
	        dispCalcSize(theTable) ;
	        WinInvalidateRect(hwnd, NULL, FALSE) ;
	    }
            break ;

	case PP_FOREGROUNDCOLORINDEX :
            len = WinQueryPresParam(hwnd, PP_FOREGROUNDCOLORINDEX, 
                    0, NULL, sizeof(ULONG), (PVOID) &clr, 0) ;
            if (len > 0 && colorSetForeIndex(hpsClient, &clr)) {
	        presClr[0] = clr ;
	        WinInvalidateRect(hwnd, NULL, FALSE) ;
	    }
            break ;

	case PP_BACKGROUNDCOLORINDEX :
            len = WinQueryPresParam(hwnd, PP_BACKGROUNDCOLORINDEX, 
                    0, NULL, sizeof(ULONG), (PVOID) &clr, 0) ;
            if (len > 0 && colorSetBackIndex(hpsClient, &clr)) {
	        presClr[1] = clr ;
	        WinInvalidateRect(hwnd, NULL, FALSE) ;
	    }
            break ;

	case PP_FOREGROUNDCOLOR :
            len = WinQueryPresParam(hwnd, PP_FOREGROUNDCOLOR, 
                    0, NULL, sizeof(ULONG), (PVOID) &clr, 0) ;
            if (len > 0 && colorSetForeColor(hpsClient, &clr)) {
	        presClr[0] = clr ;
	        WinInvalidateRect(hwnd, NULL, FALSE) ;
	    }
            break ;

	case PP_BACKGROUNDCOLOR :
            len = WinQueryPresParam(hwnd, PP_BACKGROUNDCOLOR, 
                    0, NULL, sizeof(ULONG), (PVOID) &clr, 0) ;
            if (len > 0 && colorSetBackColor(hpsClient, &clr)) {
	        presClr[1] = clr ;
	        WinInvalidateRect(hwnd, NULL, FALSE) ;
	    }
            break ;
	}        
	return (MRESULT) 0 ;
    }
    return WinDefWindowProc(hwnd, msg, mp1, mp2) ;
}

/*
 * create/destroy window
 */

BOOL    winCreate(HAB hab, TABLPTR tab, PSWP swp, PUCHAR fnt, PULONG clr)
{
    ULONG   flSwp ;
    ULONG   flFrameStyle ;
    ULONG   flFrameFlags ;
    
    theTable = tab  ;
    
    WinRegisterClass(hab, "client", clientWinProc, 
        CS_SIZEREDRAW | CS_CLIPCHILDREN | CS_CLIPSIBLINGS, 0L) ;

    flFrameStyle = 0 ;
    flFrameFlags = (FCF_TITLEBAR | FCF_SYSMENU |
                    FCF_SIZEBORDER | FCF_MINMAX |
                    FCF_MENU | FCF_ACCELTABLE | FCF_ICON |
                    FCF_SHELLPOSITION | FCF_TASKLIST) ;

    hwndFrame = WinCreateStdWindow(
            HWND_DESKTOP,               /* parent window handle     */
	    flFrameStyle,               /* frame window style       */
	    &flFrameFlags,              /* frame create flags       */
            "client",                   /* client window class      */
	    ProgramName,                /* as frame title           */
            0,                          /* client window style      */
            NULLHANDLE,                 /* use internal resource    */
	    ID_TABLET,                  /* window ID                */
	    &hwndClient) ;              /* client window handle     */

    if (hwndFrame == NULLHANDLE || hwndClient == NULLHANDLE) {
        TRACE("failed to create std windows\n") ;
        return FALSE ;
    }

    if (fnt != NULL && strlen(fnt) > 0) {
        WinSetPresParam(hwndClient,
	        PP_FONTNAMESIZE, strlen(fnt) + 1, (PVOID) fnt) ;
    }
    if (clr != NULL) {
        WinSetPresParam(hwndClient, 
	        PP_FOREGROUNDCOLOR, sizeof(LONG), (PVOID) &clr[0]) ;
        WinSetPresParam(hwndClient, 
	        PP_BACKGROUNDCOLOR, sizeof(LONG), (PVOID) &clr[1]) ;
    }

    flSwp = SWP_SHOW | SWP_ACTIVATE | SWP_ZORDER | swp->fl ;

    WinSetWindowPos(hwndFrame, HWND_TOP,
                swp->x, swp->y, swp->cx, swp->cy, flSwp) ;

    return TRUE ;
}

void    winDestroy(void)
{
    if (hpsClient != NULLHANDLE) {
        GpiDestroyPS(hpsClient) ;
        hpsClient = NULLHANDLE ;
    }
    if (hwndClient != NULLHANDLE) {
        WinDestroyWindow(hwndClient) ;
	hwndClient = NULLHANDLE ;
    }
    if (hwndFrame != NULLHANDLE) {
        WinDestroyWindow(hwndFrame) ;
	hwndFrame = NULLHANDLE ;
    }
}
