/**********************************************************************/
/*                                                                    */
/* Filter to convert Post Road Mailer address book to RexxMail format */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program forms part of the RexxMail package, and may not be    */
/* distributed separately.                                            */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* The latest version of RexxMail can be found at                     */
/*               www.degeus.com/rexx                                  */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program is released under the terms of the GNU license, see   */
/*               www.gnu.org/copyleft/gpl.html                        */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* (c) 2001 Marcus de Geus                                            */
/*          marcus@degeus.com                                         */
/*          www.degeus.com                                            */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* Use it if you like it. Don't if you don't. No legalese.            */
/*                                                                    */
/**********************************************************************/

signal on Halt  /* handle halt condition */

parse arg FolderName  /* look for an argument */

if (FolderName >< '') then  /* if we have a folder name */
do
 FolderName = strip(FolderName,'B','"')  /* get rid of any quotation marks */
end

if (lines() = 0) then  /* if we have no lines coming in */
do
 parse source . . ProgName  /* get the program name */
 ProgName = filespec('N',ProgName)  /* get the file name part */
 parse var ProgName ProgName '.' .  /* lose the extension */
 call lineout StdOut,'Usage:   '||ProgName||' [foldername] < Infile [> OutFile]'  /* report */
 call lineout StdOut,''  /* report */
 call lineout StdOut,'Example: PRM2RXML My Friends < Friends.ASC > Friends.TXT'  /* report */
 call lineout StdOut,'         This converts the Post Road Mailer ASCII address file "Friends.ASC" to'  /* report */
 call lineout StdOut,'         a RexxMail address text file, "Friends.TXT", in which each title entry'  /* report */
 call lineout StdOut,'         starts with "My Friends\". Just add the contents of the "Friends.TXT"'  /* report */
 call lineout StdOut,'         file to an existing RexxMail address text file (e.g. "mailaddr.txt" in'  /* report */
 call lineout StdOut,'         the "Configuration" folder) to create a new address template folder,'  /* report */
 call lineout StdOut,'         "My Friends", in the RexxMail "Addresses" folder when importing the'  /* report */
 call lineout StdOut,'         address text file (by means of the "Import Addresses" program object'  /* report */
 call lineout StdOut,'         in the "Accessories" folder, or by running "rexxmail /import" from a'  /* report */
 call lineout StdOut,'         command line).'  /* report */
 call lineout StdOut,'         If no folder name is provided, the address book templates will be'  /* report */
 call lineout StdOut,'         created in the main RexxMail "Addresses" folder.'  /* report */
 call Halt  /* and quit */
end

do while (lines(StdIn))  /* as long as we have lines coming in */
 call lineout StdOut,ConvertLine(linein(StdIn),FolderName)  /* read each, convert it, and send it to std. out */
end

call Halt  /* that's all, folks! */

/**********************************************************************/
ConvertLine: procedure  /* does the actual conversion */
/**********************************************************************/

/**********************************************************************/
/*                                                                    */
/* PRM fields are:                                                    */
/* Last Name<tab>                                                     */
/* First Name<tab>                                                    */
/* Organization<tab>                                                  */
/* Email 1<tab>                                                       */
/* Nickname 1<tab>                                                    */
/* ....Email 5<tab>                                                   */
/* Nickname 5<tab>                                                    */
/* Telephone 1<tab>                                                   */
/* Telephone 2<tab>                                                   */
/* Fax<tab>                                                           */
/* Notes 1<tab>                                                       */
/* Notes 2<tab>                                                       */
/* Notes 3<tab>                                                       */
/* ....                                                               */
/* Notes X                                                            */
/*                                                                    */
/**********************************************************************/

parse arg InLine,FolderName  /* get the arguments */

TAB = d2c(9)  /* define the TAB character */

if (pos(TAB,InLine) = 0) then  /* if the line contains no TABs */
do
 return ''  /* return empty-handed */
end

CRLF = d2c(13)||d2c(10)  /* define the CRLF pair */

parse var InLine LastName (TAB) InLine  /* get the last name */

parse var InLine FirstName (TAB) InLine  /* get the first name */

parse var InLine Organization (TAB) InLine  /* get the organization */

do Index = 1 to 5  /* do 5 times */
 parse var InLine Email.Index (TAB) NickName.Index (TAB) InLine  /* get any email address and nickname */
 if ((Email.Index >< '') & (NickName.Index = '')) then  /* if we have an e-mail address, but no nickname */
 do
  NickName.Index = LastName||', '||FirstName  /* use the real name as nickname */
 end
end

do Index = 1 to 2  /* do 2 times */
 parse var InLine Telephone.Index (TAB) InLine  /* get any telephone numbers */
end

parse var InLine Fax (TAB) InLine  /* get any fax number */

KeyWords = '' /* start with no keywords */

if (Organization >< '') then  /* if we have an organization */
do
 KeyWords = KeyWords||'Organization = '||Organization||CRLF  /* add an organization line */
end

if (Telephone.1 >< '') then  /* if we have a first telephone number */
do
 KeyWords = KeyWords||'Telephone 1 = '||Telephone.1||CRLF  /* add a first telephone line */
end

if (Telephone.2 >< '') then  /* if we have a second telephone number */
do
 KeyWords = KeyWords||'Telephone 2 = '||Telephone.2||CRLF  /* add a second telephone line */
end

if (Fax >< '') then  /* if we have a fax number */
do
 KeyWords = KeyWords||'Fax = '||Fax||CRLF  /* add a second telephone line */
end

do while (InLine >< '')  /* go on while we have a bit of InLine left */
 parse var InLine FirstBit (TAB) InLine  /* get the bit up to the first TAB */
 if (FirstBit >< '') then  /* if we found something */
 do
  KeyWords = KeyWords||FirstBit||CRLF  /* add another line to the keywords bit */
 end
end

OutLine = '<TITLE>'||CRLF  /* start the output with a <TITLE> tag */

if (FolderName >< '') then  /* if we have a folder name */
do
 OutLine = OutLine||FolderName||'\'  /* start the title with the folder name and add a backslash */
end

OutLine = OutLine||LastName||'.'||FirstName||CRLF||,  /* add the real name (with a separator for folder sorting) */
          '<TEXT>'||CRLF||,  /* add a <TEXT> tag */
          'To: "'||NickName.1||'" <'||Email.1||'>'  /* add the first 'To: ' line -- assuming the first e-mail entry in the PRM address book is never empty */

do Index = 2 to 5  /* run through entries 2 to 5 */
 if ((NickName.Index >< '') & (Email.Index >< '')) then  /* if we have both entries (just checking to make sure) */
 do
  OutLine = OutLine||','||CRLF||'    "'||NickName.Index||'" <'||Email.Index||'>'  /* add them to the 'To: ' entry */
 end
end

OutLine = OutLine||CRLF  /* add a new line */

OutLine = OutLine||'Subject: '||CRLF  /* add a 'Subject: ' line */

if (KeyWords >< '') then  /* if we have a keywords bit */
do
 OutLine = OutLine||'<KEYWORDS>'||CRLF||KeyWords  /* add them to the output */
end

OutLine = OutLine||'<END>'||CRLF  /* add an <END> tag */

return OutLine  /* end of ConvertLine */

/**********************************************************************/
Halt :  /* handles halt condition */
/**********************************************************************/

exit  /* that's all, folks! */
