/**
 * $Id: FilenameUtils.java,v 1.1 2001/09/21 02:48:24 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
package redlight.utils;

import java.io.File;
import java.util.StringTokenizer;

/**
 * Transforms a filename to adhere to the conventions of the host
 * platform, using the following algorithm:
 * <ol>
 * <li>First, forward slashes are converted to underscores.
 * <li>Depending on the contents of the os.name system property,
 *     one of the following is performed:
 *     <ul>
 *     <li>On MacOS Classic, the filename is mid-truncated when it
 *         it larger than 31 characters.
 *     <li>On Windows, the following special characters are
 *         replaced by an underscore: " \ : * | < > ; , ?
 *     </ul>
 * <li>The resulting filename is returned to the caller.
 * </ol>
 */
public class FilenameUtils {

    /**
     * From a given file F.EXT returns a file Fn.EXT
     * where n is a zero or more digit number (prefixed by
     * an underscore if more than zero digits) that is 
     * guaranteed not to exist. This method tries twice
     * to create files with n up to 100000, then fails
     * with an exception.
     * @return non existing file.
     * @throws RuntimeException if a non existing file could
     * not be determined.
     */
    public static File getNextUniqueFile(File f) {

        if(!f.exists())
            return f;
        
        String n = f.getName();            
        String pre = f.getName();
        String post = "";
            
        if(n.indexOf('.') != -1) {
            
            pre = n.substring(0, n.indexOf('.'));
            post = n.substring(n.indexOf('.'));
            
        }
        
        pre += "_";

        /* Arbitrary limit of 100 thousand, after that try again,
           after that just throw an exception and fail. Not sure what
           else to do. */
        
        for(int tries = 0; tries < 2; tries++) {
            
            for(int i = 1; i < 100000; i++) {
                
                f = new File(f.getParent(), pre + i + post);
                
                if(!f.exists())
                    return f;
                
            }

        }

        throw new RuntimeException("could not get unique file for " + f);
        
    }

    public static String qualify(String temp) {
        
        /* Convert all slashes to underscores. */

        String filename = TextUtils.findAndReplace(temp, "/", "_");

        if(System.getProperty("os.name").startsWith("Mac") &&
           !System.getProperty("os.name").equals("Mac OS X")) {

            /* If we're on MacOS Classic, make sure the filename
               never exceeds 31 characters. */

            if(filename.length() > 31)
                filename = filename.substring(0, 14) + "..." + 
                    filename.substring(filename.length() - 14, 
                                       filename.length());

            /* The : is dir. separator on Mac. */

            filename = TextUtils.findAndReplace(filename, ":", "_");
            
        } else if(System.getProperty("os.name").startsWith("Windows")) {

            /* On Windows, a whole bunch of illegal characters need to
               be filtered. */
            
            char[] c = filename.toCharArray();
            char[] c2 = new char[c.length];
            char t;

            for(int i = 0; i < c.length; i++) {

                t = c[i];

                switch(c[i]) {
                    
                case '\\':
                case '\"':
                case ':':
                case '*':
                case '|':
                case '<':
                case '>':
                case ';':
                case ',':
                case '?':
                    t = '_';
                default:                    
                    if(Character.isISOControl(c[i])) 
                        t = '_';
                    
                    c2[i] = t;
                    break;

                }

            }

            filename = new String(c2);

        }

        return filename;
	
    }

}

