/**
 * $Id: rlget.java,v 1.5 2001/10/04 19:11:40 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.rltools;

import java.io.*;
import redlight.hotline.*;
import redlight.utils.*;
import redlight.macfiles.*;

/**
 * rlget is a simple program to get files from a Hotline server.
 * 
 * Start rlget doing something like this:
 *
 * $ java -classpath <jarfile> redlight.rltools.rlget
 *
 * rlget returns 1 for argument errors, 0 for success, or a negative
 * integer indicating the number of files that had errors.  
 */
public class rlget {

    static boolean showProgress = false;
    static boolean verbose = false;
    static File localFile = null;
    static boolean progressShown = false;

    static void printUsage() {
        
        System.out.println("Usage: rlget [-v] [-p] [--debug] REMOTE... LOCAL");
        System.out.println("Examples:");
        System.out.println("rlget hotline://user%pass@hostname.net/dir/file /local/directory");
        System.out.println("rlget user@hostname.net/dir/file /local/directory");
        System.out.println("rlget hostname.net/dir/file1 hostname.net/dir/file2 /local/directory");
        
    }

    static RemoteFile parseArg(String arg) throws IllegalArgumentException {

        DebuggerOutput.debug("parseArg: arg = " + arg);

        RemoteFile r = new RemoteFile();

        if(arg.startsWith("hotline://"))
            arg = arg.substring(10);
        
        int atLoc = arg.indexOf('@');
        int pcLoc = arg.indexOf('%');

        if(atLoc != -1) {

            /* User or user/pass combi. */

            if(pcLoc < atLoc) {

                /* User/pass combi. */

                r.user = arg.substring(0, pcLoc);
                r.pass = arg.substring(pcLoc, atLoc);
                arg = arg.substring(atLoc);
                
            } else {

                /* Just user. */

                r.user = arg.substring(0, atLoc);
                arg = arg.substring(atLoc);

            }

        }

        DebuggerOutput.debug("parseArg: path component = " + arg);
        int slashLoc = arg.indexOf('/');

        if(slashLoc == -1)
            throw new IllegalArgumentException("Missing / in " + arg);

        r.host = arg.substring(0, slashLoc);
        r.file = arg.substring(slashLoc);

        return r;

    }

    static RemoteFile[] parseArgs(String[] args, int offset) {

        RemoteFile[] fl = new RemoteFile[(args.length - 1) - offset];

        for(int i = offset; i < args.length - 1; i++)
            fl[i - offset] = parseArg(args[i]);

        return fl;

    }

    static int parseOptions(String args[]) {
        
        int offset = 0;
        int maxOptions = 3;
        
        for(int i = 0; i < maxOptions && i < args.length; i++) {
            
            if(args[i].equals("--debug")) {
                
                DebuggerOutput.setEnabled(true);
                offset++;

            } else if(args[i].equals("-v")) {

                verbose = true;
                offset++;

            } else if(args[i].equals("-p")) {

                showProgress = true;
                offset++;

            } else if(args[i].equals("--")) {

                return offset;

            }

        }

        return offset;

    }

    /**
     * Main.
     */
    public static void main(String[] args) throws IOException, HLException, InterruptedException {

        if(args.length < 2) {

            printUsage();
            System.exit(1);

        }

        int offset = parseOptions(args);

        RemoteFile[] remoteFiles = null;

        try {

            remoteFiles = parseArgs(args, offset);

        } catch(IllegalArgumentException e) {

            System.out.println("rlget: illegal argument: " + e.getMessage());
            System.exit(1);

        }
        
        File local = new File(args[args.length - 1]);

        if(args.length > 2) {

           if(!local.isDirectory()) {
               
               System.out.println("rlget: When copying multiple files last argument must be a directory.");
               System.exit(1);
               
           }
           
           if(!local.exists()) {
               
               System.out.println("rlget: Destination directory does not exist.");
               System.exit(1);

           }

        }

        int errors = 0;

        for(int i = 0; i < remoteFiles.length; i++) {

            HLClient hlc = null;

            try {
                
                hlc = new HLClient(remoteFiles[i].host);
                hlc.connect();
                hlc.login(remoteFiles[i].user, 
                          remoteFiles[i].pass, 
                          "rlget",
                          100);
                
                String remotePath = 
                    remoteFiles[i].file.replace('/', HLProtocol.DIR_SEPARATOR);
                
                String remoteFile =
                    remotePath.substring(remotePath.lastIndexOf(HLProtocol.DIR_SEPARATOR) + 1);
                    
                localFile = local;
                
                if(localFile.isDirectory())
                    localFile = new File(local, FilenameUtils.qualify(remoteFile));
                
                MacFile mf = new SplitMacFile(localFile,
                                              new Integer(MacFile.READ_PERM | 
                                                          MacFile.WRITE_PERM));
                
                int id = hlc.requestFileDownload(remoteFile, 
                                                 mf, 
                                                 meter, 
                                                 false);
                
                hlc.waitFor(id);
                
            } catch(Exception e) {

                String reason = e.toString();

                if(e.getMessage() != null)
                    reason = e.getMessage();

                if(progressShown)
                    verboseln();

                verboseln("rlget: " + localFile + ": " + reason);

                errors++;

            } finally {

                if(hlc != null)
                    hlc.close();

            }

        }

        System.exit(-errors);
        
    }

    static void verbose(String s) {

        if(verbose)
            System.out.print(s);

    }

    static void verboseln(String s) {

        if(verbose)
            System.out.println(s);

    }

    static void verboseln() {

        if(verbose)
            System.out.println();

    }

    static Meter meter = new Meter() {
            
            MeterSource ms;
            String file;
                       
            public void startMeter(MeterSource ms, 
                                   int key, 
                                   String s, 
                                   long size) {
                
                this.ms = ms;
                this.file = s;
                verboseln("rlget: " + localFile + 
                                 ": " + size + " bytes.");
                progressShown = false;

            }
            
            public void progressMeter(int key, 
                                      long done) {
                
                if(showProgress)
                    verbose("rlget: " + localFile + 
                            ": " + done + "                     \r");
                
                progressShown = true;
                
            }
            
            public void stopMeter(int key) {
                
                if(progressShown)
                    verboseln();

                verboseln("rlget: " + localFile + 
                        ": transfer complete.             ");
                
            }
            
            public void stopMeterWithError(int key, Throwable t) {

                if(progressShown)
                    verboseln();

                verboseln("rlget: " + localFile + 
                        ": error: " + t + "           ");
                
            }
            
            public MeterSource getMeterSource() {
                
                return ms;
                
            }
            
        };
    
}


class RemoteFile {
    
    String user = "";
    String pass = "";
    String host = null;
    String file = null;
    
    public String toString() {
        
        return "RemoteFile[user = " + user + ", pass = " + pass + ", host = " + host + ", file = " + file + "]";
        
    }
    
}
    
