/**
 * $Id: HLServerPolicy.java,v 1.8 2001/09/15 01:05:33 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

package redlight.hotline;

import java.net.InetAddress;
import java.text.DateFormat;
import java.util.Hashtable;
import java.util.Calendar;
import java.util.Date;
import java.io.File;
import java.io.IOException;

import redlight.utils.QuickSort;
import redlight.crypto.UnixCrypt;
import redlight.utils.TimeFormat;
import redlight.utils.TextUtils;
import redlight.utils.DebuggerOutput;
import redlight.macfiles.MacFile;
import redlight.macfiles.SplitMacFile;

/**
 * This class implements basic Hotline server policies, such as
 * approving login and privilege checking. More or less sophisticated
 * policies can be implemented by subclassing this class and
 * overriding some or all of the <tt>approveXXX</tt> methods. These
 * methods all have a roughly identical signature: they take a client
 * and some parameters as their arguments, and return either null or
 * an error message. <p>
 *
 * Usually, the subclass should invoke their parent method and go from
 * there; that is, return the result from the parent method if it's
 * non-null, or apply their own rules if it's null. This has the
 * effect of allowing you to implement more strict or elaborate
 * policies on top of the basic ones, which is generally what you
 * want. <p>
 *
 * If on the other hand you want to loosen a restriction, you'll have
 * to reimplement the corresponding <tt>approveXXX</tt> method. If you
 * go that route, you are also responsible for emulating all the
 * side-effects of the original behaviour (checking for the existence
 * of files, etcetera -- see HLServerDispatcher.java for the
 * assumptions). <p>
 *
 * An instance of this class is set as the policy manager for a
 * server using the {@link HLServer#setHLServerPolicy} method.<p> 
 */
public class HLServerPolicy {
    
    HLServer hls;

    /**
     * Creates a default policy manager.
     * @param h the HLServer that this policy applies to.
     */
    public HLServerPolicy(HLServer h) {

        if(h == null)
            throw new IllegalArgumentException("h == null");

        hls = h;

    }

    /**
     * Returns a new policy object to associate with a new
     * HLServerDispatcher. The policy object is used as an easy
     * way to keep track of policy state for different clients.
     */
    static public Object createPolicyObject() {
        
        return new DefaultPolicyObject();
        
    }

    /**
     * Determines the current state of authentication for
     * the given client (looks it up in the policy object).
     */
    public boolean isAuthenticated(HLServerDispatcher client) {

        return ((DefaultPolicyObject) client.policyObject).isAuthenticated;

    }

    /**
     * Using the policy object to keep state information, we
     * can apply some very useful rules to approve / disapprove 
     * certain packets and immediately disconnect the client.
     * If this method returns null, the packet is considered
     * approved. If this method returns something, the client
     * is disconnected and the returned message is logged.
     * @param client the client.
     * @param packet the packet to approve.
     */
    public String approvePacket(HLServerDispatcher client, 
                                HLProtocol.Packet packet) {
        
        /* Require authentication for any packet but the login
           packet. */

        if(!isAuthenticated(client) && 
           packet.header.id != HLProtocol.HTLC_HDR_LOGIN)            
            return "Disconnecting client because an action of type 0x" + Integer.toHexString(packet.header.id) + " was attempted before login.";
                
        /* This user is a nickname thief and it doesn't look
           like he's about to change his ways ... Kick him. */
            
        if(((DefaultPolicyObject) client.policyObject).isNicknameThief && 
           packet.header.id != HLProtocol.HTLC_HDR_USER_CHANGE)
            return "Disconnecting client for nickname stealing.";
            
        return null;

    }

    /**
     * Called to authenticate a user.
     * @param client the client logging in.
     * @return null if approved, error message otherwise. 
     */
    public String approveLogin(HLServerDispatcher client) {
        
        if(((DefaultPolicyObject) client.policyObject).isAuthenticated)
            return "Disconnecting client because login request received while already logged in.";
        
        return authenticate(client);
        
    }
    
    /**
     * Called when a client wants to change user state (ie. nick, icon). 
     * The nick component in the state argument may be null.
     * @param client the client changing user state.
     * @param state the new user state that needs approval.
     * ({@see HLProtocol.UserListComponent#CAN_DISCONNECT_USERS}).  
     * @return null if approved, error message otherwise.
     */
    public String approveUserChange(HLServerDispatcher client,
                                    HLProtocol.UserListComponent state) {

        /* Check whether this client is trying to steal someone else's
           nickname, and if so, mark him as a nickname thief by setting
           the appropriate flag in the client object. */
        
        HLServerDispatcher[] clients = hls.getClients();
        
        for(int i = 0; i < clients.length; i++) {
            
            synchronized(clients[i].userStateLock) {
                
                if(client != clients[i] &&
                   clients[i].nick.equals(state.nick)) {

                    ((DefaultPolicyObject) client.policyObject).isNicknameThief = true;
                    return "You cannot change your nickname to look like someone else's nickname. Change your nickname NOW.";

                }

            }    

        }
        
        return null;
        
    }

    /**
     * Approves whether one user can kick another.
     * @param client the user kicking.
     * @param sock the socket id of the user to kick.
     * @param ban whether to ban the user to kick.
     * @return null if approved, error message otherwise.
     */
    public String approveUserKick(HLServerDispatcher client,
                                  int sock,
                                  boolean ban) {
        
        if((long) (client.privileges & 
            HLProtocol.AccountInfo.CAN_DISCONNECT_USERS) == 
           HLProtocol.AccountInfo.CAN_DISCONNECT_USERS)
            return null;
        
        return "You are not allowed to disconnect users.";
        
    }

    /**
     * Approves a chat line.
     * @param client the client requesting to chat.
     * @param chat the chat content that needs approval.
     * @param isAction whether or not this is an "action".
     * @return null if approved, error message otherwise.
     */
    public String approveChat(HLServerDispatcher client,
                              String chat,
                              boolean isAction) {

        if((long) (client.privileges & HLProtocol.AccountInfo.CAN_SEND_CHAT) == 
           HLProtocol.AccountInfo.CAN_SEND_CHAT) 
            return null;

        return "You are not allowed to send chat.";

    }

    /**
     * Approves message transmit between users.
     * @param sender the sender of the message.
     * @param recipient the socket id of the intended recipient 
     * of the message.
     * @param message the message.
     * @return null if approved, error message otherwise.
     */
    public String approveMessage(HLServerDispatcher sender,
                                 int sock,
                                 String message) {

        return null;

    }

    /**
     * Approves a flat news request.
     * @param client the client requesting flat news.
     * @return null if approved, error message otherwise.
     */
    public String approveFlatnewsGet(HLServerDispatcher client) {

        if((long) (client.privileges & HLProtocol.AccountInfo.CAN_READ_NEWS) == 
           HLProtocol.AccountInfo.CAN_READ_NEWS)
            return null;

        return "You are not allowed to read news.";

    }
                                     
    /**
     * Approves a flat news post.
     * @param client the client requesting flatnews post.
     * @param newPost the post requiring approval.
     * @return null if approved, error message otherwise.
     */
    public String approveFlatnewsPost(HLServerDispatcher client,
                                      String newPost) {

        if((long) (client.privileges & HLProtocol.AccountInfo.CAN_POST_NEWS) == HLProtocol.AccountInfo.CAN_POST_NEWS)
            return null;
        
        return "You are not allowed to post news.";
        
    }

    /**
     * Called when a client requests to read an account. The accounts
     * table is locked on entry to this method.
     * @param client the user requesting account create.
     * @param login the account to read.
     * @return null if approved, error message otherwise.
     */
    public String approveAccountRead(HLServerDispatcher client, 
                                     String login) {

        if((long) (client.privileges & HLProtocol.AccountInfo.CAN_READ_ACCOUNTS) == HLProtocol.AccountInfo.CAN_READ_ACCOUNTS)         
            return null;

        return "You are not allowed to read accounts.";

    }

    /**
     * Called when a client requests account creation. The accounts
     * table is locked on entry to this method.
     * @param client the user requesting account create.
     * @param account the account that is about to be created.
     * @return null if the account is allowed to be created,
     * an error string otherwise.
     */
    public String approveAccountCreate(HLServerDispatcher client, 
                                       HLProtocol.AccountInfo account) {
        
        if((long) (client.privileges & HLProtocol.AccountInfo.CAN_CREATE_ACCOUNTS) == 
           HLProtocol.AccountInfo.CAN_CREATE_ACCOUNTS) {
            
            if((~client.privileges & account.privileges) == 0)
                return approveAccountCreate(account);
            
            return "Cannot create an account to have more rights than you have.";
            
        } else {
            
            return "You are not allowed to create accounts.";

        }

    }

    /**
     * Should approve account creation independantly of 
     * who is asking.
     * @param account the account that is about to be created.
     * @return null if the account is allowed to be created,
     * an error string otherwise.
     */
    public String approveAccountCreate(HLProtocol.AccountInfo account) {
        
        return null;

    }
    
    /**
     * Called when a client requests account modification. The accounts
     * table is locked on entry to this method.
     * @param client the user requesting account modification.
     * @param oldAccount the old state for the account.
     * @param newAccount the new state for the account.
     * @return null if the account is allowed to be created,
     * an error string otherwise.
     */
    public String approveAccountModify(HLServerDispatcher client,
                                       HLProtocol.AccountInfo oldAccount,
                                       HLProtocol.AccountInfo newAccount) {

        if((long) (client.privileges & HLProtocol.AccountInfo.CAN_MODIFY_ACCOUNTS) == HLProtocol.AccountInfo.CAN_MODIFY_ACCOUNTS) {
            
            if((~client.privileges & oldAccount.privileges) != 0)
                return "You cannot modify an account that is more powerful than yours.";

            if((~client.privileges & newAccount.privileges) != 0)
                return "You cannot grant privileges that you do not have.";
            
            return approveAccountModify(newAccount);

        } else {
            
            return "You are not allowed to modify accounts.";

        }

    }
    
    /**
     * Should approve account modification independantly of 
     * who is asking.
     * @param account the new state of the account.
     * @return null if the account is allowed to be modified,
     * an error string otherwise.
     */
    public String approveAccountModify(HLProtocol.AccountInfo account) {
        
        return null;

    }

    /**
     * Called when a client requests account deletion. The accounts
     * table is locked on entry to this method.
     * @param client the user requesting account deletion.
     * @param account the account that is about to be deleted.
     * @return null if the account is allowed to be deleted,
     * an error string otherwise.
     */
    public String approveAccountDelete(HLServerDispatcher client, 
                                       HLProtocol.AccountInfo account) {

        if((~client.privileges & account.privileges) != 0)
            return "You cannot delete an account that is more powerful than yours.";
        if((long) (client.privileges & HLProtocol.AccountInfo.CAN_DELETE_ACCOUNTS) == HLProtocol.AccountInfo.CAN_DELETE_ACCOUNTS)
            return approveAccountDelete(account);

        return "You are not allowed to delete accounts.";

    }

    /**
     * Should approve account deletion independantly of 
     * who is asking.
     * @param account the account that is about to be deleted.
     * @return null if the account is allowed to be deleted,
     * an error string otherwise.
     */
    public String approveAccountDelete(HLProtocol.AccountInfo account) {
        
        return null;

    }

    /**
     * Called when a client requests a directory listing. This method
     * resolves the requested Hotline path to a host path name, taking
     * into account the user's home directory, then checks whether the
     * host path can be found and whether it is a directory. If the
     * path name contains the words "drop box" (case insensitive),
     * then there is an additional check whether the client has
     * permission to view drop boxes. Subclasses must always confirm
     * that the requested path resolves to an existing directory,
     * unless they also override {@link obtainDirectoryListing} in
     * such a way that it does not depend on those facts.
     * @param client the user requesting the listing.
     * @param path the (Hotline, ie. ":" separated) path requested.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approveDirectoryListing(HLServerDispatcher client,
                                          String path) {

        MacFile hostPath = null;

        try {

            hostPath = resolvePath(client, path);
            
            if(!(hostPath.exists() && hostPath.isDirectory()))
                return "There is no directory " + path;
            
            /* Test whether the path contains the word "drop box", and if so,
               verify that the user has permissions to view drop boxes. */
            
            if(path.toLowerCase().indexOf("drop box") != -1) {
                
                if((long) (client.privileges & HLProtocol.AccountInfo.CAN_VIEW_DROPBOXES) == HLProtocol.AccountInfo.CAN_VIEW_DROPBOXES)
                    return null;
                
                return "You are not allowed to view drop boxes.";
                
            }
            
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

        return null;
        
    }

    /**
     * Called when a client requests a file download. If the user has
     * privileges to download files, this method resolves the
     * requested Hotline path to a host path name, taking into account
     * the user's home directory, and verifies that the file exists
     * and is indeed a file. 
     * @param client the user requesting the listing.
     * @param path the (Hotline, ie. ":" separated) path requested.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approveFileDownload(HLServerDispatcher client,
                                      String path,
                                      String file) {

        if(!((long) (client.privileges & HLProtocol.AccountInfo.CAN_DOWNLOAD_FILES) == HLProtocol.AccountInfo.CAN_DOWNLOAD_FILES))
            return "You are not allowed to download files.";

        MacFile hostPath = null;

        try {

            hostPath = resolvePathOrPartial(client, path + file);
            
            if(!hostPath.exists())
                return "You cannot download " + file + " because it is missing or temporarily unavailable.";

            if(hostPath.isDirectory())
                return "You cannot download directories.";
            
            return null;
                        
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

    }

    /**
     * Called when a client wants to upload a file to the server. This
     * method first checks that the client has permission to upload
     * files, then verifies that there is not currently a file or
     * directory at the requested path (or an error is returned). When
     * the resume flag is false, this method also checks for the
     * existence of the file with a .hpf extension, and returns an
     * error message if there is. This means resume must be explicitly
     * requested by the client; if this method would not perform the
     * check, every upload would of a file with an existing name
     * becomes an upload. Permission is granted in case the
     * destination path name contains the word "upload", or if the
     * user has permission to upload anywhere.
     * @param client the user requesting the listing.
     * @param path the (Hotline, ie. ":" separated) path requested.
     * @param resume whether the client wants to resume a partially
     * completed transfer.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approveFileUpload(HLServerDispatcher client,
                                    String path,
                                    String file,
                                    boolean resume) {

        if(!((long) (client.privileges & HLProtocol.AccountInfo.CAN_UPLOAD_FILES) == HLProtocol.AccountInfo.CAN_UPLOAD_FILES))
            return "You are not allowed to upload files.";

        MacFile hostPath = null;

        try {

            hostPath = resolvePath(client, path + file, true);
            
            if(hostPath.exists() || hostPath.isDirectory())
                return "Can't upload " + file + " because a file or directory by that name already exists.";

            if(!resume) {

                MacFile tempFile = null;

                try {

                    tempFile = resolvePath(client, path + file + ".hpf", true);
                    
                    if(tempFile.exists() || tempFile.isDirectory())
                        return "Can't upload " + file + "; to resume a partially completed transfer, try reloading the file view, then uploading the file again.";

                } finally {

                    try {

                        if(tempFile != null)
                            tempFile.close();

                    } catch(IOException e) {}

                }

            }

            /* Test whether the path contains the word "upload" or
               "drop box" -- if not verify that the user has
               permissions to upload anywhere. */
            
            if(path.toLowerCase().indexOf("upload") == -1 &&
               path.toLowerCase().indexOf("drop box") == -1) {
                
                if((long) (client.privileges & HLProtocol.AccountInfo.CAN_UPLOAD_ANYWHERE) == HLProtocol.AccountInfo.CAN_UPLOAD_ANYWHERE)
                    
                    return null;
                
                else
                    
                    return "You are only allowed to upload to 'Uploads' folders and drop boxes.";

            }
            
            return null;
                        
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

    }

    /**
     * Called when a client requests a file / directory deletion.
     * When a file is deleted, it is not actually removed, but instead
     * moved by the server to a "trash" directory ({@see
     * {HLServer#getTrashDirectory}). This method grants permission
     * when the user has the required permissions, when the trash
     * directory exist, and when the target object exists.
     * @param client the user requesting the listing.
     * @param path the (Hotline, ie. ":" separated) path requested.
     * @param file the file requested.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approveFileDelete(HLServerDispatcher client,
                                    String path,
                                    String file) {

        MacFile hostPath = null;

        try {

            hostPath = resolvePathOrPartial(client, path + HLProtocol.DIR_SEPARATOR + file);
            
            if(hostPath.isDirectory())
                if((long) (client.privileges & HLProtocol.AccountInfo.CAN_DELETE_FOLDERS) != HLProtocol.AccountInfo.CAN_DELETE_FOLDERS)
                    return "You are not allowed to delete folders.";
            
            if((long) (client.privileges & HLProtocol.AccountInfo.CAN_DELETE_FILES) != HLProtocol.AccountInfo.CAN_DELETE_FILES)
                return "You are not allowed to delete files.";
            
            if(!hostPath.exists()) 
                return "Cannot delete " + file + " because the file or directory does not exist.";
            
            if(!hls.getTrashDirectory().exists())
                return "Cannot delete " + file + " because the trash directory does not exist. Contact the server administrator.";
            
            return null;
                
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

    }

    /**
     * Called when a client requests a file / directory move.
     * @param client the user requesting the listing.
     * @param source the (Hotline, ie. ":" separated) path requested.
     * @param target the file requested.
     * @return null if the file move is allowed,
     * an error string otherwise.  
     */
    public String approveFileMove(HLServerDispatcher client,
                                  String source,
                                  String target) {

        MacFile sourceFile = null;
        MacFile targetFile = null;

        try {

            sourceFile = resolvePathOrPartial(client, source);
            targetFile = resolvePathOrPartial(client, target);
            
            if(sourceFile.isDirectory())
                if((long) (client.privileges & HLProtocol.AccountInfo.CAN_MOVE_FOLDERS) != HLProtocol.AccountInfo.CAN_MOVE_FOLDERS)
                    return "You are not allowed to move folders.";
            
            if((long) (client.privileges & HLProtocol.AccountInfo.CAN_MOVE_FILES) != HLProtocol.AccountInfo.CAN_MOVE_FILES)
                return "You are not allowed to move files.";
            
            if(!sourceFile.exists()) 
                return "Cannot move " + source + " because the file or directory does not exist.";
            
            if(!targetFile.exists())
                return "Cannot move " + source + " because the destination directory does not exist.";

            if(!targetFile.isDirectory())
                return "Cannot move " + source + " because the destination is not a directory.";
            
            return null;
                
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(sourceFile != null)
                    sourceFile.close();

            } catch(IOException e) {}

            try {

                if(targetFile != null)
                    targetFile.close();

            } catch(IOException e) {}

        }

    }

    /**
     * Called when a client wants to create a directory.
     * @param client the user requesting the listing.
     * @param path the (Hotline, ie. ":" separated) path requested.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approveDirectoryCreate(HLServerDispatcher client,
                                         String path,
                                         String file) {

        if(!((long) (client.privileges & HLProtocol.AccountInfo.CAN_CREATE_FOLDERS) == HLProtocol.AccountInfo.CAN_CREATE_FOLDERS))
            return "You are not allowed to create directories.";

        /* Eh, perhaps some of this should be moved to a static method
           in SplitMacFile -- e.g. isValidName() */

        if(file.equals("..") || file.equals(".") || file.endsWith(".hpf") ||
           (file.startsWith(".") && 
            (file.endsWith(".rsrc") || file.endsWith(".info"))))
            return "Sorry, that name is not allowed.";

        MacFile hostPath = null;

        try {

            hostPath = resolvePath(client, path + HLProtocol.DIR_SEPARATOR + file, true);

            if(hostPath.exists()) 
                return "Cannot create directory " + path + " because a file or directory by that name already exists.";

            return null;
                
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

    }

    /**
     * Called when a client wants to get file info.
     * @param client the user requesting the listing.
     * @param path the (Hotline, ie. ":" separated) path requested.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approveFileGetInfo(HLServerDispatcher client,
                                     String path,
                                     String file) {

        MacFile hostPath = null;

        try {

            hostPath = resolvePathOrPartial(client, path + HLProtocol.DIR_SEPARATOR + file);

            if(!hostPath.exists())
                return "Cannot get info on " + path + HLProtocol.DIR_SEPARATOR + file + " because that file or directory does not exist.";

            return null;
                
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

    }

    /**
     * Called when a client wants to set file info.
     * @param client the user requesting the listing.
     * @param path the (Hotline, ie. ":" separated) path requested.
     * @param file the file requested.
     * @param newName the new name for the file or null.
     * @param newComment the new comment for the file or null.
     * @return null if the file info change is allowed,
     * an error string otherwise.  
     */
    public String approveFileSetInfo(HLServerDispatcher client,
                                     String path,
                                     String file,
                                     String newName,
                                     String newComment) {

        MacFile hostPath = null;

        try {

            hostPath = resolvePathOrPartial(client, path + HLProtocol.DIR_SEPARATOR + file);

            if(!hostPath.exists())
                return "Cannot set info on " + file + " because that file or directory does not exist.";

            /* Check whether renaming or commenting the file /
               directory is allowed. */
                    
            if(hostPath.isDirectory()) {
                
                if(newName != null)
                    if((long) (client.privileges & HLProtocol.AccountInfo.CAN_RENAME_FOLDERS) != HLProtocol.AccountInfo.CAN_RENAME_FOLDERS)
                        return "You are not allowed to rename folders.";

                if(newComment != null)
                    if((long) (client.privileges & HLProtocol.AccountInfo.CAN_COMMENT_FOLDERS) != HLProtocol.AccountInfo.CAN_COMMENT_FOLDERS)
                        return "You are not allowed to comment folders.";

            } else {

                if(newName != null)
                    if((long) (client.privileges & HLProtocol.AccountInfo.CAN_RENAME_FILES) != HLProtocol.AccountInfo.CAN_RENAME_FILES)
                        return "You are not allowed to rename files.";

                if(newComment != null)
                    if((long) (client.privileges & HLProtocol.AccountInfo.CAN_COMMENT_FILES) != HLProtocol.AccountInfo.CAN_COMMENT_FILES)
                        return "You are not allowed to comment files.";

            }

            if(newName != null) {

                if(newName.endsWith(".hpf"))
                    return "You are not allowed to create files with the .hpf extension .hpf.";
                
                if(newName.startsWith(".") && 
                   (newName.endsWith(".info") || newName.endsWith(".rsrc")))
                    return "You are not allowed to create files with the .info and .rsrc extensions.";
                
            }

            return null;
                
        } catch(IOException e) {

            return e.toString();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

    }

    /**
     * Called when a client wants to create a private chat.
     * @param client the user requesting the listing.
     * @param sock the invited user.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approvePrivateChatCreate(HLServerDispatcher client,
                                           int sock) {

        return null;

    }

    /**
     * Called when a client wants to join a private chat.
     * @param client the user requesting the listing.
     * @param refObject the chat reference.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approvePrivateChatJoin(HLServerDispatcher client, Object refObject) {

        return null;

    }

    /**
     * Called when a client invites another user to a private chat.
     * @param client the user requesting the listing.
     * @param refObject the chat reference.
     * @param sock the socket of the user being invited.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approvePrivateChatInvite(HLServerDispatcher client, Object refObject, int sock) {

        return null;

    }

    /**
     * Called when a client wants to change the subject on a private
     * chat.
     * @param client the user requesting the listing.
     * @param refObject the chat reference.
     * @param subject the new subject.
     * @return null if the directory listing is allowed,
     * an error string otherwise.  
     */
    public String approvePrivateChatSubject(HLServerDispatcher client, Object refObject, String subject) {

        return null;

    }

    /**
     * Approves whether one user can get info on another user.
     * @param client the user getting info.
     * @param sock the socket id of the user to get info on.
     * @return null if approved, error message otherwise.
     */
    public String approveUserGetInfo(HLServerDispatcher client,
                                     int sock) {
        
        DebuggerOutput.debug("approveUserGetInfo: client.privileges = " + client.privileges + ", HLProtocol.AccountInfo.CAN_GET_USER_INFO = " + (long) HLProtocol.AccountInfo.CAN_GET_USER_INFO + ", client.privileges & HLProtocol.AccountInfo.CAN_GET_USER_INFO = " + (long) (client.privileges & HLProtocol.AccountInfo.CAN_GET_USER_INFO));
        if((long) (client.privileges & 
                   HLProtocol.AccountInfo.CAN_GET_USER_INFO) == 
           HLProtocol.AccountInfo.CAN_GET_USER_INFO)
            return null;
        
        return "You are not allowed to get user info.";
        
    }

    /**
     * Returns meta-data about a file, as far as it can be determined.
     * @param client the client needing meta data.
     * @param path the path of the file.
     * @param file the object (file or directory).
     */
    public HLProtocol.FileInfo obtainFileInfo(HLServerDispatcher client,
                                              String path,
                                              String file) throws IOException {
        
        MacFile hostPath = null;
        
        try {
            
            hostPath = resolvePathOrPartial(client, path + HLProtocol.DIR_SEPARATOR + file);
            HLProtocol.FileInfo fileInfo = hls.hlp.new FileInfo();
            
            fileInfo.name = hostPath.getFile().getName();
            fileInfo.comment = hostPath.getComment();
            fileInfo.icon = hostPath.getType();
            fileInfo.size = hostPath.length();
            fileInfo.creator = hostPath.getCreator();
            fileInfo.type = hostPath.getType();
            fileInfo.created = hostPath.getCreationDate();
            fileInfo.modified = hostPath.getModificationDate();
            
            return fileInfo;

        } finally {
            
            if(hostPath != null) {
                
                try {
                    
                    hostPath.close();
                    
                } catch(IOException e) {}
                
            }
            
        }        

    }

    /**
     * Returns an array of {@link HLProtocol.FileListComponent}
     * objects describing the files in the requested directory.
     * Assumes that the given path resolves to an existing directory
     * in the host filesystem.
     * @param client the user making the request.
     * @param path the requested path.
     */
    public HLProtocol.FileListComponent[] obtainDirectoryListing(HLServerDispatcher client, String path) throws IOException {

        /* Get a list of files from the specified directory
           (depends on approveDirectoryListing to confirm that
           the specified directory exists and is in fact a
           directory). */

        HLProtocol.FileListComponent[] list = 
            new HLProtocol.FileListComponent[0];

        MacFile hostPath = null;

        try {

            hostPath = resolvePath(client, path);
            String[] hostList = hostPath.list();
            
            list = new HLProtocol.FileListComponent[hostList.length];
            
            for(int i = 0; i < hostList.length; i++) {

                DebuggerOutput.debug("obtainDirectoryListing: creating MacFile for " + hostList[i]);
                MacFile f = null;

                try {

                    /* (Creating all these MacFile's is pathologically
                       slow, have to see if that can be sped up).
                       (SplitMacFile is okay now, others probably
                       broken) */

                    f = new SplitMacFile(new File(hostPath.getFile(), 
                                                  hostList[i]), 
                                         new Integer(MacFile.READ_PERM));
                
                    String ft = f.getType();
                    String fc = f.getCreator();
                    int fs = 0;
                    
                    if(f.isDirectory()) {
                        
                        /* Directories need special information to
                           distinguish them from ordinary files. */
                        
                        ft = new String("fldr");
                        fc = new String("\0\0\0\0");
                        String[] ftl = f.list();
                        fs = ftl.length;
                        
                    } else {
                    
                        /* Otherwise get the length of this file. */
                        
                        fs = (int) f.length();
                        
                        if(fs < 0)
                            fs = 0;
                        
                        /* Don't show .hpf extension. */

                        if(f.getFile().getName().endsWith(".hpf")) 
                            hostList[i] = hostList[i].substring(0, hostList[i].length() - 4);
                        
                    }
                
                    if(fc.length() != 4) {

                        hls.log("File creator for '" + hostList[i] + "' is bogus (" + fc + "), substituting ????.");
                        fc = "????";

                    }

                    if(ft.length() != 4) {

                        hls.log("File type for '" + hostList[i] + "' is bogus (" + ft + "), substituting ????.");
                        ft = "????";

                    }

                    HLProtocol.FileListComponent dh = hls.hlp.new FileListComponent(hostList[i], ft, fc, fs, 0);

                    list[i] = dh;
                 
                } catch(IllegalArgumentException e) {

                    DebuggerOutput.stackTrace(e);

                } finally {
                    
                    try {
                        
                        if(f != null)
                            f.close();

                    } catch(IOException e) {}

                }

            }

        } catch(IOException e) {

            e.printStackTrace();

        } finally {

            try {

                if(hostPath != null)
                    hostPath.close();

            } catch(IOException e) {}

        }

        QuickSort.sort(list, 0, list.length - 1);

        return list;

    }

    /**
     * Should return a formatted chat string.
     * @param client the user who requests to chat.
     * @param chat the chat string.
     * @param isAction whether this is an action or not.
     */
    static public String formatChat(HLServerDispatcher client,
                                    String chat,
                                    boolean isAction) {

        if(isAction) 
            return "\r *** " + new String(client.getUser().nick) + " " + chat;

        String nick = TextUtils.prepad(new String(client.getUser().nick), 12);

        return "\r" + nick.substring(0, nick.length() > 13 ? 13 : nick.length()) + ":  " + chat;

    }
    
    /**
     * Should return a formatted flat news post.
     * @param client the user posting the news.
     * @param newPost the new post that was made.
     */
    public String formatFlatnewsPost(HLServerDispatcher client, 
                                     String newPost) {
        
        return "---------- " + 
            TimeFormat.formatCurrentDateTime(DateFormat.LONG, 
                                             DateFormat.MEDIUM) + 
            " [" + new String(client.getUser().nick) + "]" + 
            " ----------\r \r" + newPost + "\r \r";
        
    }

    /**
     * Resolves a Hotline path for the given client to a directory
     * on the host filesystem.
     * @param client the client to resolve for.
     * @param path the path to resolve.
     */
    public MacFile resolvePath(HLServerDispatcher client,
                               String path) throws IOException {

        return resolvePath(client, path, false);

    }

    /**
     * Resolves a Hotline path for the given client to a file / directory
     * on the host filesystem. The file / directory need not exist.
     * @param client the client to resolve for.
     * @param path the path to resolve.
     * @param writable whether to open the path for writing.
     * @return MacFile object describing the requested file / directory.
     */
    public MacFile resolvePath(HLServerDispatcher client,
                               String path,
                               boolean writable) throws IOException {

        Integer perm = new Integer(writable ? 
                                   MacFile.READ_PERM | MacFile.WRITE_PERM : 
                                   MacFile.READ_PERM);

        path = TextUtils.findAndReplace(path, ":..", "");
        MacFile hostPath = new SplitMacFile(new File(client.homeDirectory, TextUtils.findAndReplace(path, ":", System.getProperty("file.separator"))), perm);

        DebuggerOutput.debug("Hotline path '" + path + "' for client " + client + " resolves to host path '" + hostPath.toString() + "'.");

        return hostPath;

    }

    /**
     * Like resolvePath, returns a MacFile object describing the 
     * requested path. However, if the specified path points to
     * a non-existing file, this method assumes that the specified
     * path points to a "partial" file, ie. one with the .hpf 
     * extension, and returns a MacFile object for that partial
     * file instead.
     */
    public MacFile resolvePathOrPartial(HLServerDispatcher client,
                                        String path) throws IOException {
            
        /* First check whether this file exist. It may not exist,
           because the server may lie to the client about files
           with the .hpf extension. */
        
        MacFile hostPath = resolvePath(client, path);

        if(!hostPath.exists()) {
            
            /* Check whether the file exists if we append the .hpf 
               extension. */
            
            hostPath.close();
            hostPath = resolvePath(client, path + ".hpf");

            return hostPath;

        }

        return hostPath;

    }

    /**
     * Verifies whether the account specified by the class members 
     * client.login & client.password is valid, and if so sets up 
     * some class members in the client appropriately.
     * @param client the client to authenticate.
     * @return null for successfull authentication, error otherwise.
     */
    String authenticate(HLServerDispatcher client) {
        String authenticated = "Authentication failed.";

        try {

            HLServerAccountsTable accountsTable = hls.getAccountsTable();

            accountsTable.lock();
        
            DebuggerOutput.debug("Got login   : " + client.login);
            DebuggerOutput.debug("Got password: " + client.password);

            if(accountsTable.exists(client.login)) {

                String storedPassword = accountsTable.get(client.login).password;
                String salt = storedPassword.length() > 1 ? storedPassword.substring(0, 2) : "aa";

                DebuggerOutput.debug("Stored password: " + storedPassword);
                DebuggerOutput.debug("Got    password: " + UnixCrypt.crypt(salt, client.password));

                if(storedPassword.equals(UnixCrypt.crypt(salt, client.password))) {

                    /* Let her in ... */
                    
                    ((DefaultPolicyObject) client.policyObject).isAuthenticated = true;
                
                    authenticated = null;

                    /* Get the privileges from the accounts table. */

                    client.privileges = accountsTable.get(client.login).privileges;
                    
                    /* If this user has the permission to disconnect other
                       users, then adjust his status accordingly. */
                    
                    if((long) (client.privileges & 
                        HLProtocol.AccountInfo.CAN_DISCONNECT_USERS) == 
                       HLProtocol.AccountInfo.CAN_DISCONNECT_USERS)
                        client.color |= HLProtocol.UserListComponent.CAN_DISCONNECT_USERS;
                    
                    /* If this user does not have the permission to use
                       any name, set his nick to the nick stored in the
                       accounts table. */
            
                    if(!((long) (client.privileges & 
                          HLProtocol.AccountInfo.CAN_USE_ANY_NAME) == 
                         HLProtocol.AccountInfo.CAN_USE_ANY_NAME))
                        client.nick = accountsTable.get(client.login).nick;
                    
                    client.homeDirectory = accountsTable.get(client.login).homeDirectory;

                    if(client.homeDirectory == null)
                        client.homeDirectory = hls.getHomeDirectory().getPath();

                }

            }
   
            accountsTable.unlock();
            
        } catch(InterruptedException _e) {
            
            hls.log(client.client.getInetAddress(), "[internal error] Disconnecting client because of a failure to acquire the account lock.");
            client.disconnect();
            
        }

        return authenticated;

    }
        
}
