/**
 * $Id: HLIconFile.java,v 1.12 2001/09/29 18:45:00 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.awt.Color;
import java.awt.Toolkit;
import java.awt.Image;
import java.awt.image.MemoryImageSource;
import javax.swing.ImageIcon;
import javax.swing.Icon;
import java.util.Hashtable;
import java.io.File;
import java.io.IOException;

import redlight.utils.DebuggerOutput;
import redlight.hotline.iconlib.*;

/**
 * A HLIconFile is a kind of wrapper around the iconlib.* classes
 * by Markus Stengel. It also implements a little cache.
 */
public class HLIconFile {

    private Hashtable iconCache;
    private Color alpha;
    private Icon emptyIcon;
    private HLExtractor extractor;

    /* Pad the width of icons that are smaller than this size to 
       this size. */

    static int PADSIZE = 32;

    /**
     * Creates a HLIconFile, given a File which refers to a
     * Hotline .DAT file. The file must exist and actually
     * be a Hotline .DAT file, or an IOException is thrown.
     * @param f the .DAT file.
     */
    public HLIconFile(File f) throws IOException {

        extractor = new HLExtractor();

        try {

            extractor.quickStart(f.toString());
            iconCache = new Hashtable();
            alpha = new Color(0, 0, 0, 1);

            int[] emptyIconPixels = new int[PADSIZE * 16];
            emptyIcon = new ImageIcon(Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(PADSIZE, 16, emptyIconPixels, 0, PADSIZE)));

        } catch(IOException e) {
            
            extractor = null;
            throw e;

        }

    }

    /**
     * Gets an icon from the icon file or the icon cache, or returns
     * an empty 32x16 (transparent) placeholder image. This method
     * horizontally centers icons that are smaller than 32 pixel wide
     * and cuts off the first 215 pixels from icons that are bigger
     * than 215 pixels wide.
     * @return an Icon.  
     */
    public Icon getIconOrPlaceholder(int iconNumber) {

        Icon icon = getIcon(iconNumber);

        if(icon == null)
            return emptyIcon;

        return icon;

    }

    /**
     * Gets an icon from the icon file or the icon cache, or returns
     * null on error. This method horizontally centers icons that are
     * smaller than 32 pixel wide and cuts off the first 215 pixels
     * from icons that are bigger than 215 pixels wide.
     * @return an Icon, or null on error.  
     */
    public synchronized Icon getIcon(int iconNumber) {

        Integer cacheID = new Integer(iconNumber);

        /* Return icon from cache if possible. */

        if(iconCache.containsKey(cacheID))
            return (Icon) iconCache.get(cacheID);

        if(extractor == null)
            return null;

        if(!extractor.hasICON)
            return null;

        HLIconData iconData = null;

        try {

            iconData = extractor.getIcon(iconNumber);
            
        } catch(IOException e) {}
        
        if(iconData == null) 
            return null;
        
        int fpos = extractor.indexTable.get(extractor.idICONTable).find(iconNumber);
        HLData hld = extractor.indexTable.get(extractor.idICONTable).get(fpos);
            
        if (hld == null)
            return null;
        
        DebuggerOutput.debug("HLIconFile.getIcon: icon name: "  + hld.getName() + " identity: " + hld.getIdentity() + " width: " + iconData.getWidth() + " height: " + iconData.getHeight() + " palette length(b): " + iconData.getPalLengthBytes() + " filePos: " +hld.getDataOffset()+ " dataSize: " + hld.getDataSize());

        int sourceWidth = iconData.getWidth();
        int sourceHeight = iconData.getHeight();

        int destinationWidth = sourceWidth;

        if(sourceWidth < PADSIZE)
            destinationWidth = PADSIZE;

        if(sourceWidth > 215)
            destinationWidth = sourceWidth - 215;

        int[] destinationPixels = new int[destinationWidth * sourceHeight];
        int numSourcePixels = sourceWidth * sourceHeight;

        int destinationMargin = 0;

        if(iconData.getWidth() < PADSIZE)
            destinationMargin = (PADSIZE - sourceWidth) / 2;

        int destinationMarginTwice = destinationMargin * 2;
        int roundingErrorAdjustment = 0;

        if(destinationMarginTwice + sourceWidth < PADSIZE)
            roundingErrorAdjustment = PADSIZE - (destinationMarginTwice + sourceWidth);

        int destinationRowOffset = 0;
        int sourceRowOffset = 0;
        
        if(destinationMargin > 0)
            destinationRowOffset = destinationMargin + roundingErrorAdjustment;

        if(sourceWidth > 215)
            sourceRowOffset = 215;
         
        int[] pixelData = iconData.getRGBField();
        int sourceOffset = sourceRowOffset;

        /* Transfer the pixels. */

        for(int pixel = sourceRowOffset; pixel < numSourcePixels; pixel++) {
            
            if(pixel != 0 && pixel % sourceWidth == 0) {

                /* Next row. */

                destinationRowOffset += destinationMarginTwice + roundingErrorAdjustment;
                sourceOffset += sourceRowOffset;
                pixel += sourceRowOffset;

            }

            /* Copy a pixel. */

            destinationPixels[(pixel - sourceOffset) + destinationRowOffset] = (255 << 24) | pixelData[pixel];

            /* Make transparent and white pixels transparent. */

            if(pixelData[pixel] == iconData.getTransparency() ||
               pixelData[pixel] == 0x00ffffff)
                destinationPixels[(pixel - sourceOffset) + destinationRowOffset] = alpha.getRGB();
                
        }
        
        Icon icon = new ImageIcon(Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(destinationWidth, sourceHeight, destinationPixels, 0, destinationWidth)));

        /* FIXME: This keeps the cache from expanding forever but
           needs a real algorithm. */

        if(iconCache.size() > 80)
            iconCache.clear();

        iconCache.put(cacheID, icon);
        
        return icon;

    }

    /**
     * Given a HLProtocol.FileListComponent, returns an icon to 
     * represent that file.
     * @param file the file to look at. May be null to return
     * a default icon.
     */
    public Icon getIconForFile(HLProtocol.FileListComponent file) {
        
        Icon icon = getIconOrPlaceholder(400);
        
        if(file != null) {

            String fn = file.fileName.toLowerCase();

            if(file.fileType.equals("fldr"))
                icon = getIconOrPlaceholder(401);

            if(file.fileType.equals("fldr") && (fn.indexOf("drop box") != -1 || fn.indexOf("upload") != -1))
                icon = getIconOrPlaceholder(421);

            else if(file.fileType.equals("HTft") || fn.endsWith(".hpf"))
                icon = getIconOrPlaceholder(402);

            else if(file.fileType.equals("alis"))
                icon = getIconOrPlaceholder(422);

            else if(file.fileType.equals("APPL") || 
                    fn.endsWith(".exe") ||
                    fn.endsWith(".jar"))
                icon = getIconOrPlaceholder(407);

            else if(file.fileType.equals("GIFf") || 
                    fn.endsWith(".gif") ||
                    file.fileType.equals("JPEG") || 
                    fn.endsWith(".jpg") || 
                    fn.endsWith(".bmp") ||
                    fn.endsWith(".png"))
                icon = getIconOrPlaceholder(406);

            else if(file.fileType.equals("TEXT") ||
                    file.fileName.endsWith(".htm") ||
                    file.fileName.endsWith(".html"))
                icon = getIconOrPlaceholder(404);
                
            else if(file.fileCreator.equals("ttro") || fn.startsWith("readme"))
                icon = getIconOrPlaceholder(405);
                
            else if(file.fileType.equals("MPEG") || 
                    fn.endsWith(".mpg") || 
                    fn.endsWith(".mpeg") ||
                    file.fileCreator.equals("MooV") || fn.endsWith(".mov") ||
                    file.fileCreator.equals("VfW ") || fn.endsWith(".avi") ||
                    fn.endsWith(".asf")||
                    fn.endsWith(".divx") ||
                    fn.endsWith(".rm") || 
                    fn.endsWith(".ra") ||
                    fn.endsWith(".ram") ||
                    file.fileType.equals("PNRA") ||
                    file.fileType.equals("PNst"))
                icon = getIconOrPlaceholder(425);

            else if(fn.endsWith(".iso"))
                icon = getIconOrPlaceholder(423);
            
            else if(fn.endsWith(".img"))
                icon = getIconOrPlaceholder(423);
                
            else if(file.fileType.equals("AIFF") ||
                    fn.endsWith(".aif") ||
                    fn.endsWith(".aiff") ||
                    file.fileType.equals("MP3 ") ||
                    fn.endsWith(".mp3") ||
                    file.fileType.equals("WAVE") ||
                    fn.endsWith(".wav") ||
                    file.fileType.equals("SWFL") ||
                    fn.endsWith(".swf")) 
                icon = getIconOrPlaceholder(424);

            else if(file.fileType.equals("SIT!") ||
                    file.fileType.equals("SITD") ||
                    file.fileType.equals("SIT2") ||
                    fn.endsWith(".sit"))
                icon = getIconOrPlaceholder(403);

            else if(file.fileType.equals("SegM") ||
                    file.fileType.equals("Seg1") ||
                    file.fileType.equals("Seg2") ||
                    file.fileType.equals("Seg3") ||
                    file.fileType.equals("SegN"))
                icon = getIconOrPlaceholder(409);

            else if(file.fileType.equals("ZIP ") ||
                    fn.endsWith(".zip") ||
                    fn.endsWith(".lzh") ||
                    fn.endsWith(".gz") ||
                    fn.endsWith(".rar") ||
                    fn.endsWith(".arj") ||
                    fn.endsWith(".sit") ||
                    fn.endsWith(".cpt"))
                icon = getIconOrPlaceholder(426);

        }

        return icon;

    }
        
}


    
