/**
 * $Id: TrackerInterface.java,v 1.6 2001/09/21 03:05:05 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.UnknownHostException;
import java.net.InetAddress;
import java.util.Vector;
import java.util.Hashtable;
import java.util.StringTokenizer;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.DefaultListSelectionModel;

import redlight.hotline.*;
import redlight.utils.TextUtils;
import redlight.utils.FilenameUtils;
import redlight.utils.DebuggerOutput;
import redlight.graphics.Spinner;


/**
 * This class renders the column with the number of users.
 */
class TrackerTableCellRenderer extends JLabel 
    implements TableCellRenderer {

    static Color black = new Color(0), 
	white = new Color(0xffffff),
	grey = new Color(0xeeeedd);

    public Component getTableCellRendererComponent(JTable table,
						   Object value,
						   boolean isSelected,
						   boolean hasFocus,
						   int row,
						   int column) {

	setHorizontalAlignment(SwingConstants.RIGHT);
	setForeground(black);
	setText(value.toString());
	setBackground(grey);

	return this;

    }

    public void paint(Graphics g) {

	g.clearRect(0, 0, getWidth(), getHeight());
	super.paint(g);

    }

}

/**
 * This class presents an interface to a tracker.
 */
public class TrackerInterface extends AbstractTableModel
    implements Child, 
	       ActionListener, 
	       MouseListener, 
	       WindowListener {

    Parent parent;
    Container contentPane;
    JButton reload;
    JTable list;
    JTextField filter;
    Spinner spinner;
    JLabel info;
    JPopupMenu popupMenu;
    JMenuItem addToBookmarks, editServers, connectToServers;
    JButton removeTracker;
    JFrame f;
    JScrollPane scrollPane;
    JComboBox trackerChoice;
    Options rlo;
    String filtertext = "";
    Hashtable trackerGetters;
    TrackerGetter currentTrackerGetter;
    HLProtocol.ServerInfo[] serverList;
    HLProtocol.ServerInfo[] filteredServerList;

    public TrackerInterface(Parent p, Options opt) {

	parent = p;
	rlo = opt;

	Font guiFont = ((Font) Main.rlo.getProperty("Font.gui"));

	f = new JFrame("Trackers");
	contentPane = f.getContentPane();

	GridBagLayout gbl = new GridBagLayout();
	GridBagConstraints gbc = new GridBagConstraints();
	popupMenu = new JPopupMenu();
	reload = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.REFRESH_ICON]));
	filter = new JTextField(10);
	JPanel spinnerPanel = new JPanel();
	spinner = new Spinner(Main.rlo.getImageSet("Spinner"), 100, 16, 16);
	list = new JTable(this);
	scrollPane = new JScrollPane(list);
	info = new JLabel("");
	connectToServers = new JMenuItem("Connect to selected servers");
	addToBookmarks = new JMenuItem("Add selection to bookmarks");
	editServers = new JMenuItem("Edit selected servers ...");
	trackerChoice = new JComboBox(Main.rlo.getTrackers());
	removeTracker = new JButton("", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.TRASH_ICON]));
	JLabel l1 = new JLabel("Display servers containing:");

	spinnerPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
	contentPane.setLayout(gbl);
	addToBookmarks.setFont(guiFont);
	editServers.setFont(guiFont);
	connectToServers.setFont(guiFont);
	l1.setFont(guiFont);
	removeTracker.setFont(guiFont);
        removeTracker.setActionCommand("Remove");
	reload.setFont(guiFont);
        reload.setActionCommand("Reload");
        reload.setMnemonic('R');
        reload.setToolTipText("Reload the list of servers.");
	list.getColumnModel().getColumn(0).setPreferredWidth(150);
	list.getColumnModel().getColumn(1).setPreferredWidth(25);
	list.getColumnModel().getColumn(2).setPreferredWidth(500);
	list.setShowVerticalLines(false);
	list.setDefaultRenderer(Number.class, new TrackerTableCellRenderer());
	list.setFont(((Font) Main.rlo.getProperty("Font.list")));
        list.registerKeyboardAction(this, "Activate", 
                                    KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, 
                                                           0), 
                                    JComponent.WHEN_IN_FOCUSED_WINDOW);
	trackerChoice.setEditable(true);
	trackerChoice.addActionListener(this);
	trackerChoice.setToolTipText("To add a new tracker, enter it's address here and press return");
	info.setFont(((Font) Main.rlo.getProperty("Font.info")));
	removeTracker.setEnabled(Main.rlo.getTrackers().length > 1 ? 
				 true : false); 

	reload.addActionListener(this);
	filter.addActionListener(this);
	connectToServers.addActionListener(this);
	addToBookmarks.addActionListener(this);
	editServers.addActionListener(this);
	list.addMouseListener(this);
	removeTracker.addActionListener(this);
	f.addWindowListener(this);
	
	popupMenu.add(connectToServers);
	popupMenu.add(addToBookmarks);
	popupMenu.add(editServers);
	spinnerPanel.add(spinner);
	gbc.insets = (Insets) Main.rlo.getProperty("BorderInsets");
	gbc.fill = GridBagConstraints.NONE;
	gbc.anchor = GridBagConstraints.WEST;
	contentPane.add(reload, gbc);
	gbc.gridx = 1;
	contentPane.add(l1, gbc);
	gbc.gridx = 2;
	gbc.weightx = 1;
	gbc.fill = GridBagConstraints.HORIZONTAL;
	contentPane.add(filter, gbc);
	gbc.gridx = 4;
	gbc.weightx = 0;
	gbc.fill = GridBagConstraints.HORIZONTAL;
	contentPane.add(spinnerPanel, gbc);
	gbc.gridx = 0; gbc.gridy = 1;
	gbc.weightx = 1; gbc.weighty = 1;
	gbc.gridwidth = GridBagConstraints.REMAINDER;
	gbc.fill = GridBagConstraints.BOTH;
	contentPane.add(scrollPane, gbc);
	gbc.gridx = 0; gbc.gridy = 2;
	gbc.weightx = 1; gbc.weighty = 0;
	gbc.gridwidth = 4;
	contentPane.add(trackerChoice, gbc);	
	gbc.gridx = 4;
	gbc.weightx = 0; gbc.weighty = 0;
	gbc.anchor = GridBagConstraints.EAST;
	gbc.fill = GridBagConstraints.NONE;
	gbc.gridwidth = GridBagConstraints.REMAINDER;
	contentPane.add(removeTracker, gbc);
	gbc.anchor = GridBagConstraints.CENTER;
	gbc.gridx = 0; gbc.gridy = 3;
	gbc.fill = GridBagConstraints.BOTH;
	gbc.weightx = 0; gbc.weighty = 0;
	gbc.gridwidth = GridBagConstraints.REMAINDER;
	contentPane.add(info, gbc);

	f.pack();

	f.setLocation((Point) Main.rlo.getProperty("WindowLocation.Tracker"));
	f.setSize((Dimension) Main.rlo.getProperty("WindowDimension.Tracker"));

	f.show();

	trackerGetters = new Hashtable();

        for(int i = 0; i < Main.rlo.getTrackers().length; i++)
            trackerGetters.put(Main.rlo.getTrackers()[i], new TrackerGetter(Main.rlo.getTrackers()[i], this));

	trackerChoice.setSelectedIndex(getTrackerIndex(Main.rlo.getStringProperty("Tracker.Default")));


    }
    
    int getTrackerIndex(String s) {

	String[] trackers = Main.rlo.getTrackers();

	for(int i=0; i < trackers.length; i++)
	    if(s.equals(trackers[i]))
		return i;

	return 0;

    }
    
    void filterServerList() {

	String filtertext = filter.getText().toUpperCase();
        Vector filteredList = new Vector();

	/* We want to search for several words. */

	StringTokenizer st = new StringTokenizer(filtertext);
	int tokens = st.countTokens();
	String[] stComp = new String[tokens];
	int j = 0;

	while (st.hasMoreTokens()) {

            stComp[j] = st.nextToken();
            j++;

	}

	boolean found;

	for(int i = 0; i < serverList.length; i++) {

            found = true;

            if(tokens == 0) {
                
                filteredList.addElement(serverList[i]);
                
            } else {
                
                for(j = 0; j < stComp.length; j++) {

                    if(serverList[i].name.toUpperCase().indexOf(stComp[j]) == -1 &&
                       serverList[i].desc.toUpperCase().indexOf(stComp[j]) == -1) {
                        found = false;

                    }		

                }

                if(found) 
                    filteredList.addElement(serverList[i]);

            }            

        }

	filteredServerList = new HLProtocol.ServerInfo[filteredList.size()];

        for(int i = 0; i < filteredList.size(); i++)
            filteredServerList[i] = (HLProtocol.ServerInfo) filteredList.elementAt(i);

    }

    public void show() {

	f.show();
                
        if(list.getSelectionModel().isSelectionEmpty())
            if(list.getRowCount() > 0)
                list.getSelectionModel().setSelectionInterval(0, 0);
        
        list.requestFocus();

    }
    
    void itemEdit(int i) {

	new ConnectInterface(parent, 
                             filteredServerList[i].getAddress(), 
                             new Integer(filteredServerList[i].port).toString(), 
                             "guest", 
                             "", 
                             filteredServerList[i].name, 
                             true);
        
    }
    
    void itemActivated(int i) {

        new ConnectionInterface(filteredServerList[i].getAddress(),
                                filteredServerList[i].port, 
                                "guest", 
                                "",
                                filteredServerList[i].name);

    }

    synchronized void notifyThreadStarting(final TrackerGetter startingThread) {

        if(startingThread == currentTrackerGetter) {

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {
                        
                        info.setText(startingThread.status);
                        spinner.start();
                        reload.setEnabled(false);
                        
                    }

                });

        }

    }

    synchronized void notifyThreadFinished(final TrackerGetter finishedThread) {

        if(finishedThread == currentTrackerGetter) {

            serverList = finishedThread.serverList;
            filterServerList();
            fireTableDataChanged();

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        if(currentTrackerGetter.isRunning || 
                           (currentTrackerGetter.hasFinished &&
                            !currentTrackerGetter.wasSuccess)) {
                            
                            /* Show the error coming from the
                               TrackerGetter. */
                            
                            info.setText(finishedThread.status);
                            
                        } else {

                            info.setText(serverList.length + " servers (" + filteredServerList.length + " displayed).");

                        }
                        
                        spinner.stop();
                        reload.setEnabled(true);
                        
                    }

                });

        }

    }

    class TrackerGetter implements Runnable {
        String status, trackerName;
        boolean isRunning, wasSuccess, hasFinished;
        HLTracker tracker;
        TrackerInterface trackerInterface;
        HLProtocol.ServerInfo[] serverList = new HLProtocol.ServerInfo[0];
        Thread thread;

        TrackerGetter(String trackerName, 
                      TrackerInterface trackerInterface) {

            this.trackerInterface = trackerInterface;
            this.trackerName = trackerName;

            thread = null;

        }

        void start() {
            
            status = "Retrieving server information from " + trackerName;
            serverList = new HLProtocol.ServerInfo[0];
            isRunning = false;
            wasSuccess = true;
            hasFinished = false;
            
            if(thread != null)
                throw new RuntimeException("eww, trying to start thread when it's already running");

            thread = new Thread(this);
            thread.start();

        }

        public void run() {

            isRunning = true;
            trackerInterface.notifyThreadStarting(this);

            try {
                
                if(tracker == null)
                    tracker = new HLTracker(trackerName);

                serverList = tracker.getServerList();
                
            } catch(UnknownHostException e) {
                
                wasSuccess = false;
                status = "Could not locate tracker " + trackerName + ".";
                
            } catch(IOException e) {
                
                DebuggerOutput.stackTrace(e);
                wasSuccess = false;
                status = "Error: " + trackerName + ":" + e.getMessage();
                
            } finally {
                
                thread = null;
                isRunning = false;
                hasFinished = true;
                trackerInterface.notifyThreadFinished(this);

            }
            
        }

        public String toString() {

            return "TrackerGetter[trackerName = " + trackerName + "]";

        }

    }

    /**
     * Following methods extend AbstractTableModel.
     */
    public Class getColumnClass(int col) {

	if(col == 1) 
            return Number.class;

	return String.class;

    }

    public String getColumnName(int col) {

	if(col == 0) return "Name";
	if(col == 1) return "Users";
	if(col == 2) return "Description";
	return "error: col = "+col;

    }

    public int getColumnCount() { 

        return 3; 

    }

    public int getRowCount() { 
        
        if(filteredServerList == null)
            return 0;

        return filteredServerList.length;

    }

    public Object getValueAt(int row, int col) { 

	if(filteredServerList != null) {

            HLProtocol.ServerInfo server = filteredServerList[row];
            
            if(col == 0) 
                return server.name;
            if(col == 1) 
                return new Short(server.nusers);
            if(col == 2) 
                return server.desc;

	}

	return new String("error: col = "+col);

    }

    /**
     * Following method implements ActionListener.
     */
    public synchronized void actionPerformed(ActionEvent e) {

	if(e.getActionCommand().equals("Reload")) {

            currentTrackerGetter.start();

	} else if(e.getSource() == trackerChoice) {
            
	    String entry = trackerChoice.getEditor().getItem().toString();
	    Main.rlo.setStringProperty("Tracker.Default", entry);

	    if(!trackerGetters.containsKey(entry)) {

		Main.rlo.addTracker(entry);
		trackerChoice.addItem(entry);
                currentTrackerGetter = 
                    new TrackerGetter(Main.rlo.getStringProperty("Tracker.Default"), this);
                trackerGetters.put(entry, currentTrackerGetter);
                
	    } else {

                currentTrackerGetter = 
                    (TrackerGetter) trackerGetters.get(entry);

            }

            if(!currentTrackerGetter.hasFinished &&
               !currentTrackerGetter.isRunning)
                currentTrackerGetter.start();

            serverList = currentTrackerGetter.serverList;
            filterServerList();
            fireTableDataChanged();
            
            SwingUtilities.invokeLater(new Runnable() {
                    
                    public void run() {
                        
                        if(currentTrackerGetter.isRunning)
                            spinner.start();
                        else
                            spinner.stop();
                        
                        if(currentTrackerGetter.isRunning || 
                           (currentTrackerGetter.hasFinished &&
                            !currentTrackerGetter.wasSuccess)) {

                            /* Show the error coming from the
                               TrackerGetter. */

                            info.setText(currentTrackerGetter.status);
                            
                        } else {

                            info.setText(serverList.length + " servers (" + filteredServerList.length + " displayed).");

                        }
                        
                        reload.setEnabled(!currentTrackerGetter.isRunning);
                        removeTracker.setEnabled(Main.rlo.getTrackers().length > 1 ? true : false); 
                        
                    }
                    
                });

	} else if(e.getActionCommand().equals("Remove")) {

	    String entry = trackerChoice.getEditor().getItem().toString();

	    if(trackerGetters.containsKey(entry)) {

                trackerGetters.remove(entry);
		Main.rlo.removeTracker(entry);
		trackerChoice.removeItem(entry);
		Main.rlo.setStringProperty("Tracker.Default", Main.rlo.getTrackers()[0]);
		trackerChoice.setSelectedItem(Main.rlo.getStringProperty("Tracker.Default"));

	    }

	    removeTracker.setEnabled(Main.rlo.getTrackers().length > 1 ? 
				     true : false); 

	} else if(e.getSource() == filter) {

            filtertext = filter.getText();
            filterServerList();

            if(currentTrackerGetter.isRunning || 
               (currentTrackerGetter.hasFinished &&
                !currentTrackerGetter.wasSuccess)) {
                
                /* Show the error coming from the TrackerGetter. */
                
                info.setText(currentTrackerGetter.status);
                
            } else {
                
                info.setText(serverList.length + " servers (" + filteredServerList.length + " displayed).");
                
            }

            fireTableDataChanged();

	} else if(e.getSource() == addToBookmarks) {

	    int[] indices = list.getSelectedRows();

	    HLProtocol.ServerInfo server;

	    for(int i = 0; i < indices.length; i++) {

		server = filteredServerList[indices[i]];

                String address = new String(server.getAddress());

		int port = server.port;
		address += (":" + new Integer(server.port).toString());
		try {
		    HLBookmarkFile bmf = 
			new HLBookmarkFile(new File((File) Main.rlo.getProperty("Path.Bookmarks"), FilenameUtils.qualify(new String(server.name))));
		    bmf.put("address", address);
		    bmf.put("login", "guest");
		    bmf.put("password", "");

		    if(!((File) Main.rlo.getProperty("Path.Bookmarks")).exists()) 
			if(!((File) Main.rlo.getProperty("Path.Bookmarks")).mkdir())
			    throw(new IOException("cannot create " + 
						  ((File) Main.rlo.getProperty("Path.Bookmarks")).toString()));
		    bmf.store();

		} catch(IOException ex) {

		    new Error("Could not save bookmark: "+ex.getMessage());

		}		

	    }

	    parent.displayPropertyChanged("bookmarkentries", "");

	} else if(e.getSource() == editServers) {

	    int[] indices = list.getSelectedRows();
	    for(int i = 0; i < indices.length; i++) 
		itemEdit(indices[i]);	    

	} else if(e.getSource() == connectToServers) {

	    int[] indices = list.getSelectedRows();
	    for(int i = 0; i < indices.length; i++) 
		itemActivated(indices[i]);    

	} else if(e.getActionCommand().equals("Activate")) {

            if(list.getSelectedRow() != -1)
                itemActivated(list.getSelectedRow());

        }

    }

    /**
     * Following methods implement MouseListener.
     */
    public void mouseClicked(MouseEvent e) {

	if(e.getClickCount() == 2 && (e.getModifiers() & InputEvent.BUTTON1_MASK) == InputEvent.BUTTON1_MASK)
	    itemActivated(list.getSelectedRow());

    }

    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {

        mouseReleased(e);

    }

    public void mouseReleased(MouseEvent e) {

        if(e.isPopupTrigger()) {
            
	    addToBookmarks.setEnabled(list.getSelectedRows().length == 0 ? 
				      false : true);
            
	    popupMenu.show(list, e.getX(), e.getY());
            
	}

    }

    /**
     * Following methods implement Child.
     */
    public void close() {

	Main.rlo.setProperty("WindowLocation.Tracker", f.getLocation());
	Main.rlo.setProperty("WindowDimension.Tracker", f.getSize());
	f.setVisible(false);

    }
    
    public void displayPropertyChanged(String what, Object property) {

	if(what.equals("listfont")) {

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        list.setFont(((Font) Main.rlo.getProperty("Font.list")));
                        list.repaint();

                    }

                });

	}

    }

    /**
     * Following methods implement WindowListener.
     */
    public void windowActivated(WindowEvent e) {}
    public void windowDeactivated(WindowEvent e) {}
    public void windowClosed(WindowEvent e) { }
    public void windowClosing(WindowEvent e) {
	close();
    }
    public void windowIconified(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}

}







