/**
 * $Id: Shell.java,v 1.4 2001/10/07 04:43:39 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;
 
import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.swing.*;
import javax.swing.event.MenuKeyListener;
import javax.swing.event.MenuKeyEvent;

import redlight.hotline.*;
import redlight.utils.DebuggerOutput;
import redlight.utils.FilenameUtils;
import redlight.utils.TextUtils;
import redlight.graphics.PicturePanel;

/**
 * A Shell can maintain several connections in a tabbed pane.
 */
public class Shell implements Parent, 
                              ActionListener {

    Hashtable connections;
    Container contentPane;
    public JFrame f;
    Shell self;
    JMenu bookmarkMenu;
    JMenuItem closeConnectionMenuItem, saveBookmarkAsMenuItem, addBookmarkMenuItem, homeLineMenuItem, setHomeMenuItem;
    JTabbedPane connectionsPanel;
    ConnectInterface tripConnect;
    int activeTransfers = 0;

    public boolean existsActiveConnection = false;

    public Shell() {

        self = this;
        connections = new Hashtable();
	Font smallFont = (Font) Main.rlo.getProperty("Font.gui");

	f = new JFrame("Red Light");
	contentPane = f.getContentPane();
        connectionsPanel = new JTabbedPane();

	contentPane.setLayout(new BorderLayout());
        contentPane.add(connectionsPanel, BorderLayout.CENTER);

        JMenu fileMenu = new JMenu("File");
        JMenuItem newConnectionMenuItem = fileMenu.add("New connection ...");
        newConnectionMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, Event.CTRL_MASK));
        newConnectionMenuItem.addActionListener(this);
        JMenuItem openNewWindowMenuItem = fileMenu.add("Open new window");
        openNewWindowMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, Event.ALT_MASK | Event.CTRL_MASK));
        openNewWindowMenuItem.addActionListener(this);
        //        saveBookmarkAsMenuItem = fileMenu.add("Save bookmark as ...");
        //        saveBookmarkAsMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, Event.CTRL_MASK));
        //        saveBookmarkAsMenuItem.addActionListener(this);
        //        saveBookmarkAsMenuItem.setEnabled(false);
        closeConnectionMenuItem = fileMenu.add("Close connection");
        closeConnectionMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_W, Event.CTRL_MASK));
        closeConnectionMenuItem.addActionListener(this);
        closeConnectionMenuItem.setEnabled(false);
        fileMenu.addSeparator();
        JMenuItem quitMenuItem = fileMenu.add("Quit");
        quitMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, Event.CTRL_MASK));
        quitMenuItem.addActionListener(this);
        JMenu editMenu = new JMenu("Edit");
        editMenu.add("Cut").setEnabled(false);
        editMenu.add("Copy").setEnabled(false);
        editMenu.add("Paste").setEnabled(false);
        editMenu.addSeparator();
        JMenuItem optionsMenuItem = editMenu.add("Options ...");
        optionsMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, Event.CTRL_MASK));
        optionsMenuItem.addActionListener(this);
        optionsMenuItem.setActionCommand("ActivateOptions");
        
        bookmarkMenu = new JMenu("Bookmarks");
        rebuildBookmarkMenu();

        JMenu trackerMenu = new JMenu("Trackers");
        JMenuItem manageTrackerMenuItem = trackerMenu.add("Manage trackers ...");
        manageTrackerMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, Event.CTRL_MASK));
        manageTrackerMenuItem.addActionListener(this);

        JMenu sharingMenu = new JMenu("Sharing");
        JMenuItem sharingSetupMenuItem = sharingMenu.add("Sharing setup ...");
        sharingSetupMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, Event.CTRL_MASK));
        sharingSetupMenuItem.addActionListener(this);

        JMenu helpMenu = new JMenu("Help");
        JMenuItem aboutMenuItem = helpMenu.add("About ...");
        JMenuItem manualMenuItem = helpMenu.add("Read this ...");
        helpMenu.addSeparator();
        homeLineMenuItem = helpMenu.add("Red Light home");

        if(redlight.Version.RELEASE_TYPE.equals("development")) {

            helpMenu.addSeparator();
            helpMenu.add("Toggle debugging output").addActionListener(new ActionListener() {

                    public void actionPerformed(ActionEvent e) {

                        DebuggerOutput.setEnabled(!DebuggerOutput.on);

                    }

                });

            helpMenu.add("Collect garbage").addActionListener(new ActionListener() {

                    public void actionPerformed(ActionEvent e) {

                        System.gc();

                    }

                });

        }
        
        manualMenuItem.setActionCommand("ActivateManual");
        aboutMenuItem.setActionCommand("ActivateAbout");
        homeLineMenuItem.setActionCommand("ActivateHomeline");
        homeLineMenuItem.addActionListener(this);
        aboutMenuItem.addActionListener(this);
        manualMenuItem.addActionListener(this);
        
        JMenuBar menuBar = new JMenuBar();
        menuBar.add(fileMenu);
        menuBar.add(editMenu);
        menuBar.add(bookmarkMenu);
        menuBar.add(trackerMenu);
        menuBar.add(sharingMenu);
        menuBar.add(helpMenu);

        fileMenu.setMnemonic('F');
        editMenu.setMnemonic('E');
        bookmarkMenu.setMnemonic('B');
        trackerMenu.setMnemonic('T');
        sharingMenu.setMnemonic('S');
        helpMenu.setMnemonic('H');

        menuBar.setSelected(fileMenu);
        f.setJMenuBar(menuBar);
        f.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);

        setConnectionKeyboardShortcuts();

	f.addWindowListener(new WindowAdapter() {
                
                public void windowActivated(WindowEvent e) {
                    
                    Main.setActiveShell(self);
                    
                }
                public void windowClosing(WindowEvent e) {
                    
                    boolean doClose = false;
                    
                    if(activeTransfers > 0) {
                        
                        int r = JOptionPane.showConfirmDialog(null, TextUtils.wordWrap("There are file transfers in progress, which will be interrupted when you close this window (although they may be resumed later). Are you sure you wish to close it?", 40), "Close?", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
                        
                        if(r == JOptionPane.YES_OPTION) 
                            doClose = true;
                        
                    } else {
                        
                        doClose = true;
                        
                    }
                    
                    if(doClose) {
                        
                        for(Enumeration en = connections.keys(); en.hasMoreElements();) 
                            ((ConnectionInterface) connections.get(en.nextElement())).close();
                        
                        /* Force remembering of window position +
                           dimension before close. */
                        
                        Main.setActiveShell(self);
                        
                        f.setVisible(false);
                        f.dispose();
                        f = null;
                        contentPane = null;
                        
                        Main.removeShell(self);
                        
                    }
                    
                }
                
            });
        
	f.pack();

	f.setLocation((Point) Main.rlo.getProperty("WindowLocation.Main"));
        f.setSize((Dimension) Main.rlo.getProperty("WindowDimension.Main"));

	f.show();

        Main.addShell(this);
        Main.setActiveShell(this);

    }

    public void setConnectionKeyboardShortcuts() {

        connectionsPanel.registerKeyboardAction(this, "ActivateConnection1", KeyStroke.getKeyStroke(KeyEvent.VK_F1, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection2", KeyStroke.getKeyStroke(KeyEvent.VK_F2, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection3", KeyStroke.getKeyStroke(KeyEvent.VK_F3, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection4", KeyStroke.getKeyStroke(KeyEvent.VK_F4, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection5", KeyStroke.getKeyStroke(KeyEvent.VK_F5, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection6", KeyStroke.getKeyStroke(KeyEvent.VK_F6, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection7", KeyStroke.getKeyStroke(KeyEvent.VK_F7, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection8", KeyStroke.getKeyStroke(KeyEvent.VK_F8, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection9", KeyStroke.getKeyStroke(KeyEvent.VK_F9, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);
        connectionsPanel.registerKeyboardAction(this, "ActivateConnection10", KeyStroke.getKeyStroke(KeyEvent.VK_F10, Event.CTRL_MASK), JComponent.WHEN_IN_FOCUSED_WINDOW);

    }

    public void addConnectionPane(final String name, ConnectionInterface tci, final Component pane) {

        connections.put(pane, tci);

        SwingUtilities.invokeLater(new Runnable() { 

                public void run() { 
                    
                    connectionsPanel.addTab(name, new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), pane);
                    connectionsPanel.setSelectedComponent(pane);
                    closeConnectionMenuItem.setEnabled(true);
                    addBookmarkMenuItem.setEnabled(true);
                    //                    saveBookmarkAsMenuItem.setEnabled(true);
                    existsActiveConnection = true;

                }

            });
        

    }

    public void removeConnectionPane(final Component pane) {

        connections.remove(pane);

        SwingUtilities.invokeLater(new Runnable() { 
                    
                public void run() { 
                    
                    connectionsPanel.remove(pane);

                    if(connectionsPanel.getTabCount() == 0) {
                        
                        f.getContentPane().remove(connectionsPanel);
                        connectionsPanel = new JTabbedPane();
                        f.getContentPane().setLayout(new BorderLayout());
                        f.getContentPane().add(connectionsPanel, BorderLayout.CENTER);
                        closeConnectionMenuItem.setEnabled(false);
                        //                        saveBookmarkAsMenuItem.setEnabled(false);
                        addBookmarkMenuItem.setEnabled(false);
                        existsActiveConnection = false;

                    }
                    
                }
                
            });
        
    }

    public void rebuildBookmarkMenu() {
        
        final ActionListener listener = this;

        SwingUtilities.invokeLater(new Runnable() {
             
                public void run() {

                    bookmarkMenu.removeAll();
                    setHomeMenuItem = bookmarkMenu.add("Set current site as home");
                    setHomeMenuItem.setActionCommand("ActivateSetHome");
                    setHomeMenuItem.addActionListener(self);
                    addBookmarkMenuItem = bookmarkMenu.add("Add bookmark");
                    addBookmarkMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, Event.CTRL_MASK));
                    addBookmarkMenuItem.addActionListener(self);
                    addBookmarkMenuItem.setEnabled(existsActiveConnection);
                    JMenuItem manageBookmarksMenuItem = bookmarkMenu.add("Manage bookmarks ...");
                    manageBookmarksMenuItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B, Event.CTRL_MASK));
                    manageBookmarksMenuItem.addActionListener(self);
        
                    bookmarkMenu.addSeparator();

                    BookmarkInterface.populateContainerWithBookmarks(bookmarkMenu.getPopupMenu(), listener);
                    
                }
                
            });
             
    }
                                  
    /**
     * Following method implements ActionListener.
     */
    public void actionPerformed(ActionEvent e) {

	if(e.getActionCommand().equals("ActivateHomeline")) {

            new ConnectionInterface("redlight.wox.org",
                                    5500, 
                                    "tubar", 
                                    "tubar", 
                                    "Red Light");

	} else if(e.getActionCommand().equals("ActivateOptions")) {

	    if(Main.rloInterface == null) {

		Main.rloInterface = 
		    new OptionsInterface(this, Main.rlo);

	    } else {
                
                Main.rloInterface.show();

            }

        } else if(e.getActionCommand().equals("ActivateBookmark")) {

            HLBookmarkFile bmf = ((BookmarkMenuItem) e.getSource()).getBookmarkFile();
            String address = bmf.get("address");
            int port = 5500;
            
            /* Get the port from the address if it is there. */
            
            if(address.indexOf(":") != -1) {
                
                port = new Integer(address.substring(address.indexOf(":") + 1)).intValue(); 
                address = address.substring(0, address.indexOf(":"));
                
            }
            
            new ConnectionInterface(address, 
                                    port, 
                                    bmf.get("login"), 
                                    bmf.get("password"), 
                                    bmf.getFile().getName());
            
	} else if(e.getActionCommand().equals("Sharing setup ...")) {

            if(Main.sharingInterface == null) {

                Main.sharingInterface =
                    new SharingInterface(Main.rlo);

            } else {

                Main.sharingInterface.show();

            }

	} else if(e.getActionCommand().equals("Manage trackers ...")) {

	    if(Main.tubeTracker == null) {

		Main.tubeTracker = 
		    new TrackerInterface(this, Main.rlo);

	    } else {

                Main.tubeTracker.show();

            }

	} else if(e.getActionCommand().equals("Manage bookmarks ...")) {

	    if(Main.tubeBookmarks == null) {

		Main.tubeBookmarks = 
		    new BookmarkInterface(this, Main.rlo);

	    } else { 

                Main.tubeBookmarks.show();

            }

	} else if(e.getActionCommand().equals("New connection ...")) {

	    if(tripConnect == null) {

		tripConnect = 
		    new ConnectInterface(this);

	    } else {

                tripConnect.show(true);

            }

	} else if(e.getActionCommand().equals("Close connection")) {

           ConnectionInterface tci = ((ConnectionInterface) connections.get(connectionsPanel.getSelectedComponent()));
           if(tci != null)
               tci.close();

	} else if(e.getActionCommand().equals("Open new window")) {
            
            new Shell();

	} else if(e.getActionCommand().equals("ActivateSetHome")) {

            Main.rlo.setProperty("Home", null);
                
            if(connectionsPanel.getSelectedComponent() != null) {

                ConnectionInterface activeConnection = (ConnectionInterface) connections.get(connectionsPanel.getSelectedComponent());
                
                if(activeConnection != null)                    
                    if(activeConnection.rlm != null)
                        Main.rlo.setProperty("Home", new Connection(activeConnection.rlm.getServer(), activeConnection.rlm.getPort(), activeConnection.rlm.getLogin(), activeConnection.rlm.getPassword(), activeConnection.rlm.getServerName()));
                
            }

	} else if(e.getActionCommand().equals("Add bookmark")) {

            if(connectionsPanel.getSelectedComponent() != null) {

                ConnectionInterface activeConnection = (ConnectionInterface) connections.get(connectionsPanel.getSelectedComponent());
                
                if(activeConnection != null && activeConnection.rlm != null) {
                    
                    try {
                        
                        HLBookmarkFile bmf = new HLBookmarkFile(new File(((File) Main.rlo.getProperty("Path.Bookmarks")), FilenameUtils.qualify(activeConnection.rlm.getServerName())));

                        bmf.put("address", activeConnection.rlm.getServer() + ":" + activeConnection.rlm.getPort());
                        bmf.put("login", activeConnection.rlm.getLogin());
                        bmf.put("password", activeConnection.rlm.getPassword());
                        if(!((File) Main.rlo.getProperty("Path.Bookmarks")).exists()) 
                            if(!((File) Main.rlo.getProperty("Path.Bookmarks")).mkdir())
                                throw(new IOException("cannot create " + ((File) Main.rlo.getProperty("Path.Bookmarks")).toString()));
                        
                        bmf.store();
                        
                    } catch(IOException ex) {
                        
                        new Error("Could not save bookmark: "+ex.getMessage());
                        
                    }		
                    
                }
                
                displayPropertyChanged("bookmarkentries", "");

            }
            
        } else if(e.getActionCommand().equals("Quit")) {
           
            Main.systemEvents.handleQuit();

        } else if(e.getActionCommand().startsWith("ActivateConnection")) {

            int n = Integer.parseInt(e.getActionCommand().substring(new String("ActivateConnection").length())) - 1;
            activateConnection(n);
            
        } else if(e.getActionCommand().startsWith("ActivateAbout")) {

	    if(Main.tubeAboutInterface == null) {

		Main.tubeAboutInterface = 
		    new AboutInterface();

	    } else {
                
                Main.tubeAboutInterface.show();

            }

        } else if(e.getActionCommand().startsWith("ActivateManual")) {

	    if(Main.tubeManualInterface == null) {

		Main.tubeManualInterface = 
		    new ManualInterface();

	    } else {
                
                Main.tubeManualInterface.show();

            }

	} else {

            DebuggerOutput.debug("''" + e.getActionCommand() + "'' not implemented");

        }

    }
	
    public void activateConnection(final int n) {

        if(n < connectionsPanel.getTabCount()) {

            connectionsPanel.setSelectedComponent(connectionsPanel.getComponentAt(n));
            connectionsPanel.grabFocus();

        }

    }

    /**
     * Following methods implement WindowListener.
     */

    public void windowIconified(WindowEvent e) {}
    public void windowDeiconified(WindowEvent e) {}
    public void windowOpened(WindowEvent e) {}

    /**
     * Following methods implement Parent.
     */
    public void displayPropertyChanged(String what, Object property) {

	for(Enumeration en = Main.getMachines(); en.hasMoreElements(); ) {

            Interface iface = ((Machine) en.nextElement()).getInterface();

	    if(iface != null)
                iface.displayPropertyChanged(what, property);

        }

	if(Main.tubeBookmarks != null) 
	    Main.tubeBookmarks.displayPropertyChanged(what, property);

	if(Main.tubeTracker != null) 
	    Main.tubeTracker.displayPropertyChanged(what, property);

	if(what.equals("bookmarkentries"))
            rebuildBookmarkMenu();

    }

    public void error(Object o) {

	new Error(o.toString());

    }

}
