/**
 * $Id: SharingInterface.java,v 1.4 2001/10/07 15:13:50 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.util.Date; 
import java.util.Random;
import java.util.Enumeration;
import java.io.IOException;
import java.io.File;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.Box;
import java.text.DateFormat;

import redlight.hotline.*;
import redlight.crypto.UnixCrypt;
import redlight.server.AccountsTable;
import redlight.server.TrackerTable;
import redlight.utils.DebuggerOutput;
import redlight.graphics.Spinner;
import redlight.utils.TimeFormat;

/**
 */
public class SharingInterface {

    JFrame f;
    ServerStatusPanel serverStatusPanel;
    ServerTrackerPanel serverTrackerPanel;
    ServerLogPanel serverLogPanel;
    JButton cancel, startStop, admin;
    Thread serverThread;
    Random randomGenerator;
    String adminPassword;
    HLServer hls;
    HLProtocol hlp;
    AccountsTable accountsTable;
    HLServerTrackerTable trackerTable;
    HLProtocol.AccountInfo adminAccount;
    Options rlo;
    ActionListener trackerActionListener = new ActionListener() {
            
            public void actionPerformed(ActionEvent e) {
                
                int index = Integer.parseInt(e.getActionCommand().substring(e.getActionCommand().length() - 1));
                
                try {
                    
                    String addressString = serverTrackerPanel.addresses[index].getText();
                    
                    if(addressString.equals(""))
                        return;
                    
                    InetAddress address =
                        InetAddress.getByName(addressString);
                    
                    String password = 
                        serverTrackerPanel.passwords[index].getText();
                    
                    if(password.equals(""))
                        password = null;
                    
                    trackerTable.remove(InetAddress.getByName(Main.rlo.getStringProperty("ServerTrackerAddress" + index)));
                    trackerTable.put(address, password);
                    Main.rlo.setProperty("ServerTrackerAddress" + index, address.getHostName());
                    
                } catch(UnknownHostException ex) {
                    
                    new Error("Sorry, unknown host: " + ex.getMessage());
                    
                }
                    
                DebuggerOutput.debug(trackerTable.toString());
                    
            }
            
        };


    public SharingInterface(Options rlo) {
        
        this.rlo = rlo;

	f = new JFrame("Sharing setup");        
	Container contentPane = f.getContentPane();

        serverStatusPanel = new ServerStatusPanel();
        HelpTextPanel serverStatusHelpPanel = new HelpTextPanel("Server status", "Give your server a name, a description, and select a directory that you want to share files from. Then click Start to start the server. Click the Admin button to login to your server as an administrator and manage your server.", serverStatusPanel);
        serverTrackerPanel = new ServerTrackerPanel(trackerActionListener);
        HelpTextPanel serverTrackerHelpPanel = new HelpTextPanel("Tracker servers", "Enter the addresses of tracker servers that you want your server to appear on. ", serverTrackerPanel);
        serverLogPanel = new ServerLogPanel();

        JTabbedPane tabbedPane = new JTabbedPane();
        tabbedPane.addTab("Status", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), serverStatusHelpPanel, "Server status");
        tabbedPane.addTab("Trackers", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), serverTrackerHelpPanel, "Tracker servers");
        tabbedPane.addTab("Log", new ImageIcon(Main.rlo.getImageSet("FunctionIcons")[Options.SPACER_ICON]), serverLogPanel, "Server log");

	cancel = new JButton("Close");
        cancel.setActionCommand("ActivateCancel");
        cancel.addActionListener(cancelListener);

	startStop = new JButton("Start");
        startStop.setActionCommand("ActivateStartStop");        
        startStop.addActionListener(startStopListener);

	admin = new JButton("Admin");
        admin.setEnabled(false);
        admin.setActionCommand("ActivateAdmin");
        admin.addActionListener(new ActionListener() {
            
            public void actionPerformed(ActionEvent e) {
                
                if(e.getActionCommand().equals("ActivateAdmin"))
                    new ConnectionInterface("127.0.0.1", 
                                            5500,
                                            adminAccount.login,
                                            adminPassword,
                                            serverStatusPanel.serverName.getText());
                
            }
            
        });

        serverStatusPanel.serverName.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {

                    Main.rlo.setProperty("ServerName", serverStatusPanel.serverName.getText());
                    hls.setServerName(serverStatusPanel.serverName.getText());

                }

            });

        serverStatusPanel.serverDescription.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {

                    Main.rlo.setProperty("ServerDescription", serverStatusPanel.serverDescription.getText());
                    hls.setServerDescription(serverStatusPanel.serverName.getText());

                }

            });
        
        ActionListener homeDirListener = new ActionListener() {
                
                public void actionPerformed(ActionEvent e) {
                    
                    if(e.getActionCommand().equals("ActivateBrowse")) {
                        
                        FileDialog fd = new FileDialog(Main.getActiveShell().f, "Select a file in the directory that you want to share", FileDialog.LOAD);
                        fd.setDirectory(((File) Main.rlo.getProperty("Path.ServerHome")).getPath());
                        fd.setFile(((File) Main.rlo.getProperty("Path.ServerHome")).getName());
                        fd.show();
                        
                        if(fd.getDirectory() == null)
                            return;
                        
                        serverStatusPanel.homeDirectory.setText(fd.getDirectory());
                        
                    }

                    File newDirectory = new File(serverStatusPanel.homeDirectory.getText());

                    try {

                        hls.setHomeDirectory(newDirectory);
                        Main.rlo.setProperty("Path.ServerHome", newDirectory);

                    } catch(IllegalArgumentException ex) {

                        new Error(ex.getMessage() + ". Please select another home directory.");

                    }
                    
                    
                }
                
            };
                
        serverStatusPanel.browseButton.addActionListener(homeDirListener);
        serverStatusPanel.homeDirectory.addActionListener(homeDirListener);
        serverStatusPanel.setVisible(false);

        JButton[] adminButton = new JButton[1];
        adminButton[0] = admin;
        JButton[] buttons = new JButton[2];
        buttons[0] = cancel;
        buttons[1] = startStop;
	ButtonPanel buttonPanel = new ButtonPanel(adminButton, buttons);
        ButtonedPanel buttonedPanel = new ButtonedPanel(tabbedPane,
                                                        buttonPanel);

        contentPane.setLayout(new BorderLayout());
        contentPane.add(buttonedPanel, BorderLayout.CENTER);

        SwingUtilities.getRootPane(f).setDefaultButton(startStop);

        f.addWindowListener(new WindowAdapter() {

                public void windowClosing(WindowEvent e) {

                    close();

                }
                
            });

        KeyStroke ks = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
        buttonedPanel.registerKeyboardAction(cancelListener,
                                             "ActivateCancel", 
                                             ks, 
                                             JComponent.WHEN_IN_FOCUSED_WINDOW);
        
	f.pack();

	Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension w = f.getSize();

        f.setLocation((d.width / 2) - (w.width / 2), 
                      (d.height / 2) - (w.height / 2));

        randomGenerator = new Random(System.currentTimeMillis());

        hlp = new HLProtocol();
        hls = new HLServer();
        
        hls.setMaximumUsers(10);
        hls.setMaximumDownloads(2);
        hls.setMaximumUploads(3);
        hls.setMaximumDownloadsPerUser(1);
        hls.setMaximumUploadsPerUser(1);
        hls.setMaximumDownloadQueueSpots(20);
        hls.setMaximumUploadQueueSpots(20);

        trackerTable = 
            new HLServerTrackerTable(hls);

        accountsTable = 
            new AccountsTable(hls,
                              new File(Main.CONFIGURATION_DIRECTORY, 
                                       "accounts")) {
                    
                    public boolean exists(String login) {

                        if(login.equals(adminAccount.login))
                            return true;

                        return super.exists(login);

                    }

                    public HLProtocol.AccountInfo get(String login) {
                        
                        if(login.equals(adminAccount.login))
                            return adminAccount;

                        return super.get(login);
                        
                    }

                    public void put(String login,
                                    HLProtocol.AccountInfo newAccount) {

                        /* Never store an admin account. */

                        table.remove(adminAccount.login);
                        super.put(login, newAccount);
                        table.put(adminAccount.login, adminAccount);

                    }

                };

        try {

            hls.setHomeDirectory((File) Main.rlo.getProperty("Path.ServerHome"));

        } catch(IllegalArgumentException e) {

            Main.rlo.setProperty("Path.ServerHome", hls.getHomeDirectory());
            serverStatusPanel.homeDirectory.setText(hls.getHomeDirectory().toString());
            new Error(e.getMessage() + ". Please select another path.");

        }

        hls.setHLServerListener(listener);
        hls.setAccountsTable(accountsTable);
        hls.setTrackerTable(trackerTable);

        show();
    }

    public void close() {

        f.setVisible(false);

    }

    public void show() {

        f.setVisible(true);

    }

    String createRandomString() {

        return Integer.toHexString(randomGenerator.nextInt()) +
            Integer.toHexString(randomGenerator.nextInt());

    }

    HLProtocol.AccountInfo createRandomAdminAccount() {
        
        adminPassword = createRandomString();

        return hlp.new AccountInfo(createRandomString(), 
                                   "rl admin user",
                                   UnixCrypt.crypt("aa", adminPassword),
                                   -1,
                                   null);

    }

    class ServerThread extends Thread {

            public void run() {

                try {

                    adminAccount = createRandomAdminAccount();
                    serverLogPanel.writeln("\nStarting server ...");
                    hls.listen();

                } catch(Exception e) {

                    String reason = e.toString();
                    
                    if(e.getMessage() != null)
                        reason = e.getMessage();
                    
                    serverLogPanel.writeln("Server shut down (" + reason + ")");
                    DebuggerOutput.stackTrace(e);

                }

                /* Reset UI. */

                startStop.setText("Start");
                admin.setEnabled(false);
                serverStatusPanel.setVisible(false);
                serverStatusPanel.reset();

            }

        };
    
    /**
     * This class responds to HLServer events.
     */
    HLServerListener listener = new HLServerListener() {
            
            /**
             * Called when the server is ready and listening.
             */
            public void serverReady() {
                
                serverStatusPanel.setVisible(true);
                admin.setEnabled(true);
                startStop.setText("Stop");
                serverStatusPanel.values[ServerStatusPanel.SINCE].setText(TimeFormat.formatCurrentDateTime(DateFormat.SHORT, DateFormat.MEDIUM));
                
                try {

                    InetAddress[] all = InetAddress.getAllByName(InetAddress.getLocalHost().getHostName());
                    String adresses = "";

                    for(int i = 0; i < all.length; i++) {

                        adresses += all[i].getHostAddress();

                        if(i + 1 < all.length)
                            adresses += ", ";

                    }

                    serverStatusPanel.values[ServerStatusPanel.SERVER_ADDRESS].setText(adresses);

                } catch(UnknownHostException e) {}
                
            }
            
            /**
             * Called when a single line should be logged.
             * @param line the log line.
             */
            public void logLine(String line) {
                
                serverLogPanel.writeln(line);
                
            }
            
            /**
             * Called just after a user has successfully logged in.
             * @param client the client logging in.
             */
            public void userJoined(HLServerDispatcher client) {
                
                updateTextField(ServerStatusPanel.CURRENTLY_CONNECTED, 1);
                updateTextField(ServerStatusPanel.CONNECTION_COUNTER, 1);

                int currentlyConnected = getTextFieldValue(ServerStatusPanel.CURRENTLY_CONNECTED);
                int connectionPeak = getTextFieldValue(ServerStatusPanel.CONNECTION_PEAK);

                if(currentlyConnected > connectionPeak)
                    updateTextField(ServerStatusPanel.CONNECTION_PEAK, 1);

            }

            /**
             * Called when a user leaves.
             * @param client the client that has left.
             */
            public void userLeft(HLServerDispatcher client) {

                updateTextField(ServerStatusPanel.CURRENTLY_CONNECTED, -1);

            }
                        
            /**
             * Called when a transfer goes in progress.
             * @param tr the transfer request in progress.
             */
            public void transferProgressStart(HLServer.TransferRequest tr) {

                updateProgress(tr, 1);
                updateTransfer(tr, 1, ServerStatusPanel.DOWNLOAD_COUNTER, ServerStatusPanel.UPLOAD_COUNTER);

            }
            
            /**
             * Called when a transfer stops being in progress.
             * @param tr the transfer request.
             */
            public void transferProgressStop(HLServer.TransferRequest tr) {

                updateProgress(tr, -1);

            }

            /**
             * Called when a transfer enters the queue.
             * @param tr the transfer request.
             */
            public void transferQueueStart(HLServer.TransferRequest tr) {
                
                updateQueue(tr, 1);

            }
            
            /**
             * Called when a transfer leaves the queue.
             * @param tr the transfer request.
             */
            public void transferQueueStop(HLServer.TransferRequest tr) {

                updateQueue(tr, -1);

            }

            void updateProgress(HLServer.TransferRequest tr, int delta) {
                
                updateTransfer(tr, delta, ServerStatusPanel.DOWNLOADS_IN_PROGRESS, ServerStatusPanel.UPLOADS_IN_PROGRESS);

            }

            void updateQueue(HLServer.TransferRequest tr, int delta) {
                
                updateTransfer(tr, delta, ServerStatusPanel.DOWNLOADS_IN_QUEUE, ServerStatusPanel.UPLOADS_IN_QUEUE);
                
            }

            void updateTransfer(HLServer.TransferRequest tr, int delta,
                                int dlIndex, int ulIndex) {
                
                int index = dlIndex;
                
                if(tr.type == tr.FILE_UPLOAD)
                    index = ulIndex;

                updateTextField(index, delta);
                
            }

            void updateTextField(int index, int delta) {

                int oldValue = getTextFieldValue(index);
                serverStatusPanel.values[index].setText("" + (oldValue + delta));

            }

            int getTextFieldValue(int index) {

                return Integer.parseInt(serverStatusPanel.values[index].getText());

            }
            
        };
    
    ActionListener cancelListener = new ActionListener() {
            
            public void actionPerformed(ActionEvent e) {
                
                if(e.getActionCommand().equals("ActivateCancel"))
                    close();
                
            }
            
        };

    ActionListener startStopListener = new ActionListener() {

            
            public void actionPerformed(ActionEvent e) {
                
                if(e.getActionCommand().equals("ActivateStartStop")) {

                    if(serverThread != null && serverThread.isAlive()) {
                       
                        hls.shutdown("Stopped by admin");
                        serverStatusPanel.reset();
                         
                    } else {

                        serverThread = new ServerThread();
                        serverThread.start();

                    }

                }

            }
            
        };
    
}

class ServerStatusPanel extends JPanel {

    JTextField serverName, serverDescription, homeDirectory;
    JButton browseButton;

    JLabel[] values = {

        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),
        new JLabel("0"),

    };

    JLabel[] labels = {

        new JLabel("Server addresses:"),
        new JLabel("Up since:"),
        new JLabel("Currently connected:"),
        new JLabel("Downloads in progress:"),
        new JLabel("Uploads in progress:"),
        new JLabel("Downloads in queue:"),
        new JLabel("Uploads in queue:"),
        new JLabel("Download counter:"),
        new JLabel("Upload counter:"),
        new JLabel("Connection peak:"),
        new JLabel("Connection counter:")

    };

    static int SERVER_ADDRESS = 0;
    static int SINCE = 1;
    static int CURRENTLY_CONNECTED = 2;
    static int DOWNLOADS_IN_PROGRESS = 3;
    static int UPLOADS_IN_PROGRESS = 4;
    static int DOWNLOADS_IN_QUEUE = 5;
    static int UPLOADS_IN_QUEUE = 6;
    static int DOWNLOAD_COUNTER = 7;
    static int UPLOAD_COUNTER = 8;
    static int CONNECTION_PEAK = 9;
    static int CONNECTION_COUNTER = 10;

    ServerStatusPanel() {

        setOpaque(true);
        setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.anchor = gbc.WEST;

        serverName = new JTextField(Main.rlo.getStringProperty("ServerName"), 25);
        serverDescription = new JTextField(Main.rlo.getStringProperty("ServerDescription"), 25);
        homeDirectory = new JTextField(((File) Main.rlo.getProperty("Path.ServerHome")).toString());
        homeDirectory.setActionCommand("ActivateSetHomeDir");
        browseButton = new JButton("Browse...");
        browseButton.setActionCommand("ActivateBrowse");

        JPanel homeDirPanel = new JPanel();
        homeDirPanel.setLayout(new BorderLayout());
        homeDirPanel.add(homeDirectory, BorderLayout.CENTER);
        homeDirPanel.add(browseButton, BorderLayout.EAST);
        addRow(new JLabel("Server name:"), serverName, gbc);
        addRow(new JLabel("Server description:"), serverDescription, gbc);
        addRow(new JLabel("Share files from:"), homeDirPanel, gbc);
        
        for(int i = 0; i < labels.length; i++)
            addRow(i, gbc);

        gbc.fill = gbc.BOTH;
        gbc.weightx = 1;
        gbc.weighty = 1;
        addRow(Box.createGlue(), Box.createGlue(), gbc);

    }

    void addRow(int i, GridBagConstraints gbc) {

        addRow(labels[i], values[i], gbc);

    }

    void addRow(Component l, Component v, GridBagConstraints gbc) {

        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.WEST;
        add(l, gbc);
        gbc.gridx++;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.EAST;
        add(v, gbc);
        gbc.gridx = 1;
        gbc.gridy++;

    }

    void reset() {

        for(int i = 0; i < values.length; i++)
            values[i].setText("0");

    }

    public void setVisible(boolean b) {

        //        super.setVisible(b);

        for(int i = 0; i < values.length; i++) {

            labels[i].setVisible(b);
            values[i].setVisible(b);

        }

    }

}

class ServerLogPanel extends BorderedPanel {

    JTextArea textArea;

    ServerLogPanel() {

        super("Server log");

        textArea = new JTextArea(20, 65);
        textArea.setFont((Font) Main.rlo.getProperty("Font.text"));
        
        setLayout(new BorderLayout());        
        add(new JScrollPane(textArea), BorderLayout.CENTER);

    }

    void writeln(final String s) {

        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    textArea.append(s + "\n");
                    textArea.setCaretPosition(textArea.getText().length());

                }

            });

    }

}

class ServerTrackerPanel extends JPanel {

    JTextField[] addresses = new JTextField[4];
    JTextField[] passwords = new JTextField[4];

    ServerTrackerPanel(ActionListener actionListener) {

        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.anchor = gbc.WEST;

        for(int i = 0; i < 4; i++) {

            addresses[i] = new JTextField((String) Main.rlo.getProperty("ServerTracker" + (i + 1)), 15);
            addresses[i].setActionCommand("ActivateTrackerAddress" + i);
            addresses[i].addActionListener(actionListener);
            passwords[i] = new JTextField((String) Main.rlo.getProperty("ServerTracker" + (i + 1)), 10);
            passwords[i].setActionCommand("ActivateTrackerPassword" + i);
            passwords[i].addActionListener(actionListener);
            addRow(i, gbc);

        }

        gbc.weightx = 1;
        gbc.weighty = 1;
        gbc.fill = gbc.BOTH;
        addRow(Box.createGlue(), 
               Box.createGlue(), 
               Box.createGlue(), 
               Box.createGlue(), 
               gbc);

    }

    void addRow(int i, GridBagConstraints gbc) {

        addRow(new JLabel("Address:"), 
               addresses[i], 
               new JLabel("Password: "), 
               passwords[i],
               gbc);

    }

    void addRow(Component l, 
                Component v,
                Component l2,
                Component p,
                GridBagConstraints gbc) {

        gbc.fill = gbc.NONE;
        gbc.anchor = gbc.WEST;
        add(l, gbc);
        gbc.gridx++;
        gbc.fill = gbc.HORIZONTAL;
        gbc.anchor = gbc.EAST;
        add(v, gbc);
        gbc.gridx++;
        gbc.fill = gbc.NONE;
        add(l2, gbc);
        gbc.gridx++;
        gbc.fill = gbc.HORIZONTAL;
        add(p, gbc);
        gbc.gridx = 1;
        gbc.gridy++;

    }

}
