/**
 * $Id: MessageInterface.java,v 1.9 2001/10/04 01:03:59 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.io.IOException;
import java.util.Date;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.*;

import redlight.hotline.*;
import redlight.utils.TextUtils;

/**
 * Common methods for receiving and sending messages.
 */
public abstract class MessageInterface implements ActionListener, 
                                                  WindowListener, 
                                                  Child {
    
    Machine rlm;
    Container contentPane;
    JTextArea message;
    JButton send, cancel;
    JScrollPane scrollPane;
    JFrame f;
    HLProtocol hlp;
    User user;

    public MessageInterface(Machine rlm, 
                            User user,
                            boolean isReceived) {

	this.rlm = rlm;
        this.user = user;

	Font guiFont = (Font) Main.rlo.getProperty("Font.gui");

	f = new JFrame("Message for " + user.nick);
	contentPane = f.getContentPane();
	GridBagLayout gbl = new GridBagLayout();
	GridBagConstraints gbc = new GridBagConstraints();
	message = new JTextArea("", 10, 50);
	scrollPane = new JScrollPane(message);
        JPanel messagePane = new JPanel();
	JPanel buttonPanel = new JPanel();
        JLabel date = new JLabel(new Date(System.currentTimeMillis()).toString());

        JLabel from = new UserCellRenderer(user, false);
        UserToolBar toolBar = new UserToolBar(rlm, user);
	send = new JButton("Send");
	cancel = new JButton("Dismiss");

        Border emptyBorder = 
            new EmptyBorder((Insets) Main.rlo.getProperty("BorderInsets"));

        Border lineBorder;
        JTextArea messageHelpText;

        if(isReceived) {

            lineBorder = 
                BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Message received");
            messageHelpText = new JTextArea("You've received a message from \"" + user.nick + "\":");
            message.setEditable(false);

        } else {

            lineBorder = 
                BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Sending message");
            messageHelpText = new JTextArea("Sending a message to \"" + user.nick + "\":");

        }

        Border messageBorder = 
            BorderFactory.createCompoundBorder(lineBorder, emptyBorder);

	contentPane.setLayout(gbl);
        messagePane.setLayout(new GridBagLayout());
        buttonPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));

        toolBar.msg.setVisible(false);
        toolBar.reload.setVisible(false);
        messageHelpText.setVisible(false);

	message.setBackground(rlm.getSchemeColor("background"));
	message.setForeground(rlm.getSchemeColor("foreground"));
	message.setLineWrap(true);
	message.setWrapStyleWord(true);
	message.setFont((Font) Main.rlo.getProperty("Font.text"));
	message.setCaretPosition(0);
	scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        SwingUtilities.getRootPane(f).setDefaultButton(send);

        send.setActionCommand("Send");

        if(isReceived) {

            send.setText("Reply");
            send.setMnemonic('R');
            send.setToolTipText("Reply to this message");

        } else {

            send.setMnemonic('S');
            send.setToolTipText("Send this message");

        }

        cancel.setMnemonic('D');
        cancel.setToolTipText("Discard the message");

        messagePane.setBorder(messageBorder);

        messageHelpText.setLineWrap(true);
        messageHelpText.setWrapStyleWord(true);
        messageHelpText.setOpaque(false);
        messageHelpText.setEditable(false);

        date.setFont(guiFont);
        //        from.setFont(guiFont);
        date.setForeground(messageHelpText.getForeground());
        //        from.setForeground(messageHelpText.getForeground());

        cancel.registerKeyboardAction(this, "Cancel", KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), JComponent.WHEN_IN_FOCUSED_WINDOW);
        
	send.addActionListener(this);
	cancel.addActionListener(this);

        gbc.anchor = gbc.NORTHEAST;
	gbc.gridx = 0; 
        gbc.gridy = 0;
        gbc.ipadx = 4;
        gbc.ipady = 4;
        gbc.weightx = 0;
        gbc.fill = gbc.HORIZONTAL;
        Insets oldInsets = gbc.insets;
        gbc.gridwidth = 2;
        gbc.insets = new Insets(0, 0, 8, 0);
        messagePane.add(messageHelpText, gbc);
        gbc.insets = oldInsets;
        gbc.gridwidth = 1;
        gbc.gridy = 1;
        gbc.gridx = 0;
        gbc.anchor = gbc.EAST;
        messagePane.add(new JLabel(isReceived ? "From:" : "To:"), gbc);
        gbc.gridy++;
        messagePane.add(new JLabel("Date:"), gbc);
        gbc.gridx++;
        gbc.gridy = 1;
        messagePane.add(from, gbc);
        gbc.gridy++;
        messagePane.add(date, gbc);
        gbc.gridy++;
        gbc.gridx = 0;
        gbc.gridwidth = 2;
        messagePane.add(toolBar, gbc);
        gbc.gridy++;
        gbc.gridx = 0;
	gbc.weightx = 1; 
        gbc.weighty = 1;
	gbc.fill = GridBagConstraints.BOTH;
        messagePane.add(scrollPane, gbc);

	buttonPanel.add(cancel);
	buttonPanel.add(send);
        gbc.insets = (Insets) Main.rlo.getProperty("BorderInsets");

	gbc.weightx = 1; 
        gbc.weighty = 1;
	gbc.gridx = 0; 
        gbc.gridy = 0;
	gbc.fill = GridBagConstraints.BOTH;
	contentPane.add(messagePane, gbc);
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.weighty = 0;
        gbc.insets = new Insets(0, 0, 0, 8);
        contentPane.add(buttonPanel, gbc);

	f.pack();

	Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension w = f.getSize();
	f.setLocation((d.width / 2) - (w.width / 2), 
		      (d.height / 2) - (w.height / 2));

	rlm.getInterface().registerChild(this);

    }
	
    void closeWindow() {

        close();
	f.dispose();
        f = null;
        rlm = null;

    }

    /**
     * Following method implement ActionListener.
     */
    public void actionPerformed(ActionEvent e) {

	try {

	    if(e.getActionCommand().equals("Send")) {

		rlm.getHLC().requestMessageSend(user.sock, TextUtils.findAndReplace(message.getText(), "\n", "\r"));
		closeWindow();

	    } else {

		closeWindow();

	    }

	} catch (Exception ex) {

	    ex.printStackTrace();
	    rlm.getInterface().error(ex);

	}

    }

    /**
     * Following methods implement WindowListener.
     */
    public void windowActivated(WindowEvent e) { }
    public void windowDeactivated(WindowEvent e) { }
    public void windowClosed(WindowEvent e) { }
    public void windowClosing(WindowEvent e) {

	closeWindow();

    }
    public void windowIconified(WindowEvent e) { }
    public void windowDeiconified(WindowEvent e) { }
    public void windowOpened(WindowEvent e) {}

    /**
     * Following methods implement Child.
     */

    public void close() {

        /* This method may be invoked by our "interface parent" when
           the connection collapses. In that case we don't want to
           close the window because people would lose their
           messages. So instead we just disable the send / reply
           button. */

	send.setEnabled(false);

        /* Or, the interface parent may have disappeared by the time
           the user closes this window (the other possibility). */

        if(rlm.getInterface() != null)
            rlm.getInterface().unregisterChild(this);

    }

    public void displayPropertyChanged(String what, final Object property) {

	if(what.equals("textfont")) {

            SwingUtilities.invokeLater(new Runnable() {

                    public void run() {

                        message.setFont((Font) property);
                        message.setEnabled(false);
                        message.setText(message.getText());
                        message.setCaretPosition(0);
                        message.setEnabled(true);

                    }

                });

	} else {

	    message.setBackground(rlm.getSchemeColor("background"));
	    message.setForeground(rlm.getSchemeColor("foreground"));

	}

    }
		
}
