/**
 * $Id: DownloadInterface.java,v 1.8 2001/10/08 19:56:49 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import redlight.hotline.*;
import redlight.utils.TextUtils;
import redlight.utils.FilenameUtils;
import redlight.utils.DebuggerOutput;
import redlight.macfiles.*;
import redlight.utils.Meter;

/**
 * Interface for downloading files. Checks whether the file already
 * exists and asks the user whether or not to resume. 
 */
public class DownloadInterface extends TransferInterface {

    String remoteFile;
    HLProtocol.FileListComponent file;
    ResumeInterface resumeUI;

    /* The .hpf temporary file that receives the download. */

    File tempFile;

    public DownloadInterface(Machine machine, 
                             String path, 
                             HLProtocol.FileListComponent file) {

        this(machine, path, file, false);

    }

    public DownloadInterface(Machine machine, 
                             String path, 
                             HLProtocol.FileListComponent file, 
                             boolean autoResume) {

	super(machine, file.fileName);
	this.path = path;
	this.file = file;
        this.remoteFile = path;
        this.autoResume = autoResume;

        if(Main.iconFile != null)
            info.setIcon(Main.iconFile.getIconForFile(file));
        
	/* When doRequest() succeeds, granted() is called by the
           superclass. */

	doRequest();

    }

    /**
     * Invoked by the superclass when this transfer is ready to be
     * started.
     */
    public void granted() {
        
        /* Try to create the download path set in Options. */
        
        File destination = 
            (File) Main.rlo.getProperty("Path.Downloads");
        
        if(!destination.exists()) 
            destination.mkdirs();
        
        if(!destination.exists()) {
            
            rlm.getInterface().error("Cannot download because the directory " + destination + " could not be created.");
            close();
            return;
            
        }
        
        /* Ask the user for a new name for this download when
           a file by this name already exists. */

        if(tempFile == null) {

            String qualifiedFilename = FilenameUtils.qualify(filename);
            File finalFile = new File(destination, qualifiedFilename);
            
            if(finalFile.exists())
                qualifiedFilename = AskNewFilename.ask(finalFile);
            
            if(qualifiedFilename == null) {
                
                close();
                return;
                
            }

            if(interrupted)
                return;

            tempFile = new File(destination, qualifiedFilename + ".hpf");

        }

        try {
            
            /* Create the MacFile wrapper object to contain the file
               we're about to download. */

            localFile = MacFile.createMacFile((String) Main.rlo.getProperty("MacFileMethod"), tempFile, MacFile.READ_PERM | MacFile.WRITE_PERM);
           
        } catch(Exception ex) {
            
            rlm.getInterface().error("Could not download " + file.fileName + ": " + ex.getMessage());
            DebuggerOutput.stackTrace(ex);
            close();
            
        }
        
        /* Ask the user whether or not to resume. */
        
        resumeUI = new ResumeInterface(file.fileName, autoResume, new ResumeActionListener() {
                
                public boolean canResume() {
                    
                    return localFile.exists();
                    
                }
                
                public void resumeCancelled() {
                    
                    if(!interrupted)
                        close();
        
                }
                
                public void resumeAnswered(boolean resume) {
                    
                    /* Delete the local file because the user doesn't
                       want to resume. */
                    
                    if(!resume)
                        localFile.delete();
                    
                    try {
                        
                        DebuggerOutput.debug("DownloadInterface.confirm: requesting file download " + remoteFile);
                        setTransferTask(rlm.getHLC().requestFileDownload(remoteFile, localFile, meter));
                        info.setText(filename + "   [waiting for server to respond]        ");
                        repaintStatus();
                        closeLocalFile = false;
                        
                    } catch(IOException e) {}
                    
                }
                
            });
                
    }

    /** 
     * Following methods overridden from superclass.
     */

    public void stopMeter(int ref) {

        super.stopMeter(ref);

        /* Get rid of the .hpf extension. */

        File file = localFile.getFile();
        String parent = file.getParent();
        String name = file.getName();
        File newFile = new File(parent, name.substring(0, name.lastIndexOf(".hpf")));
        DebuggerOutput.debug("DownloadInterface.stopMeter: renaming " + file + " to " + newFile);
        localFile.renameTo(newFile);

    }

    /**
     * This method overridden to ignore the results from any
     * outstanding "Resume downloads" windows when we get closed.
     */
    protected void interrupt() {

	if(resumeUI != null)
            resumeUI.interrupt();

	super.interrupt();

    }

}

class AskNewFilename {

    static public String ask(File f) {
        
        FileDialog fd = new FileDialog(Main.getActiveShell().f, 
                                       "Select a new file:", 
                                       FileDialog.SAVE);

        fd.setDirectory(f.getParent());
        fd.setFile(FilenameUtils.getNextUniqueFile(f).getName());
        DebuggerOutput.debug("AskNewFilename.ask: showing FileDialog");
        fd.show();
        
        if(fd.getFile() != null)
            return fd.getFile();
     
        return null;

    }

}
    
